
/*********************************************************************
****               ESCHER Next Generation - Surface               ****
**** (c) 1997-2012, G. Ausiello, G. Cesareni, M. Helmer Citterich ****
****               NG version by Alessandro Pedretti              ****
*********************************************************************/


#include "my_macros.h"

#include "globstr.h"
#include "nsc.h"
#include "surf.h"
#include "utils.h"

#define  ES_READBUF_SIZE                256


/**** Calculate the surface ****/

ES_BOOL SrfCalc(char *name, ES_BRK *Brk, ES_ULONG NumDots, float ProbeRad)
{
  double        Area, Volume;
  ES_SURFACE    *BegSrf;
  ES_ULONG      TotDots;
  ES_BOOL       Ret;

  if ((Ret = NSC(Brk -> atom, Brk -> n_atoms, NumDots, ProbeRad, FLAG_DOTS, &Area, NULL,
                 &Volume, &BegSrf, &TotDots)) == TRUE) {
    SrfSortByAtm(&BegSrf, Brk -> n_atoms);
    Ret = SrfWrite(name, BegSrf, Brk);
    SrfFree(BegSrf);
  }

  return Ret;
}


/**** Copy the surface ****/

ES_BOOL SrfCopy(struct surf *dest, const struct surf *source)
{
  ES_BOOL       Ret = FALSE;

  dest -> atom    = NULL;
  dest -> col     = NULL;
  dest -> x       = NULL;
  dest -> y       = NULL;
  dest -> z       = NULL;
  dest -> org     = source -> org;
  dest -> n_atoms = source -> n_atoms;
  dest -> n_dots  = source -> n_dots;
  if ((dest -> atom = (struct atom_ref *)Alloca(dest -> n_atoms * sizeof(struct atom_ref))) != NULL) {
    if ((dest -> col = (int *)Alloca(dest->n_dots * sizeof(int))) != NULL) {
      if ((dest -> x = (float *)Alloca(dest -> n_dots * sizeof(float))) != NULL) {
        if ((dest -> y = (float *)Alloca(dest -> n_dots * sizeof(float))) != NULL) {
          if ((dest -> z = (float *)Alloca(dest -> n_dots * sizeof(float))) != NULL) {
            memcpy(dest -> col , source -> col , dest -> n_dots  * sizeof(int));
            memcpy(dest -> x   , source -> x   , dest -> n_dots  * sizeof(float));
            memcpy(dest -> y   , source -> y   , dest -> n_dots  * sizeof(float));
            memcpy(dest -> z   , source -> z   , dest -> n_dots  * sizeof(float));
            memcpy(dest -> atom, source -> atom, dest -> n_atoms * sizeof(struct atom_ref));
            Ret = TRUE;
          }
        }
      }
    }
  }

  if (!Ret) SrfDelete(dest);

  return Ret;
}


/**** Free the surface ****/

void SrfDelete(struct surf* surf)
{
  if (surf -> atom) FREE(surf -> atom);
  if (surf -> x)    FREE(surf -> x);
  if (surf -> y)    FREE(surf -> y);
  if (surf -> z)    FREE(surf -> z);
  if (surf -> col)  FREE(surf -> col);
}


/**** Free a memory list of surface dots ****/

void SrfFree(register ES_SURFACE *Dot)
{
  register ES_SURFACE *Succ;

  if (Dot) {
    do {
      Succ = Dot -> Ptr;
      FREE(Dot);
      Dot = Succ;
    } while(Dot);
  }
}


/**** Extract the surface name from file path ****/

void SrfName(char *Src, char *Dest)
{
  register char         *Ptr1;

  for(Ptr1 = Dest; *Src; ++Ptr1, ++Src) {
    if (*Src == ' ') *Ptr1 = '_';
    else *Ptr1 = *Src;
  }
  *Ptr1 = 0;

  for(Ptr1 = Dest + strlen(Dest); Ptr1 != Dest; --Ptr1) {
    if (*Ptr1 == '.') {
      *Ptr1 = 0;
      break;
    }
  } /* End of for */
}


/**** Read the surface in Insight format ****/

ES_BOOL SrfRead(char *name, struct surf *surf)
{
  char          buffer[ES_READBUF_SIZE];
  char          *Ptr;
  FILE          *f;
  int           i, buffer_counter, atoms_counter;

  ES_BOOL       Ret   = TRUE;
  size_t        Begin = 0;

  if ((f = fopen(name, "rb")) != NULL) {

   /**** Check the file format ****/

   while(fgets(buffer, ES_READBUF_SIZE, f)) {
     if (!strncmp(buffer, "DOTS", 4)) {
       Begin = ftell(f);
       break;
     }
   } /* End of while */

   if (Begin) {

     /**** Count the dots and the atoms ****/

     surf -> n_atoms = 0;
     surf -> n_dots  = 0;
     while(fgets(buffer, ES_READBUF_SIZE, f)) {
       if ((*buffer != ' ') && (*buffer != '\n') &&
           (*buffer != '\t') && (*buffer != '\r')) {
         if (*buffer == 'A')
           ++surf -> n_atoms;
         else
           ++surf -> n_dots;
       }
     } /* End of while */

     /**** Allocate the memory ****/

      if ((surf -> n_atoms) && (surf -> n_dots)) {
        if ((surf -> atom = (struct atom_ref *)Alloca(surf -> n_atoms * sizeof(struct atom_ref))) != NULL) {
          if ((surf -> col = (int *)Alloca(surf -> n_dots * sizeof(int))) != NULL) {
            if ((surf -> x = (float *)Alloca(surf -> n_dots * sizeof(float))) != NULL) {
              if ((surf -> y = (float *)Alloca(surf -> n_dots * sizeof(float))) != NULL) {
                if ((surf -> z = (float *)Alloca(surf -> n_dots * sizeof(float))) != NULL) {
                  fseek(f, Begin, SEEK_SET);
                  for(i = 0; i < surf -> n_atoms; i++) surf -> atom[i].atom_dots = 0;
                  i             = 0;
                  atoms_counter = -1;
                  while(fgets(buffer, ES_READBUF_SIZE, f)) {
                    if ((*buffer != ' ') && (*buffer != '\n') &&
                        (*buffer != '\t') && (*buffer != '\r')) {
                      if (buffer[0] == 'A') {
                        atoms_counter++;
                        buffer_counter = 0;
                        for(Ptr = buffer; (*Ptr) && (*Ptr != ' '); ++Ptr);
                        if (*Ptr) {
                          ++Ptr;
                          while(Ptr[buffer_counter++] != ' ');
                          if (buffer_counter > 30) buffer_counter = 30;
                          memcpy(surf -> atom[atoms_counter].label, buffer, buffer_counter);
                          surf -> atom[atoms_counter].label[buffer_counter + 1] = 0;
                          surf -> atom[atoms_counter].atom_dots                 = 0;
                        } else Ret = CatErr(MSG_ERR_SRF_CORRUPT);
                      } else {
                        sscanf(buffer, "%f %f %f %d", surf -> x + i, surf -> y + i, surf -> z + i, surf -> col + i);
                        surf -> atom[atoms_counter].atom_dots++;
                        ++i;
                      }
                    }
                  } /* End of while */

                  surf -> org.x = surf -> org.y = surf -> org.z = 0.0f;
                } else Ret = FALSE;
              } else Ret = FALSE;
            } else Ret = FALSE;
          } else Ret = FALSE;
          if (!Ret) SrfDelete(surf);
        } else Ret = FALSE;
      } else Ret = CatErr(MSG_ERR_SRF_CORRUPT);
    } else Ret = CatErr(MSG_ERR_SRF_UNKFORMAT);
    fclose(f);
  } else Ret = CatErr(MSG_ERR_FILE_NOTFOUND, name);

  return Ret;
}


/**** X rotate ****/

void SrfRotX(struct surf *surf, float rx)
{
  int           i;
  float         raggio, angolo;

  for(i = 0; i < surf -> n_dots; i++) {
    raggio = sqrtf(surf -> z[i] * surf -> z[i] + surf -> y[i] * surf -> y[i]);
    angolo = rx + atan2f(surf -> y[i], surf -> z[i]);
    surf -> z[i] = raggio * cosf(angolo);
    surf -> y[i] = raggio * sinf(angolo);
  } /* End of for */
}


/**** Y rotate ****/

void SrfRotY(struct surf *surf, float ry)
{
  int           i;
  float         raggio, angolo;

  for(i = 0; i < surf -> n_dots; i++) {
    raggio = sqrtf(surf -> z[i] * surf -> z[i] + surf -> x[i] * surf -> x[i]);
    angolo = ry + atan2f(surf -> x[i], surf -> z[i]);
    surf -> z[i] = raggio * cosf(angolo);
    surf -> x[i] = raggio * sinf(angolo);
  } /* End of for (i) */
}


/**** Z rotate ****/

/*
void SrfRotZ(struct surf* surf, float rz)
{
  int           i;
  float         raggio,angolo;

  for(i = 0; i < surf -> n_dots; i++) {
    raggio = sqrtf(surf -> x[i] * surf -> x[i] + surf -> y[i] * surf -> y[i]);
    angolo = rz + atan2f(surf -> y[i], surf -> x[i]);
    surf -> x[i] = raggio * cosf(angolo);
    surf -> y[i] = raggio * sinf(angolo);
  }
}
*/


/**** Sort the surface dots by atom number ****/

void SrfSortByAtm(ES_SURFACE **InizSrf, ES_ULONG TotAtm)
{
  ES_SURFACE            *Dot, *PrecDot;
  ES_ULONG              CurAtm;

  ES_SURFACE *          DestDot = *InizSrf;
  ES_SURFACE *          NewDot  = NULL;

  *InizSrf = NULL;

  for(CurAtm = 1; CurAtm <= TotAtm; ++CurAtm) {
    Dot = PrecDot = DestDot;
    while(Dot) {
      if (CurAtm == Dot -> AtmNum) {
        if (*InizSrf) NewDot -> Ptr = Dot;
        else *InizSrf = Dot;
        NewDot = Dot;
        if (Dot == DestDot) PrecDot = DestDot = Dot -> Ptr;
        else PrecDot -> Ptr = Dot -> Ptr;
      } else PrecDot = Dot;
      Dot = Dot -> Ptr;
    } /* End of while */
  } /* End of for (CurAtm)*/

  if (NewDot) NewDot -> Ptr = NULL;
}


/**** Translate the surface ****/

void SrfTranslate(struct surf *surf, float x, float y, float z)
{
  int           i;

  for (i = 0; i < surf -> n_dots; i++) {
    surf -> x[i] += x;
    surf -> y[i] += y;
    surf -> z[i] += z;
  } /* End of for */

  surf -> org.x -= x;
  surf -> org.y -= y;
  surf -> org.z -= z;
}


/**** Save the surface in Insight format ****/

ES_BOOL SrfWrite(char *File, ES_SURFACE *Dot, ES_BRK *Brk)
{
  char                  FileNam[MAX_PATH];
  ES_ATOM               *Atm;
  FILE                  *FH;

  ES_BOOL               Ret        = TRUE;
  ES_ULONG              NMol       = 1;
  ES_ULONG              PrecAtmNum = 0;

  if ((FH = fopen(File, "w")) != NULL) {
    if (fprintf(FH, "DOTS\n") >= 0) {
      SrfName(File, FileNam);
      Upper(FileNam);
      while(Dot && Ret) {
        if (PrecAtmNum != Dot -> AtmNum) {
          Atm = &(Brk -> atom[Dot -> AtmNum - 1]);
          if (fprintf(FH, "ATOM_REF %s:%d:%.4s %.6f %.6f %.6f\n",
                     FileNam, NMol, Atm -> Name.C, Atm -> x, Atm -> y,
                     Atm -> z) < 0)
            Ret = PrintDosErr();
        }
        if ((Ret) && (fprintf(FH, "%.6f %.6f %.6f 120\n", Dot -> x, Dot -> y, Dot -> z) < 0))
          Ret = PrintDosErr();
        PrecAtmNum = Dot -> AtmNum;
        Dot = Dot -> Ptr;
      } /* End of dot loop */
    } else Ret = PrintDosErr();
    fclose(FH);
  } else Ret = PrintDosErr();

  return Ret;
}


struct point surf_center(struct surf* surf)
{
  float         z_min, z_max, x_min, x_max, y_min, y_max;
  int           i;
  struct point  centro;

  z_min = surf -> z[0]; z_max = surf -> z[0];
  x_min = surf -> x[0]; x_max = surf -> x[0];
  y_min = surf -> y[0]; y_max = surf -> y[0];

  for(i = 0; i < surf -> n_dots; i++) {
    z_min = min(z_min, surf -> z[i]);
    x_min = min(x_min, surf -> x[i]);
    y_min = min(y_min, surf -> y[i]);
    z_max = max(z_max, surf -> z[i]);
    x_max = max(x_max, surf -> x[i]);
    y_max = max(y_max, surf -> y[i]);
  } /* End of for (i) */

  centro.x = x_min + (x_max - x_min) / 2.0f;
  centro.y = y_min + (y_max - y_min) / 2.0f;
  centro.z = z_min + (z_max - z_min) / 2.0f;

  return centro;
}


void surf_max_min(struct surf* surf, struct point *smax, struct point *smin)
{
   int i;
   float z_min,z_max,x_min,x_max,y_min,y_max;

   z_min=surf->z[0];z_max=surf->z[0];
   x_min=surf->x[0];x_max=surf->x[0];
   y_min=surf->y[0];y_max=surf->y[0];
   for (i=0;i<surf->n_dots;i++)
   {
      z_min=min(z_min,surf->z[i]);
      x_min=min(x_min,surf->x[i]);
      y_min=min(y_min,surf->y[i]);
      z_max=max(z_max,surf->z[i]);
      x_max=max(x_max,surf->x[i]);
      y_max=max(y_max,surf->y[i]);
   }
   smax->x=x_max;smax->y=y_max;smax->z=z_max;
   smin->x=x_min;smin->y=y_min;smin->z=z_min;
}

float surf_radius_max(struct surf* surf,struct point centro)
{
 float  x,  y,  z;
 int    i;
 float  radius_max = 0.0f;


 for(i = 0; i < surf -> n_dots; i++) {
   x=(surf -> x[i] - centro.x) * (surf -> x[i] - centro.x);
   y=(surf -> y[i] - centro.y) * (surf -> y[i] - centro.y);
   z=(surf -> z[i] - centro.z) * (surf -> z[i] - centro.z);
   radius_max = max(radius_max, sqrtf(x + y + z));
 } /* End of for (i) */

 return radius_max;
}
