
/*************************************************
****      VEGA - Connectivity Generator       ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <math.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#ifdef __WIN32__
#  include <windows.h>
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "bond.h"
#include "comerr.h"
#include "count.h"

#define  DISTCUTOFF       6.0

#ifdef LITTLE_ENDIAN
#  define  H_ELEMENT     0x48
#else
#  define  H_ELEMENT   0x4800
#endif

/**** Definitions ****/

typedef struct _BoxAtm {
  struct _BoxAtm        *Ptr;
  ATOMO                 *Atm;
  float                 Rad;
} VG_CONNBOXATM;

typedef struct {
  VG_CONNBOXATM         *FirstAtm;
  VG_CONNBOXATM         *LastAtm;
} VG_CONNBOX;


/**** Translate the aminoacid name ****/

VG_QCHAR AaTranslateName(VG_QCHAR ResName)
{
  VG_QCHAR      *Ptr;

  VG_QCHAR      Ret = ResName;

  for(Ptr = AAEquivTab; *(*Ptr).C; Ptr += 2) {
    if ((*Ptr).L == ResName.L) {
      ResName.L = Ptr[1].L;
      break;
    }
  } /* End of for */

  return Ret;
}


/**** Set covalent radii ****/

float *AssignCovRad(register ATOMO *Atomo, VG_ULONG TotAtomi)
{
  register ATMTAB    *Tab;
  register float     *CovRad;
  register VG_ULONG  k;

  if ((CovRad = (float *)Alloca(sizeof(float) * TotAtomi))) {
    for(k = 0; k < TotAtomi; ++k) {
      for(Tab = AtmTable; *Tab -> Atom.C; ++Tab) {
        if (Tab -> Atom.S == Atomo -> Elem.S) {
          CovRad[k] = Tab -> CovRad;
          break;
        }
      } /* End of for */
      Atomo = Atomo -> Ptr;
    } /* End of for */
  }

  return CovRad;
}


/****           Set Van Der Waals radii         ****/
/**** If MaxRad is null, the maximum VDW radius ****/
/**** is not calculated.                        ****/

float *AssignVdwRad(register ATOMO *Atm, VG_ULONG TotAtm, float ProRad, float *MaxRad)
{
  VG_BOOL               WrtAtm;
  float                 *VdwRad;
  register float        *Rad;
  register VG_ULONG     k;

  VG_BOOL                  Err = FALSE;

  if ((VdwRad = (float *)Alloca(TotAtm * sizeof(float)))) {
    Rad     = VdwRad;
    if (MaxRad) *MaxRad = 0.0f;
    while(Atm) {
      WrtAtm = FALSE;
      for(k = 0;*AtmTable[k].Atom.C;++k) {
      	if (AtmTable[k].Atom.S == Atm -> Elem.S) {
          *Rad  = AtmTable[k].VdwRad + ProRad;
          if ((MaxRad) && (*Rad > *MaxRad)) *MaxRad = *Rad;
          WrtAtm = TRUE;
          ++Rad;
          break;
      	}
      } /* End of element loop */
      if (!WrtAtm) {
        GlobErr = VG_PARET_MOLCONVDW;
        PrintAtmErr(GetStr(MSG_ERR_CONNECT_UNKVDW), Atm, FALSE);
        Err = TRUE;
      }
      Atm = Atm -> Ptr;
    } /* End of atom loop */
    if (Err) {
      FREE(VdwRad);
      VdwRad = NULL;
    }
  }

  return VdwRad;
}


/****           Set Van Der Waals radii         ****/
/**** If MaxRad is null, the maximum VDW radius ****/
/**** is not calculated.                        ****/

VG_BOOL AssignVdwNew(register ATOMO *Atm, float ProRad, float *MaxRad)
{
  VG_BOOL               WrtAtm;
  register ATMTAB       *Tab;

  VG_BOOL               Ret = TRUE;

  if (MaxRad) *MaxRad = 0.0f;
  while(Atm) {
    WrtAtm = FALSE;
    for(Tab = AtmTable;Tab -> Atom.S;++Tab) {
      	if (Tab -> Atom.S == Atm -> Elem.S) {
          Atm -> Rad  = Tab -> VdwRad + ProRad;
          if ((MaxRad) && (Atm -> Rad > *MaxRad)) *MaxRad = Atm -> Rad;
          WrtAtm = TRUE;
          break;
      	}
    } /* End of element loop */
    if (!WrtAtm) {
      GlobErr = VG_PARET_MOLCONVDW;
      Ret = PrintAtmErr(GetStr(MSG_ERR_CONNECT_UNKVDW), Atm, FALSE);
    }
    Atm = Atm -> Ptr;
  } /* End of atom loop */

  return Ret;
}


/**** This fast code is based on ReConnect() routine  included  in  Qmol ****/
/**** package by J. Gans Cornell University and on the routines included ****/
/**** in VMD  software  by  Theoretical  Biophysics  Group  of  Illinois ****/
/**** University.                                                        ****/

VG_BOOL Connect(ATOMO *InizAtm, VG_ULONG TotAtomi, float ConnTol, VG_BOOL ActiveOnly)
{
  ATOMO                 *Atm1;
  ATMTAB                *Tab;
  float                 Xmin, Xmax, Ymin, Ymax, Zmin, Zmax;
  float                 BoxSizeX, BoxSizeY, BoxSizeZ;
  float                 CutOff, Dist, Dr;
  VG_CONNBOX            *BoxAtom, *NbrBox, *TmpBox;
  VG_CONNBOXATM         *AtmList1, *AtmList2, *StartI;
  VG_LONG               Xb, Yb, Zb, XyTotb, Totb;
  VG_LONG               Axb, Ayb, Azb;
  VG_LONG               Aindex, Bindex;
  VG_LONG               j, Xi, Yi, Zi;

  VG_BOOL               Ret   = TRUE;
  VG_BOOL               UnAcc = FALSE;


  PrintProg(MSG_CONNECT_PROGPMT);

  /**** Calculate the molecule dimensions ****/

  Xmin = Xmax = InizAtm -> x;
  Ymin = Ymax = InizAtm -> y;
  Zmin = Zmax = InizAtm -> z;
  for(Atm1 = InizAtm -> Ptr; Atm1; Atm1 = Atm1 -> Ptr) {
    if (((!ActiveOnly) || (Atm1 -> Active)) &&
        (!(Atm1 -> Flags & VG_ATMF_CENTROID))) {
      if (Atm1 -> x < Xmin) Xmin = Atm1 -> x;
      else if (Atm1 -> x > Xmax) Xmax = Atm1 -> x;

      if (Atm1 -> y < Ymin) Ymin = Atm1 -> y;
      else if (Atm1 -> y > Ymax) Ymax = Atm1 -> y;

      if (Atm1 -> z < Zmin) Zmin = Atm1 -> z;
      else if (Atm1 -> z > Zmax) Zmax = Atm1 -> z;
    }
  } /* End of for */

  /**** Set the box size ****/

  BoxSizeX = (float)((Xmax - Xmin >= 200.0f ? (Xmax - Xmin) / 50.0f : 4.0f));
  BoxSizeY = (float)((Ymax - Ymin >= 200.0f ? (Ymax - Ymin) / 50.0f : 4.0f));
  BoxSizeZ = (float)((Zmax - Zmin >= 200.0f ? (Zmax - Zmin) / 50.0f : 4.0f));
  Xb       = (VG_LONG)((Xmax - Xmin) / BoxSizeX) + 1;
  Yb       = (VG_LONG)((Ymax - Ymin) / BoxSizeY) + 1;
  Zb       = (VG_LONG)((Zmax - Zmin) / BoxSizeZ) + 1;

  XyTotb   = Yb * Xb;
  Totb     = XyTotb * Zb;

  /**** Put the atoms in their box ****/

  if ((BoxAtom = (VG_CONNBOX *)Alloca(sizeof(VG_CONNBOX) * Totb)) != NULL) {
    for(Atm1 = InizAtm; (Ret) && (Atm1); Atm1 = Atm1 -> Ptr) {
      if (((!ActiveOnly) || (Atm1 -> Active)) &&
          (!(Atm1 -> Flags & VG_ATMF_CENTROID))) {
        Axb = (VG_LONG)((Atm1 -> x - Xmin) / BoxSizeX);
        Ayb = (VG_LONG)((Atm1 -> y - Ymin) / BoxSizeY);
        Azb = (VG_LONG)((Atm1 -> z - Zmin) / BoxSizeZ);

        Aindex = Azb * XyTotb + Ayb * Xb + Axb;

        if ((AtmList1 = (VG_CONNBOXATM *)Alloca(sizeof(VG_CONNBOXATM))) != NULL) {
          if (BoxAtom[Aindex].FirstAtm)
            BoxAtom[Aindex].LastAtm -> Ptr = AtmList1;
          else
            BoxAtom[Aindex].FirstAtm = AtmList1;
          AtmList1 -> Atm          = Atm1;
          BoxAtom[Aindex].LastAtm  = AtmList1;
        } else Ret = FALSE;
      }
    } /* End of for */

    if (Ret) {
      ConnTol = ConnTol / 100.0f + 1.0f;

      /**** Assign the covalent radii ****/

      for(j = 0; j < Totb; ++j) {
        for(AtmList1 = BoxAtom[j].FirstAtm; AtmList1; AtmList1 = AtmList1 -> Ptr) {
          for(Tab = AtmTable; *Tab -> Atom.C; ++Tab) {
            if (Tab -> Atom.S == AtmList1 -> Atm -> Elem.S) {
              AtmList1 -> Rad = Tab -> CovRad * ConnTol;
              break;
            }
          } /* End of for */
        } /* End of for */
      } /* End of for */

      /**** Build the connectivity ****/

      Aindex = 0;
      for(Zi = 0; Zi < Zb; Zi++) {
        for(Yi = 0; Yi < Yb; Yi++) {
          for(Xi = 0; Xi < Xb; Xi++) {
            TmpBox = &BoxAtom[Aindex];

            /**** Check boxes that contain at least one atom ****/

            if (TmpBox -> FirstAtm) {
              for(j = 0; j < 14; j ++) {
                switch(j) {
                case 0:   /* Same */
                  Bindex = Aindex;
                  break;

                case 1:   /* X */
                  if (Xi < (Xb - 1))
                    Bindex = Aindex + 1;
                  else continue;
                  break;

                case 2:   /* Y */
                  if (Yi < (Yb - 1))
                    Bindex = Aindex + Xb;
                  else continue;
                  break;

                case 3:   /* Z */
                  if (Zi < (Zb - 1))
                    Bindex = Aindex + XyTotb;
                  else continue;
                  break;

                case 4:   /* XY */
                  if ((Xi < (Xb - 1)) && (Yi < (Yb - 1)))
                    Bindex = Aindex + Xb + 1;
                  else continue;
                  break;

                case 5:   /* XZ */
                  if ((Xi < (Xb - 1)) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb + 1;
                  else continue;
                  break;

                case 6:   /* YZ */
                  if ((Yi < (Yb - 1)) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb + Xb;
                  else continue;
                  break;

                case 7:   /* X-Y */
                  if ((Xi < (Xb - 1)) && (Yi > 0))
                    Bindex = Aindex - Xb + 1;
                  else continue;
                  break;

                case 8:   /* -XZ */
                  if ((Xi > 0) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb - 1;
                  else continue;
                  break;

                case 9:   /* -YZ */
                  if ((Yi > 0) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb - Xb;
                  else continue;
                  break;

                case 10:  /* XYZ */
                  if ((Xi < (Xb - 1)) && (Yi < (Yb - 1)) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb + Xb + 1;
                  else continue;
                  break;

	        case 11:  /* -XYZ */
                  if ((Xi > 0) && (Yi < (Yb - 1)) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb + Xb - 1;
                  else continue;
                  break;

                case 12:  /* X-YZ */
                  if ((Xi < (Xb - 1)) && (Yi > 0) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb - Xb + 1;
                  else continue;
                  break;

                case 13:  /* -X-YZ */
                  if ((Xi > 0) && (Yi > 0) && (Zi < (Zb - 1)))
                    Bindex = Aindex + XyTotb - Xb - 1;
                  else continue;
                  break;
                } /* End of switch */

                NbrBox = &BoxAtom[Bindex];

                /**** Skip empty boxes ****/

                if (NbrBox -> FirstAtm == NULL) continue;

                for(AtmList1 = TmpBox -> FirstAtm; AtmList1; AtmList1 = AtmList1 -> Ptr) {
                  if (Bindex == Aindex) StartI = AtmList1 -> Ptr;
                  else StartI = NbrBox -> FirstAtm;

                  for(AtmList2 = StartI; AtmList2; AtmList2 = AtmList2 -> Ptr) {
                    CutOff  = AtmList1 -> Rad + AtmList2 -> Rad;
                    CutOff *= CutOff;
                    Dr      = AtmList2 -> Atm -> x - AtmList1 -> Atm -> x;
                    if ((Dist = Dr * Dr) < CutOff) {
                      Dr = AtmList2 -> Atm -> y - AtmList1 -> Atm -> y;
                      if ((Dist += Dr * Dr) < CutOff) {
                        Dr = AtmList2 -> Atm -> z - AtmList1 -> Atm -> z;
                        if ((Dist += Dr * Dr) < CutOff) {
                          if ((AtmList1 -> Atm -> NSost < MAXBOND) && (AtmList2 -> Atm -> NSost < MAXBOND)) {
                            if (!BndCheck(AtmList1 -> Atm, AtmList2 -> Atm, NULL)) {
                              AtmList1 -> Atm -> Conn[(int)AtmList1 -> Atm -> NSost]  = AtmList2 -> Atm;
                              AtmList2 -> Atm -> Conn[(int)AtmList2 -> Atm -> NSost]  = AtmList1 -> Atm;
                              AtmList1 -> Atm -> Order[(int)AtmList1 -> Atm -> NSost] = VG_BOND_SINGLE;
                              AtmList2 -> Atm -> Order[(int)AtmList2 -> Atm -> NSost] = VG_BOND_SINGLE;
                              ++AtmList1 -> Atm -> NSost;
                              ++AtmList2 -> Atm -> NSost;
                            }
                          } else if (!UnAcc) {
                            CatErr(MSG_ERR_CONNECT_UNACCBD);
                            UnAcc = TRUE;
                          }
                        }
                      }
                    }
                  } /* End of for */
                } /* End of for */

              } /* End of j for */
            }

            ++Aindex;
          } /* End of X box for */
        } /* End of Y box for */
      } /* End of Z box for */
    }

    /**** Free all lists ****/

    for(j = 0; j < Totb; ++j) {
      AtmList1 = BoxAtom[j].FirstAtm;
      while(AtmList1) {
        AtmList2 = AtmList1 -> Ptr;
        FREE(AtmList1);
        AtmList1 = AtmList2;
      } /* End of while */
    } /* End of for */
    FREE(BoxAtom);
  } else Ret = FALSE;

  return Ret;
}


/**** Get the atom mass ****/

float GetAtmMass(register VG_UWORD Elem)
{
  register ATMTAB       *Tab = AtmTable;

  while(Tab -> Atom.S) {
    if (Tab -> Atom.S == Elem)
      return Tab -> Mass;
    ++Tab;
  } /* End of while */

  return 0.0f;
}


/**** Get the atomic number ****/

VG_UWORD GetAtmNum(register VG_UWORD Elem)
{
  register ATMTAB       *Tab = AtmTable;
  register VG_UWORD     Ret  = 0;

  while(Tab -> Atom.S) {
    if (Tab -> Atom.S == Elem) {
      Ret = Tab -> AtmNum;
      break;
    }
    ++Tab;
  } /* End of while */

  return Ret;
}


/**** Normalize the coordinates ****/

void Normalize(register ATOMO *Atomo)
{
  register VG_SURFACE   *Dot;
  XYZ                   GeoCent;

  PrintProg(MSG_CONNECT_NORMPROG);

  CalcGeoCent(Atomo, &GeoCent, FALSE);

  /**** Translate all atoms ****/

  while(Atomo) {
    Atomo -> x -= GeoCent.x;
    Atomo -> y -= GeoCent.y;
    Atomo -> z -= GeoCent.z;
    Atomo       = Atomo -> Ptr;
  }

  /**** Translate the surface ****/

  if (BegSrf) {
    for(Dot = BegSrf; Dot; Dot = Dot -> Next) {
      Dot -> x -= GeoCent.x;
      Dot -> y -= GeoCent.y;
      Dot -> z -= GeoCent.z;
    } /* End of for */
  }
}


#ifdef __VG_OPENGL

/**** Clear the connectivity ****/

void ConnClear(register ATOMO *Atm, VG_BOOL ActiveOnly)
{
  register VG_UBYTE      k = 0;

  while(Atm) {
    if ((!ActiveOnly) || (Atm -> Active)) {
      for(k = 0; k < Atm -> NSost; ++k) {
        Atm -> Conn[k]  = NULL;
        Atm -> Order[k] = 0;
      } /* End of while */
      Atm -> NSost = 0;
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Get the covalent radius ****/

float GetAtmCovRad(register VG_UWORD Elem)
{
  register ATMTAB       *Tab   = AtmTable;
  register float        CovRad = 0.77;

  while(Tab -> Atom.S) {
    if (Tab -> Atom.S == Elem) {
      CovRad = Tab -> CovRad;
      break;
    }
    ++Tab;
  } /* End of while */

  return CovRad;
}


/**** Get the Van Der Waals radius ****/

float GetAtmVdwRad(register VG_UWORD Elem)
{
  register ATMTAB       *Tab   = AtmTable;
  register float        VdwRad = 1.70;

  while(Tab -> Atom.S) {
    if (Tab -> Atom.S == Elem) {
      VdwRad = Tab -> VdwRad;
      break;
    }
    ++Tab;
  } /* End of while */

  return VdwRad;
}


/**** Convert a 3D structure to 2D ****/

#if 0

void Mol3dTo2d(ATOMO *InizAtm)
{
  ATOMO         *Atm, **Ptr;
  float         *AngleVal, *BondVal, Val, Rad;
  VG_UBYTE      *ConMtx;
  VG_ULONG      Angles, Bonds, k, j;

  ATOMO         **AngleMtx    = NULL;
  ATOMO         **BondMtx     = NULL;

  if ((Bonds = GenBondMtx(InizAtm, &BondMtx)) != 0) {
    if ((BondVal = (float *)Alloca(Bonds * sizeof(float))) != NULL) {
      if ((Angles = GenAngleMtx(BondMtx, &AngleMtx)) != 0) {
        if ((AngleVal = (float *)Alloca(Angles * sizeof(float))) != NULL) {
          if ((ConMtx = (VG_UBYTE *)Alloca(TotalAtm)) != NULL) {

            /**** Calculate all bond distances ****/

            for(k = 0, Ptr = BondMtx; k < Bonds; ++k, Ptr += 2) {
              BondVal[k] = SQR(Quad(Ptr[0] -> x - Ptr[1] -> x) +
                               Quad(Ptr[0] -> y - Ptr[1] -> y) +
                               Quad(Ptr[0] -> z - Ptr[1] -> z));
            } /* End of for */

            /**** Calculate all angles ****/

            for(k = 0, Ptr = AngleMtx; k < Angles; ++k, Ptr += 3)
              AngleVal[k] = BondAnglef(Ptr[0], Ptr[1], Ptr[2]);

            /**** Clear the Z coordinates ****/

            for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) Atm -> z = 0.0f;

            /**** Set the bond lengths ****/

            for(k = 0, Ptr = BondMtx; k < Bonds; ++k, Ptr += 2) {
              Val = BondVal[k] / SQR(Quad(Ptr[0] -> x - Ptr[1] -> x) +
                    Quad(Ptr[0] -> y - Ptr[1] -> y));

              /**** Build the move list ****/

              j = 0;
              ZeroMemory(ConMtx, TotalAtm);
              ConMtx[Ptr[1] -> Num - 1] = TRUE;
              BndFindConnAtms(ConMtx, Ptr[1], &j);

              /**** Translate the atoms ****/

              for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
                if (ConMtx[Atm -> Num - 1]) {
                  Atm -> x = (Atm -> x - Ptr[0] -> x) * Val + Ptr[0] -> x;
                  Atm -> y = (Atm -> y - Ptr[0] -> y) * Val + Ptr[0] -> y;
                }
              } /* End of for */
            } /* End of for */

            /**** Set the angles ****/

            for(k = 0, Ptr = AngleMtx; k < Angles; ++k, Ptr += 3) {
              Val = (BondAnglef(Ptr[0], Ptr[1], Ptr[2]) - AngleVal[k]) * DEG_TO_RAD;

              /**** Build the move list ****/

              j = 0;
              ZeroMemory(ConMtx, TotalAtm);
              ConMtx[Ptr[2] -> Num - 1] = TRUE;
              BndFindConnAtms(ConMtx, Ptr[2], &j);
              ConMtx[Ptr[0] -> Num - 1] = FALSE;
              ConMtx[Ptr[1] -> Num - 1] = FALSE;

              /**** Rotate the atoms ****/

              for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
                if (ConMtx[Atm -> Num - 1]) {
                  Atm -> x = (Atm -> x - Ptr[1] -> x) * cos(Val) + Ptr[1] -> x;
                  Atm -> y = (Atm -> y - Ptr[1] -> y) * sin(Val) + Ptr[1] -> y;
                }
              } /* End of for */
            } /* End of for */

            FREE(ConMtx);
          }
          FREE(AngleVal);
        }
        FREE(AngleMtx);
      }
      FREE(BondVal);
    }
    FREE(BondMtx);
  }

  Normalize(InizAtm);
}
#endif

#endif



