
/*************************************************
****            VEGA - Zip database          ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <malloc.h>
#include <time.h>

#define  Byte zByte
#include <zlib.h>
#include "../ZipLib/unzip.h"
#include "../ZipLib/zip.h"
#undef  Byte

#define  USE_ZLIB
#define  WINDLL
#include "../Zip/api.h"

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "db_engine.h"

#ifdef WIN32
#  include <windows.h>
#endif

/**** Constants ****/

#define  VG_DBZIP_BUFSIZE               65536

/**** Local prototypes ****/

static VG_BOOL DbZipOpenRead(VG_DBINFO *);
static int     UnZipFile(VG_DBINFO *, char *, char *);
static int     ZipFile(VG_DBINFO *, char *, BOOL);

/**** Local variables ****/

static char                    TmpFileName[VG_MAX_PATH];

#ifdef WIN32

/**** Library function pointers ****/

int            (WINAPI *VG_unzClose)(unzFile)                                    = NULL;
int            (WINAPI *VG_unzCloseCurrentFile)(zipFile)                         = NULL;
int            (WINAPI *VG_unzGetCurrentFileInfo)(unzFile, unz_file_info *,
               char *, uLong, void *, uLong, char *, uLong)                      = NULL;
int            (WINAPI *VG_unzGetGlobalInfo)(unzFile, unz_global_info *)         = NULL;
int            (WINAPI *VG_unzGoToFirstFile)(unzFile)                            = NULL;
int            (WINAPI *VG_unzGoToNextFile)(unzFile)                             = NULL;
unzFile        (WINAPI *VG_unzOpen)(const char *)                                = NULL;
int            (WINAPI *VG_unzOpenCurrentFile)(unzFile)                          = NULL;
int            (WINAPI *VG_unzReadCurrentFile)(unzFile, voidp, unsigned)         = NULL;
int            (WINAPI *VG_ZpArchive)(ZCL)                                       = NULL;
int            (WINAPI *VG_ZpInit)(LPZIPUSERFUNCTIONS)                           = NULL;
BOOL           (WINAPI *VG_ZpSetOptions)(LPZPOPT)                                = NULL;

/**** Global variables ****/

extern HINSTANCE                hZip32;
extern HINSTANCE                hZipLib;

/**** Local variables ****/

static LPZIPUSERFUNCTIONS       lpZipUserFunctions;

/**** Local prototypes ****/

static VG_BOOL DbZipLibInit(void);
static VG_BOOL DbZip32Init(void);

static int WINAPI Zip32Password(LPSTR, int, LPCSTR, LPCSTR);
static int WINAPI Zip32Print(char far *, unsigned long);
static int WINAPI Zip32Comment(char far *);

/**** Global prototypes ****/

VG_BOOL        GetProcAddresses(HINSTANCE *, LPCSTR, INT, ... );
#endif


/**** Close the database ****/

void DbZipClose(VG_DBINFO *Dbh)
{
  if (Dbh -> Handle) {
    VG_unzClose(Dbh -> Handle);
    Dbh -> Handle = NULL;
  }
  if (Dbh -> GlobalInfo) FREE(Dbh -> GlobalInfo);
}


/**** Create a new database ****/

VG_BOOL DbZipCreate(VG_DBINFO *Dbh)
{
  VG_BOOL       Ret = FALSE;

#ifdef WIN32
  if (!DbZip32Init()) return FALSE;
#endif

  ChangeExt(Dbh -> FileName, "zip", NULL);
  GetTmpDir(TmpFileName, VG_MAX_PATH);
  strcat(TmpFileName, DbDataStr);
  if (DbDatCreate(TmpFileName, Dbh)) {
    if (ZipFile(Dbh, TmpFileName, FALSE) == 0)
      Ret = TRUE;
#ifdef WIN32
    DeleteFile(TmpFileName);
#else
    remove(TmpFileName);
#endif
  }

  return Ret;
}


/**** Get a molucule ****/

ATOMO *DbZipGet(VG_DBINFO *Dbh, char *MolName, VG_LONG *FormatID)
{
  ATOMO                 *LastSeg;
  VG_ULONG              TmpTotAtm;

  ATOMO                 *TmpIniz = NULL;

  if (DbZipOpenRead(Dbh)) {
    GetTmpFileName(TmpFileName);
    if (UnZipFile(Dbh, TmpFileName, MolName) == UNZ_OK) {
      LastSeg         = LastAtm;
      GLOBSW_CONNCALC = TRUE;
      TmpIniz         = Loader(TmpFileName, &TmpTotAtm, FormatID, NULL, 0, &BegSrf, &TotalSrf);
      if ((TmpIniz) && (TmpIniz != (ATOMO *)-1) && (TmpIniz != (ATOMO *)-2)) {
        TotalAtm += TmpTotAtm;
        if (BegAtm) {
          LastSeg -> Ptr  = TmpIniz;
          TmpIniz -> Prev = LastSeg;
        } else BegAtm = TmpIniz;
        if (GLOBSW_CONNCALC) Connect(TmpIniz, TmpTotAtm, Prefs.CONNTOL, FALSE);
        else GLOBSW_CONNCALC = TRUE;
      } else TmpIniz = NULL;
    }
#ifdef WIN32
    DeleteFile(TmpFileName);
#else
    remove(TmpFileName);
#endif
  }

  return TmpIniz;
}


/**** Init the DLLs ****/

#ifdef WIN32
static VG_BOOL DbZipLibInit(void)
{
  if (hZipLib) return TRUE;

  if (GetProcAddresses(&hZipLib, "ziplib.dll", 9,
                       &VG_unzClose             , "unzClose"             ,
                       &VG_unzCloseCurrentFile  , "unzCloseCurrentFile"  ,
                       &VG_unzGetCurrentFileInfo, "unzGetCurrentFileInfo",
                       &VG_unzGetGlobalInfo     , "unzGetGlobalInfo"     ,
                       &VG_unzGoToFirstFile     , "unzGoToFirstFile"     ,
                       &VG_unzGoToNextFile      , "unzGoToNextFile"      ,
                       &VG_unzOpen              , "unzOpen"              ,
                       &VG_unzOpenCurrentFile   , "unzOpenCurrentFile"   ,
                       &VG_unzReadCurrentFile   , "unzReadCurrentFile"
                      ) == FALSE)
    return FALSE;

  return TRUE;
}

static VG_BOOL DbZip32Init(void)
{
  VG_BOOL       Ret = FALSE;

  if (!hZip32) {
    if (GetProcAddresses(&hZip32, "zip32.dll", 3,
                         &VG_ZpArchive            , "ZpArchive"            ,
                         &VG_ZpInit               , "ZpInit"               ,
                         &VG_ZpSetOptions         , "ZpSetOptions"
                        )) {
      if ((lpZipUserFunctions = (LPZIPUSERFUNCTIONS)Alloca(sizeof(ZIPUSERFUNCTIONS))) != NULL) {
        lpZipUserFunctions -> print    = Zip32Print;
        lpZipUserFunctions -> password = Zip32Password;
        lpZipUserFunctions -> comment  = Zip32Comment;
        if (VG_ZpInit(lpZipUserFunctions))
          Ret = TRUE;
        else
          FREE(lpZipUserFunctions);
      }
      if (!Ret) CatErr(MSG_ERR_DBENGINE_ZIP32INIT);
    }
  } else Ret = TRUE;

  return Ret;
}
#endif


/**** Read the molecule list ****/

VG_BOOL DbZipMolListRead(VG_DBINFO *Dbh)
{
  char                  ZipFileName[VG_MAX_PATH];
  VG_ULONG              j;
  unz_file_info         ZipFI;

  if (!DbZipOpenRead(Dbh)) return FALSE;

  if (VG_unzGoToFirstFile(Dbh -> Handle) == UNZ_OK) {
    for(j = 0; j < ((unz_global_info *)Dbh -> GlobalInfo) -> number_entry; ++j) {
      if (VG_unzGetCurrentFileInfo(Dbh -> Handle, &ZipFI, ZipFileName,
                                   VG_MAX_PATH, NULL, 0, NULL, 0) == UNZ_OK) {
        if ((strcasecmp(ZipFileName, DbDataStr)) &&
            (DbMolListAdd(Dbh, ZipFileName) == NULL))
          return FALSE;
      } else return FALSE;
      if (VG_unzGoToNextFile(Dbh -> Handle) != UNZ_OK)
        return TRUE;
    } /* End of for */
  } else return FALSE;

  return TRUE;
}


/**** Open an existing database ****/

VG_BOOL DbZipOpen(VG_DBINFO *Dbh)
{
  VG_BOOL       Ret = FALSE;

#ifdef WIN32
  if (!DbZipLibInit()) return FALSE;
#endif

  if ((Dbh -> GlobalInfo = (unz_global_info *)Alloca(sizeof(unz_global_info))) != NULL) {
    if (DbZipOpenRead(Dbh)) {
      GetTmpFileName(TmpFileName);
      if ((UnZipFile(Dbh, TmpFileName, (char *)DbDataStr) == UNZ_OK) &&
          (DbDatRead(TmpFileName, Dbh))) Ret = TRUE;
#ifdef WIN32
      DeleteFile(TmpFileName);
#else
      remove(TmpFileName);
#endif
    } else FREE(Dbh -> GlobalInfo);
  }


  return Ret;
}


/**** Open a database to read ****/

static VG_BOOL DbZipOpenRead(VG_DBINFO *Dbh)
{
  if (Dbh -> Handle) return TRUE;

  if (((Dbh -> Handle = VG_unzOpen(Dbh -> FileName)) != NULL) &&
      (VG_unzGetGlobalInfo(Dbh -> Handle, (unz_global_info *)Dbh -> GlobalInfo) == UNZ_OK)) {
    return TRUE;
  }

  return FALSE;
}


/**** Put a molecule ****/

VG_BOOL DbZipPut(VG_DBINFO *Dbh, char *MolName)
{
  VG_BOOL               Ret = FALSE;

  GetTmpDir(TmpFileName, VG_MAX_PATH);
  strcat(TmpFileName, MolName);
  if ((Saver(TmpFileName, Dbh -> MolFormat, 0, BegAtm, TotalAtm)) &&
      (ZipFile(Dbh, TmpFileName, FALSE) == 0))
    Ret = TRUE;
#ifdef WIN32
  DeleteFile(TmpFileName);
#else
  remove(TmpFileName);
#endif

  return Ret;
}


/**** Remove a molecule ****/

VG_BOOL DbZipRemove(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm)
{
  if (ZipFile(Dbh, Itm -> MolName, TRUE) == 0)
    return TRUE;

  return FALSE;
}


/**** Unzip a single file ****/

static int UnZipFile(VG_DBINFO *Dbh, char *Dest, char *Src)
{
  char                  ZipFileName[VG_MAX_PATH];
  char                  *WriteBuf;
  DWORD                 BytesWritten;
  HANDLE                FH;
  int                   k;
  unsigned int          j, Len;
  unz_file_info         ZipFI;

  int                   Err = UNZ_OK;

  /**** Locate the file ingnoring the path ****/

  if ((Err = VG_unzGoToFirstFile(Dbh -> Handle)) == UNZ_OK) {
    Len = strlen(Src);
    for(j = 0; (Err == UNZ_OK) && (j < ((unz_global_info *)Dbh -> GlobalInfo) -> number_entry); ++j) {
      if ((Err = VG_unzGetCurrentFileInfo(Dbh -> Handle, &ZipFI, ZipFileName,
                                          VG_MAX_PATH, NULL, 0, NULL, 0)) == UNZ_OK) {
        if ((ZipFI.size_filename >= Len) &&
            (!stricmp(GetFileName(ZipFileName), Src))) {

          /**** Unzip the file ****/

          if ((WriteBuf = (char *)Alloca(VG_DBZIP_BUFSIZE)) != NULL) {
            if ((Err = VG_unzOpenCurrentFile(Dbh -> Handle)) == UNZ_OK) {
              if ((FH = CreateFile(Dest, GENERIC_WRITE, FILE_SHARE_READ, NULL,
                   CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE) {
                while((k = VG_unzReadCurrentFile(Dbh -> Handle, WriteBuf, VG_DBZIP_BUFSIZE)) > 0) {
                  if (!WriteFile(FH, WriteBuf, k, &BytesWritten, NULL)) {
                    WinErr();
                    break;
                  }
                } /* End of while */
                CloseHandle(FH);
              } else {
                WinErr();
                Err = -1;
              }
              VG_unzCloseCurrentFile(Dbh -> Handle);
            }
            FREE(WriteBuf);
          } else {
            WinErr();
            Err = -1;
          }
          break;
        }
        Err = VG_unzGoToNextFile(Dbh -> Handle);
      }
    } /* End of for */
  }

  return Err;
}


/**** Zip a single file ****/

static int ZipFile(VG_DBINFO *Dbh, char *SrcFile, BOOL Del)
{
  char          *File[1];
  char          Path[VG_MAX_PATH];
  ZCL           ZpZCL;
  ZPOPT         *ZpOpt;

  int           Err = TRUE;

#ifdef WIN32
  if (!DbZip32Init()) return Err;
#endif

  if (Dbh -> Handle) {
    VG_unzClose(Dbh -> Handle);
    Dbh -> Handle = NULL;
  }
  if ((ZpOpt = (ZPOPT *)Alloca(sizeof(ZPOPT))) != NULL) {
    strcpy(Path, SrcFile);
    GetFilePath(Path);
    *File = GetFileName(SrcFile);
    ZpOpt -> szRootDir      = Path;
    ZpOpt -> fDeleteEntries = Del;
    ZpZCL.argc              = 1;
    ZpZCL.lpszZipFN         = Dbh -> FileName;
    ZpZCL.FNV               = File;
    if (VG_ZpSetOptions(ZpOpt))
      Err = VG_ZpArchive(ZpZCL);
    FREE(ZpOpt);
  }

  return Err;
}

#ifdef WIN32

/**** Password entry routine ****/

static int WINAPI Zip32Password(LPSTR p, int n, LPCSTR m, LPCSTR name)
{
  return TRUE;
}


/**** Dummy "print" routine ****/

static int WINAPI Zip32Print(char far *buf, unsigned long size)
{
//  LocPrintf(stdout, "%s", buf);

  return size;
}


/**** Dummy "comment" routine ****/

static int WINAPI Zip32Comment(char far *szBuf)
{
  szBuf[0] = '\0';

  return TRUE;
}

#endif


