
/************************************************
****          VEGA - Hooke minimizer         ****
**** Copyright 1995-2003 Alessandro Pedretti ****
************************************************/


/*
 * Nonlinear Optimization using the algorithm of Hooke and Jeeves
 * Original code by: Mark G. Johnson
 */

/* Find a point X where the nonlinear function f(X) has a local    */
/* minimum.  X is an n-vector and f(X) is a scalar.  In mathe-	   */
/* matical notation  f: R^n -> R^1.  The objective function f()    */
/* is not required to be continuous.  Nor does f() need to be	   */
/* differentiable.  The program does not use or require            */
/* derivatives of f().                                             */

/* The software user supplies three things: a subroutine that      */
/* computes f(X), an initial "starting guess" of the minimum point */
/* X, and values for the algorithm convergence parameters.  Then   */
/* the program searches for a local minimum, beginning from the    */
/* starting guess, using the Direct Search algorithm of Hooke and  */
/* Jeeves.                                                         */

/* This C program is adapted from the Algol pseudocode found in    */
/* "Algorithm 178: Direct Search" by Arthur F. Kaupe Jr., Commun-  */
/* ications of the ACM, Vol 6. p.313 (June 1963).  It includes the */
/* improvements suggested by Bell and Pike (CACM v.9, p. 684, Sept */
/* 1966) and those of Tomlin and Smith, "Remark on Algorithm 178"  */
/* (CACM v.12).  The original paper, which I don't recommend as    */
/* highly as the one by A. Kaupe, is:  R. Hooke and T. A. Jeeves,  */
/* "Direct Search Solution of Numerical and Statistical Problems", */
/* Journal of the ACM, Vol. 8, April 1961, pp. 212-229.            */

/* Calling sequence:                                               */
/*  int hooke(nvars, startpt, endpt, rho, epsilon, itermax)        */
/*                                                                 */
/*     nvars	   {an integer}  This is the number of dimensions  */
/*		   in the domain of f().  It is the number of      */
/*		   coordinates of the starting point (and the      */
/*		   minimum point.)				   */
/*     startpt	   {an array of doubles}  This is the user-	   */
/*		   supplied guess at the minimum.		   */
/*     endpt	   {an array of doubles}  This is the location of  */
/*		   the local minimum, calculated by the program    */
/*     rho	   {a double}  This is a user-supplied convergence */
/*		   parameter (more detail below), which should be  */
/*		   set to a value between 0.0 and 1.0.	Larger	   */
/*		   values of rho give greater probability of	   */
/*		   convergence on highly nonlinear functions, at a */
/*		   cost of more function evaluations.  Smaller	   */
/*		   values of rho reduces the number of evaluations */
/*		   (and the program running time), but increases   */
/*		   the risk of nonconvergence.	See below.	   */
/*     epsilon	   {a double}  This is the criterion for halting   */
/*		   the search for a minimum.  When the algorithm   */
/*		   begins to make less and less progress on each   */
/*		   iteration, it checks the halting criterion: if  */
/*		   the stepsize is below epsilon, terminate the    */
/*		   iteration and return the current best estimate  */
/*		   of the minimum.  Larger values of epsilon (such */
/*		   as 1.0e-4) give quicker running time, but a	   */
/*		   less accurate estimate of the minimum.  Smaller */
/*		   values of epsilon (such as 1.0e-7) give longer  */
/*		   running time, but a more accurate estimate of   */
/*		   the minimum. 				   */
/*     itermax	   {an integer}  A second, rarely used, halting    */
/*		   criterion.  If the algorithm uses >= itermax    */
/*		   iterations, halt.				   */


/* The user-supplied objective function f(x,n) should return a C   */
/* "double".  Its  arguments are  x -- an array of doubles, and    */
/* n -- an integer.  x is the point at which f(x) should be	   */
/* evaluated, and n is the number of coordinates of x.	That is,   */
/* n is the number of coefficients being fitted.		   */

/* rho, the algorithm convergence control			   */
/*	The algorithm works by taking "steps" from one estimate of */
/*    a minimum, to another (hopefully better) estimate.  Taking   */
/*    big steps gets to the minimum more quickly, at the risk of   */
/*    "stepping right over" an excellent point.  The stepsize is   */
/*    controlled by a user supplied parameter called rho.  At each */
/*    iteration, the stepsize is multiplied by rho  (0 < rho < 1), */
/*    so the stepsize is successively reduced.			   */
/*	Small values of rho correspond to big stepsize changes,    */
/*    which make the algorithm run more quickly.  However, there   */
/*    is a chance (especially with highly nonlinear functions)	   */
/*    that these big changes will accidentally overlook a	   */
/*    promising search vector, leading to nonconvergence.	   */
/*	Large values of rho correspond to small stepsize changes,  */
/*    which force the algorithm to carefully examine nearby points */
/*    instead of optimistically forging ahead.	This improves the  */
/*    probability of convergence.				   */
/*	The stepsize is reduced until it is equal to (or smaller   */
/*    than) epsilon.  So the number of iterations performed by	   */
/*    Hooke-Jeeves is determined by rho and epsilon:		   */
/*	    rho**(number_of_iterations) = epsilon		   */
/*	In general it is a good idea to set rho to an aggressively */
/*    small value like 0.5 (hoping for fast convergence).  Then,   */
/*    if the user suspects that the reported minimum is incorrect  */
/*    (or perhaps not accurate enough), the program can be run	   */
/*    again with a larger value of rho such as 0.85, using the	   */
/*    result of the first minimization as the starting guess to    */
/*    begin the second minimization.				   */


#include <stdio.h>
#include <math.h>
#include <time.h>

#include "gl_hooke.h"


/**** Global variables ****/

static unsigned int     funevals = 0;

/**** Local prototypes ****/

static float best_nearby(float *, float *, float, unsigned int,
                         float (*EvalFunc)(float *, void *), void *EvalArg);


/**** Given a point, look for a better one nearby, one coord at a time ****/

static float best_nearby(register float *delta, float *InizPoint,
                         float prevbest, unsigned int nvars,
                         float (*EvalFunc)(float *, void *), void *EvalArg)
{
  float			ZMtx[VARS];
  float			ftmp;

  float			minf   = prevbest;
  register float	*point = InizPoint;
  register float	*z     = ZMtx;
  register unsigned int	i      = nvars;


  while(i--) *z++ = *point++;
  i     = nvars;
  point = InizPoint;
  z     = ZMtx;
  while(i--) {
    *z = *point + *delta;

    /**** Function to calculate the energy ****/

    ftmp = EvalFunc(ZMtx, EvalArg);
    ++funevals;

    if (ftmp < minf) minf = ftmp;
    else {
      *delta = 0.0 - *delta;
      *z     = *point + *delta;

      /**** Function to calculate the energy ****/

      ftmp = EvalFunc(ZMtx, EvalArg);
      ++funevals;

      if (ftmp < minf) minf = ftmp;
      else *z = *point;
    }
    ++delta;
    ++point;
    ++z;
  }

  i     = nvars;
  point = InizPoint;
  z     = ZMtx;
  while(i--) *point++ = *z++;

  return minf;
}


/**** Hooke minimization routine ****/

float MinHooke(unsigned int nvars, float *startpt, float *endpt,
               unsigned int Steps, float Rms, float rho, float (*EvalFunc)(float *, void *),
               void *EvalArg)
{
  short         keep;
  float		delta[VARS];
  float		newf, fbefore, tmp;
  float		xbefore[VARS], newx[VARS];
  unsigned int	i;

  float		steplength = rho;
  int           iadj       = 0;
  unsigned int 	iters      = 0;


  for (i = 0; i < nvars; i++) {
    newx[i]  = xbefore[i] = startpt[i];
    delta[i] = fabs(startpt[i] * rho);
    if (delta[i] == 0.0) delta[i] = rho;
  }
  fbefore = EvalFunc(newx, EvalArg);

  newf = fbefore;
  while ((iters < Steps) && (steplength > Rms)) {
    iters++;
    iadj++;

    /* Find best new point, one coord at a time */

    for (i = 0; i < nvars; i++) newx[i] = xbefore[i];
    newf = best_nearby(delta, newx, fbefore, nvars, EvalFunc, EvalArg);

    /* If we made some improvements, pursue that direction */

    keep = 1;
    while ((newf < fbefore) && (keep)) {
      iadj = 0;
      for (i = 0; i < nvars; i++) {

        /* Firstly, arrange the sign of delta[] */

        if (newx[i] <= xbefore[i]) delta[i] = 0.0 - fabs(delta[i]);
        else delta[i] = fabs(delta[i]);

        /* Now, move further in this direction */

        tmp        = xbefore[i];
        xbefore[i] = newx[i];
        newx[i]    = newx[i] + newx[i] - tmp;
      }
      fbefore = newf;
      newf    = best_nearby(delta, newx, fbefore, nvars, EvalFunc, EvalArg);

      /* If the further (optimistic) move was bad ... */

      if (newf >= fbefore) break;

      /* Make sure that the differences between the new */
      /* and the old points are due to actual           */
      /* displacements; beware of roundoff errors that  */
      /* might cause newf < fbefore                     */

      keep = 0;
      for (i = 0; i < nvars; i++) {
        keep = 1;
	if (fabs(newx[i] - xbefore[i]) > (0.5 * fabs(delta[i]))) break;
        else keep = 0;
      }
    }
    if ((steplength >= Rms) && (newf >= fbefore)) {
      steplength = steplength * rho;
      for (i = 0; i < nvars; i++) delta[i] *= rho;
    }
  }

  for (i = 0; i < nvars; i++) endpt[i] = xbefore[i];

  return fbefore;
}
