
/*************************************************
****         VEGA - IFF loader & saver        ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * Uses an Amiga IFF chunk structure to save molecule.
 * Now uses special chunks to include VEGA extra infos.
 *
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <malloc.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "bond.h"
#include "iff.h"
#include "surface.h"

#ifdef __VG_OPENGL
#  define  __DEFSONLY
#  include "gl_global.h"
#  include "gl_matrix.h"

extern float            RotMat[4][4];
extern VG_GLPREFS       ViewPrefs;

#endif

/**** Local prototypes ****/

static VG_BOOL IffRdXyz(FILE *, XYZ *);
static VG_BOOL IffWrtHdr(FILE *, VG_LONG, VG_ULONG);
static VG_BOOL IffWrtSrf(FILE *, VG_SURFACE *, VG_ULONG);
static VG_BOOL IffWrtXyz(FILE *, XYZ *);


/**** Interchange File Format (IFF) Loader ****/

ATOMO *IFFLoad(FILE *IFFIN, VG_ULONG IFFLen, VG_ULONG *TotAtomi,
               VG_SURFACE **InizSrf, VG_ULONG *TotSrf)
{
  ATOMO         **ConMtx;
  VG_BYTE       Size;
  IFFHDR        ChunkHdr;
  VG_BYTE       ChainID;
  VG_QCHAR      ResName;
  VG_SURFACE    *Dot;
  VG_UBYTE      Col[4];
  VG_ULONG      nb, i, k, ResNum, Tot;
  VG_UWORD      j;
  VECTYPE       DCoord;

  ATOMO         *Atm     = NULL;
  ATOMO         *InizAtm = (void *)-1;

  /**** Connectivity structure ****/

  struct __ConAt {
    VG_ULONG i, j;
    VG_UBYTE Type;
  } ConAt;

  /**** Switches ****/

  VG_BOOL       Ret     = TRUE;
  VG_BOOL       Tst1    = FALSE;  /* Single precision coordinates ? */
  VG_BOOL       Tst2    = FALSE;  /* Residue number readed ?        */

  if (fread(&ChunkHdr, sizeof(IFFHDR), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
    Swap(&ChunkHdr.Size);
    Swap(&ChunkHdr.ID);
#endif
    if (ChunkHdr.ID == IFF_ATOMO) {
      if (fread(&Tot, sizeof(VG_ULONG), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
          Swap(&Tot);
#endif
        if (Tot > 0) {

          /**** Allocate the memory for each atom ****/

          for(k = 0;(Ret) && (k < Tot);++k) {
            if ((Atm = AllocAtm(&InizAtm, TotAtomi)) != NULL) {
              if (fread(Atm -> Elem.C, 2, 1, IFFIN) == 1) {
                if (Atm -> Elem.C[1] == ' ') Atm -> Elem.C[1] = 0;
                *Atm -> Name.C    = *Atm -> Elem.C;
                Atm -> Name.C[1]  = Atm -> Elem.C[1];
              } else Ret = PrintDosErr();
            } else Ret = FALSE;
          } /* End of atom loop */
          if (Ret) {
            LastAtm       = Atm;
            Atm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
            if ((ConMtx = (ATOMO **)Alloca(sizeof(ATOMO *) * Tot)) != NULL) {
              k = 0;
              for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
                ConMtx[k++] = Atm;

              while((Ret) && (fread(&ChunkHdr, sizeof(IFFHDR), 1, IFFIN))) {
#ifdef LITTLE_ENDIAN
                Swap(&ChunkHdr.Size);
                Swap(&ChunkHdr.ID);
#endif
                Atm = InizAtm;

                switch(ChunkHdr.ID) {
                case IFF_XYZ1:  /* Cartesian coordinate chunk (single precision) */
                  do {
                    if (fread(&Atm -> x, sizeof(float) * 3, 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                      Swap(&Atm -> x);
                      Swap(&Atm -> y);
                      Swap(&Atm -> z);
#endif
                      Atm = Atm -> Ptr;
                    } else Ret = PrintDosErr();
                  } while((Ret) && (Atm));
                  Tst1 = TRUE;
                  break;

                case IFF_XYZ2:  /* Cartesian coordinate chunk (double precision) */
                 if (!Tst1) {
                   do {
                     if (fread(&DCoord, sizeof(VECTYPE), 1, IFFIN) == 1) {
#ifndef LITTLE_ENDIAN
                       Atm -> x = (float)DCoord.x;
                       Atm -> y = (float)DCoord.y;
                       Atm -> z = (float)DCoord.z;
#else
                       Atm -> x = (float)SwapD(DCoord.x);
                       Atm -> y = (float)SwapD(DCoord.y);
                       Atm -> z = (float)SwapD(DCoord.z);
#endif
                       Atm      = Atm -> Ptr;
                     } else Ret = PrintDosErr();
                   } while((Ret) && (Atm));
                 }
                 break;

                case IFF_CONX:  /* Connectivity chunk */

                  if (fread(&nb, sizeof(VG_ULONG), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                    Swap(&nb);
#endif
                    for(k = 0;(Ret) && (k < nb);++k) {
                      if (fread(&ConAt, sizeof(VG_ULONG) * 2 + 1, 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                        Swap(&ConAt.i);
                        Swap(&ConAt.j);
#endif
                        BndMake(ConMtx[--ConAt.i], ConMtx[--ConAt.j], ConAt.Type);
                      } else Ret = PrintDosErr();
                    } /* End of for loop */
                    GLOBSW_CONNCALC = FALSE;
                  } else Ret = PrintDosErr();
                  break;

                case IFF_IIUB: /* IUPAC-IUB chunk for atom names                 */

                  if (fread(&Size, sizeof(VG_BYTE), 1, IFFIN) == 1) {
                    do {
                      if (fread(Atm -> Name.C, (int)Size, 1, IFFIN) != 1)
                        Ret = PrintDosErr();
                      Atm = Atm -> Ptr;
                    } while((Ret) && (Atm));
                  } else Ret = PrintDosErr();
                  break;

                case IFF_CALC: /* Atomic charge and force field chunks           */

                  k  = ChunkHdr.Size;
                  nb = ReadNull(IFFIN, FFName, 20);
                  while((k > nb) && (fread(&ChunkHdr, sizeof(IFFHDR), 1, IFFIN))) {
#ifdef LITTLE_ENDIAN
                    Swap(&ChunkHdr.ID);
                    Swap(&ChunkHdr.Size);
#endif

                    Atm  = InizAtm;
                    nb  += ChunkHdr.Size + 8;
                    switch(ChunkHdr.ID) {

                    case IFF_CHRG:  /* Charge chunk        */
                      do {
                        if (fread(&Atm -> Charge, sizeof(float), 1, IFFIN) != 1)
                          Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                        else Swap(&Atm -> Charge);
#endif
                        Atm = Atm -> Ptr;
                      } while((Ret) && (Atm));
                      GLOBSW_IFFCALC = TRUE;
                      break;

                    case IFF_ATYP:  /* Force field chunk   */
                      if (fread(&Size, 1, 1, IFFIN) == 1) {
                        do {
                          if (fread(&Atm -> Pot.C, (int)Size, 1, IFFIN) != 1)
                            Ret = PrintDosErr();
                          Atm = Atm -> Ptr;
                        } while((Ret) && (Atm));
                      } else Ret = PrintDosErr();
                      break;

                    default:       /* Don't read the chunk */
                      fseek(IFFIN, ChunkHdr.Size, SEEK_CUR);
                    }
                  } /* End of chunk loop */
                  GLOBSW_IFFCALC = TRUE;
                  break;

                  case IFF_FIXA:    /* Constraints chunk */
                    do {
                      if (fread(&Atm -> Fix, sizeof(float), 1, IFFIN) != 1)
                        Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                      else Swap(&Atm -> Fix);
#endif
                      Atm = Atm -> Ptr;
                    } while((Ret) && (Atm));
                    break;

                  case IFF_RESI:  /* Residue name, residue number and chain        */
                    k = ChunkHdr.Size / (3 * sizeof(VG_ULONG) + 1);
                    do {
                      if ((fread(&nb       , sizeof(VG_ULONG), 1, IFFIN) == 1) &&
                          (fread(&ResName.C, sizeof(VG_QCHAR), 1, IFFIN) == 1) &&
                          (fread(&ResNum   , sizeof(VG_ULONG), 1, IFFIN) == 1) &&
                          (fread(&ChainID  , sizeof(VG_BYTE ), 1, IFFIN) == 1)) {
#ifdef LITTLE_ENDIAN
                        Swap(&nb);
#endif
                        if (!ChainID) ChainID = ' ';

                        do {
                          Atm -> ResName.L = ResName.L;
                          Atm -> ResSeq.L  = ResNum;
                          Atm -> ChainID   = ChainID;
                          Atm = Atm -> Ptr;
                        } while ((Atm) && (--nb));
                      } else Ret = PrintDosErr();
                    } while((Ret) && (--k));
                    Tst2 = TRUE;
                    break;

                  case IFF_RSNU:  /* Chunk of residue number                       */
                    do {
                      if (fread(Atm -> ResSeq.C, sizeof(VG_QCHAR), 1, IFFIN) != 1)
                        Ret = PrintDosErr();
                      Atm = Atm -> Ptr;
                    } while((Ret) && (Atm));
                    Tst2 = TRUE;
                    break;

                case IFF_RSNA:  /* Residue name chunk                            */

                  if (fread(&j, sizeof(VG_UWORD), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                    SwapW(&j);
#endif
                    for(k = 0;(Ret) && (k < (VG_ULONG)j);++k) {
                      if ((fread(&ResNum, sizeof(VG_QCHAR), 1, IFFIN) == 1) &&
                          (fread(ResName.C, sizeof(VG_QCHAR), 1, IFFIN) == 1)) {
                        Size = 0;
                        for(Atm  = InizAtm; Atm; Atm = Atm -> Ptr) {
                          if ((Size) && (ResNum != (VG_ULONG)Atm -> ResSeq.L)) break;
                          if (ResNum == (VG_ULONG)Atm -> ResSeq.L) {
                            Atm -> ResName.L = ResName.L;
                            Size = 1;
                          }
                        } /* End of for loop */
                      } else Ret = PrintDosErr();
                    } /* End of for loop */
                  } else Ret = PrintDosErr();
                  break;

                case IFF_MOLN:  /* Name of molecule(s) chunk                     */

                  if (fread(&j, sizeof(VG_UWORD), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                    SwapW(&j);
#endif
                    for(k = 0;(Ret) && (k < (VG_ULONG)j);++k) {
                      if ((fread(&i , sizeof(VG_ULONG), 1, IFFIN) == 1) &&
                          (fread(&nb, sizeof(VG_ULONG), 1, IFFIN) == 1)) {
#ifdef LITTLE_ENDIAN
                        Swap(&i);
                        Swap(&nb);
#endif
                        while(getc(IFFIN));
                        ConMtx[i + nb - 2] -> Flags |= VG_ATMF_MOLEND;
                      } else Ret = PrintDosErr();
                    } /* End of for loop */
                  } else Ret = PrintDosErr();
                  break;

                case IFF_SRFA:  /* Surface atom number chunk */
                  if ((*InizSrf) && ((ChunkHdr.Size >> 2) == *TotSrf)) {
                    for(Dot = *InizSrf; (Ret) && (Dot); Dot = Dot -> Next) {
                      if (fread(&Dot -> AtmNum, sizeof(VG_ULONG), 1, IFFIN) != 1)
                        Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                      else Swap(&Dot -> AtmNum);
#endif
                    } /* End of for */
                  }

                  break;

                case IFF_SEGM:  /* Segments chunk */
                    k = ChunkHdr.Size / sizeof(VG_ULONG);
                    do {
                      if (fread(&nb, sizeof(VG_ULONG), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                        Swap(&nb);
#endif
                        ConMtx[nb - 1] -> Flags |= VG_ATMF_SEGEND;
                      } else Ret = PrintDosErr();
                    } while((Ret) && (--k));
                  break;

                case IFF_SRFC:  /* Surface color chunk */
                  if ((*InizSrf) && ((ChunkHdr.Size >> 2) == *TotSrf)) {
                    for(Dot = *InizSrf; (Ret) && (Dot); Dot = Dot -> Next) {
                      if (fread(Col, 4, 1, IFFIN) == 1) {
                        Dot -> Color[0] = Col[1];
                        Dot -> Color[1] = Col[2];
                        Dot -> Color[2] = Col[3];
                      } else Ret = PrintDosErr();
                    } /* End of for */
                  }
                  break;

                case IFF_SRFV: /* Surface value chunk */
                  if ((*InizSrf) && ((ChunkHdr.Size >> 2) == *TotSrf)) {
                    for(Dot = *InizSrf; (Ret) && (Dot); Dot = Dot -> Next) {
                      if (fread(&Dot -> Val, sizeof(float), 1, IFFIN) != 1)
                        Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                      else
                        Swap(&Dot -> Val);
#endif
                    } /* End of for */
                  }
                  break;

                case IFF_SURF: /* Surface chunk */
                  if (*InizSrf) SrfFree(*InizSrf);
                  *TotSrf  = 0;
                  for(k = 0;(Ret) && (k < ChunkHdr.Size);k += sizeof(XYZ)) {
                    if ((Dot = SrfAlloc(InizSrf, TotSrf))) {
                      Ret = IffRdXyz(IFFIN, (XYZ *)&Dot -> x);
                    } else {
                      SrfFree(*InizSrf);
                      *InizSrf   = NULL;
                      *TotSrf    = 0;
                      break;
                    }
                  } /* End of for */
#ifdef __VG_OPENGL
                  if (InizSrf) {
                    ViewPrefs.SrfAct = TRUE;
                    LastSrf          = Dot;
                  }
#endif
                  break;

                case IFF_VGAC: /* VEGA atom activity chunk */
                  do {
                    if (fread(&Atm -> Active, 1, 1, IFFIN) == 1) {
                      Atm = Atm -> Ptr;
                    } else Ret = PrintDosErr();
                  } while((Ret) && (Atm));
                  break;

                case IFF_VGCL: /* VEGA color chunk */
                  do {
                    if (fread(&Atm -> ColorID, 1, 1, IFFIN) == 1) {
                      Atm = Atm -> Ptr;
                    } else Ret = PrintDosErr();
                  } while((Ret) && (Atm));
                  break;

                case IFF_VGLB: /* VEGA label chunk */
                  do {
                    if (fread(&Atm -> Label, 1, 1, IFFIN) == 1) {
                      Atm = Atm -> Ptr;
                    } else Ret = PrintDosErr();
                  } while((Ret) && (Atm));
                  break;

#ifdef __VG_OPENGL
                case IFF_VGTR: /* VEGA transformation */
                  if (GLOBSW_OPENGL) {
                    if (IffRdXyz(IFFIN, &ViewPrefs.ViewCenter) &&
                        IffRdXyz(IFFIN, &ViewPrefs.StepRot) &&
                        IffRdXyz(IFFIN, &ViewPrefs.StepTrans)) {
                      if (fread(&ViewPrefs.Scale, sizeof(float), 1, IFFIN) == 1) {
#ifdef LITTLE_ENDIAN
                        Swap(&ViewPrefs.Scale);
#endif
                        if (fread(&RotMat, sizeof(RotMat), 1, IFFIN) != 1)
                          Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                        else GL_Mat4SwapEndian(RotMat);
#endif
                      } else Ret = PrintDosErr();
                    }
                  } else fseek(IFFIN, ChunkHdr.Size, SEEK_CUR);
                  break;
#endif

                default:       /* Generic chunk */

                  fseek(IFFIN, ChunkHdr.Size, SEEK_CUR);
                }
              }
              FREE(ConMtx);
            } else Ret = FALSE;
          }

          if ((Ret) && (!Tst2)) {
            for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
              *Atm -> ResSeq.C = '1';
          }
        } else Ret = CatErr(MSG_ERR_IFF_INVFILE);
      } else Ret = PrintDosErr();
    } else Ret = CatErr(MSG_ERR_IFF_INVFILE);
  } else Ret = PrintDosErr();

  if ((!Ret) && (InizAtm != (void *)-1)) {
    FreeAtm(InizAtm);
    InizAtm = (void *)-1;
  }

  return InizAtm;
}


/**** Interchange File Format (IFF) Saver ****/

VG_BOOL IFFSave(FILE *OUT, ATOMO *AtmIniz, VG_ULONG Tot)
{
  ATOMO             *Atm;
  char              Name[2];
  IFFHDR            Hdr;
  VG_BYTE           PrecChainID, ChainID;
  VG_DCHAR          Elem;
  VG_LONG           PrecResSeq, PrecResName;
  VG_ULONG          Count, Iniz;

#ifdef LITTLE_ENDIAN
  float             TmpF;
  VG_ULONG          Tmp, Tmp2;
  VG_UWORD          TmpW;
  XYZ               Coord;
#endif

  VG_BOOL           Ret       = TRUE;
  VG_UBYTE          SizeFF    = 0;
  VG_UBYTE          SizeIIUB  = 3;
  VG_ULONG          nb        = 0;
  VG_UWORD          NMol      = 0;
  VG_ULONG          TotSeg    = 0;
  VG_ULONG          NumRes    = 1;

  Name[1] = ' ';


  /**** Calculates nb, NMol, NumRes, SizeFF, SizeIIUB and TotSeg ****/

  PrecChainID = AtmIniz -> ChainID;
  PrecResName = AtmIniz -> ResName.L;
  PrecResSeq  = AtmIniz -> ResSeq.L;
  for(Atm = AtmIniz; Atm; Atm = Atm -> Ptr) {
    if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
    if (Atm -> Flags & VG_ATMF_SEGEND) ++TotSeg;
    if (Atm -> Name.C[3]) SizeIIUB = 4;
    if ((Atm -> ResSeq.L  != PrecResSeq) ||
        (Atm -> ResName.L != PrecResName) ||
        (Atm -> ChainID   != PrecChainID)) {
      PrecResName = Atm -> ResName.L;
      PrecResSeq  = Atm -> ResSeq.L;
      PrecChainID = Atm -> ChainID;
      ++NumRes;
    }

    for(Count = 0;(Atm -> Conn[Count]) && (Count < MAXBOND); ++Count)
      if (Atm -> Conn[Count] -> Num > Atm -> Num) ++nb;;
    for(Count = 0;(Count < 8) && (Atm -> Pot.C[Count]); ++Count);
    if (SizeFF < (VG_UBYTE)Count) SizeFF = (VG_UBYTE)Count;
  } /* End of atom loop */

  /**** Header chunk ****/

  if (!GLOBSW_IFFCALC) GLOBSW_IFFCALC = ChkCharges(AtmIniz, Tot, NULL);

  Hdr.ID   = IFF_FORM;
  Hdr.Size = IFF_HDRLEN * (11 + GLOBSW_IFFCALC) + 4 +
             IFF_ATOMO_SIZE + IFF_XYZ1_SIZE + IFF_MOLN_SIZE + IFF_COMM_SIZE +
/*           IFF_RSNU_SIZE  + IFF_RSNA_SIZE + */
             IFF_RESI_SIZE + IFF_VGAC_SIZE + IFF_VGCL_SIZE +
             IFF_VGLB_SIZE  +
             IFF_IIUB_SIZE +
             IFF_CONX_SIZE +
             IFF_CALC_SIZE * GLOBSW_IFFCALC +
             IFF_FIXA_SIZE * GLOBSW_FIXSAV;


#ifdef __VG_OPENGL
#define  TotSrf         TotalSrf
  if (BegSrf) {
    Hdr.Size += IFF_HDRLEN * 4 +
                IFF_SURF_SIZE + IFF_SRFA_SIZE + IFF_SRFC_SIZE + IFF_SRFV_SIZE;
  }
  if (GLOBSW_OPENGL) {
    Hdr.Size += IFF_HDRLEN + IFF_VGTR_SIZE;
  }
#undef  TotSrf
#endif

#ifdef LITTLE_ENDIAN
  Swap(&Hdr.ID);
  Swap(&Hdr.Size);
#endif

  if ((fwrite(&Hdr, IFF_FORM_SIZE, 1, OUT) == 1) &&
      (fwrite("MOLE", 4, 1, OUT) == 1)) {

    /**** Atom names chunk ****/

    if ((Ret = IffWrtHdr(OUT, IFF_ATOMO, IFF_ATOMO_SIZE))) {
#ifdef LITTLE_ENDIAN
      Tmp = SwapI(Tot);
      if (fwrite(&Tmp, sizeof(VG_ULONG), 1, OUT) == 1) {
#else
      if (fwrite(&Tot, sizeof(VG_ULONG), 1, OUT) == 1) {
#endif
        for(Atm = AtmIniz;(Ret) && (Atm); Atm = Atm -> Ptr) {
          Elem.S = Atm -> Elem.S;
          if (!Elem.C[1]) Elem.C[1] = ' ';
          if (fwrite(Elem.C, 2, 1, OUT) != 1)
            Ret = PrintDosErr();
        } /* End of atom loop */

        /**** Cartesian coordinates chunk ****/

        if (Ret) {
          if ((Ret = IffWrtHdr(OUT, IFF_XYZ1, IFF_XYZ1_SIZE))) {
            for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {

#ifdef LITTLE_ENDIAN
              Coord.x = Atm -> x;
              Coord.y = Atm -> y;
              Coord.z = Atm -> z;
              Swap(&Coord.x);
              Swap(&Coord.y);
              Swap(&Coord.z);
              if (fwrite(&Coord, sizeof(XYZ), 1, OUT) != 1)
#else
              if (fwrite(&Atm -> x, sizeof(XYZ), 1, OUT) != 1)
#endif
                Ret = PrintDosErr();
            }

            /**** Connectivity chunk ****/

            if (Ret) {
              if ((Ret = IffWrtHdr(OUT, IFF_CONX, IFF_CONX_SIZE))) {
#ifdef LITTLE_ENDIAN
                Tmp = SwapI(nb);
                if (fwrite(&Tmp, sizeof(VG_ULONG), 1, OUT) == 1) {
#else
                if (fwrite(&nb, sizeof(VG_ULONG), 1, OUT) == 1) {
#endif
                  for(Atm = AtmIniz;(Ret) && (Atm); Atm = Atm -> Ptr) {
                    for(Count = 0;(Ret) && (Atm -> Conn[Count]) && (Count < MAXBOND); ++Count) {
                      if (Atm -> Conn[Count] -> Num > Atm -> Num) {
#ifdef LITTLE_ENDIAN
                        Tmp  = SwapI(Atm -> Num);
                        Tmp2 = SwapI(Atm -> Conn[Count] -> Num);
                        if ((fwrite(&Tmp , sizeof(VG_ULONG), 1, OUT) != 1) ||
                            (fwrite(&Tmp2, sizeof(VG_ULONG), 1, OUT) != 1) ||
                            (fwrite(&Atm -> Order[Count], 1, 1, OUT) != 1))
#else
                        if ((fwrite(&Atm -> Num, sizeof(VG_ULONG), 1, OUT) != 1) ||
                            (fwrite(&Atm -> Conn[Count] -> Num, sizeof(VG_ULONG), 1, OUT) != 1) ||
                            (fwrite(&Atm -> Order[Count], 1, 1, OUT) != 1))
#endif
                          Ret = PrintDosErr();
                      }
                    } /* End of for */
                  } /* End of atom loop */
                } else Ret = PrintDosErr();
              }

              /**** IUPAC-IUB chunk ****/

              if (Ret) {
                if ((Ret = IffWrtHdr(OUT, IFF_IIUB, IFF_IIUB_SIZE))) {
                  if (fwrite(&SizeIIUB, 1, 1, OUT) == 1) {
                    for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                      if (fwrite(Atm -> Name.C, SizeIIUB, 1, OUT) != 1)
                        Ret = PrintDosErr();
                  } else Ret = PrintDosErr();
                }

                /**** Potential & charges chunk ****/

                if (Ret) {
                  if (GLOBSW_IFFCALC) {
                    if ((Ret = IffWrtHdr(OUT, IFF_CALC, IFF_CALC_SIZE))) {
                      if (fwrite(FFName, strlen(FFName) + 1, 1, OUT) == 1) {
                        if ((Ret = IffWrtHdr(OUT, IFF_CHRG, IFF_CHRG_SIZE))) {
                          for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
#ifdef LITTLE_ENDIAN
                            TmpF = Atm -> Charge;
                            Swap(&TmpF);
                            if (fwrite(&TmpF, sizeof(float), 1, OUT) != 1)
#else
                            if (fwrite(&Atm -> Charge, sizeof(float), 1, OUT) != 1)
#endif
                              Ret = PrintDosErr();
                          } /* End of atom loop */
                          if ((Ret) && (Ret = IffWrtHdr(OUT, IFF_ATYP, IFF_ATYP_SIZE))) {
                            if (fwrite(&SizeFF, 1, 1, OUT) == 1) {
                              for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                                if (fwrite(Atm -> Pot.C, SizeFF, 1, OUT) != 1)
                                  Ret = PrintDosErr();
                            } else Ret = PrintDosErr();
                          }
                        }
                      } else Ret = PrintDosErr();
                    }
                  }

                  /**** Chains ****/

                  if (Ret) {

                    /**** Residues & chains chunk ****/

                    if (Ret) {
                      if ((Ret = IffWrtHdr(OUT, IFF_RESI, IFF_RESI_SIZE))) {
                        PrecChainID = AtmIniz -> ChainID;
                        PrecResName = AtmIniz -> ResName.L;
                        PrecResSeq  = AtmIniz -> ResSeq.L;
                        Count       = 0;
                        for(Atm = AtmIniz;; Atm = Atm -> Ptr) {
                          if ((!Atm) ||
                              (Atm -> ResSeq.L  != PrecResSeq ) ||
                              (Atm -> ResName.L != PrecResName) ||
                              (Atm -> ChainID   != PrecChainID)) {

                            if (PrecChainID != ' ') ChainID = PrecChainID;
                            else ChainID = 0;

#ifdef LITTLE_ENDIAN
                            Swap(&Count);
#endif

                            if ((fwrite(&Count      , sizeof(VG_ULONG), 1, OUT) != 1) ||
                                (fwrite(&PrecResName, sizeof(VG_ULONG), 1, OUT) != 1) ||
                                (fwrite(&PrecResSeq , sizeof(VG_ULONG), 1, OUT) != 1) ||
                                (fwrite(&ChainID    , sizeof(VG_BYTE ), 1, OUT) != 1)) {
                              Ret = PrintDosErr();
                              break;
                            }

                            if (!Atm) break;

                            PrecResName = Atm -> ResName.L;
                            PrecResSeq  = Atm -> ResSeq.L;
                            PrecChainID = Atm -> ChainID;
                            Count       = 1;
                          } else ++Count;
                        }
                      }
/*
                      if ((Ret = IffWrtHdr(OUT, IFF_RSNU, IFF_RSNU_SIZE))) {
                        for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                          if (fwrite(Atm -> ResSeq.C, sizeof(VG_QCHAR), 1, OUT) != 1)
                            Ret = PrintDosErr();
                      }
*/
                      /**** Name of residues chunk ****/

                      if (Ret) {
/*
                        if ((Ret = IffWrtHdr(OUT, IFF_RSNA, IFF_RSNA_SIZE))) {
#ifdef LITTLE_ENDIAN
                          TmpW = NumRes;
                          SwapW(&TmpW);
                          if (fwrite(&TmpW, sizeof(VG_UWORD), 1, OUT) == 1) {
#else
                          if (fwrite(&NumRes, sizeof(VG_UWORD), 1, OUT) == 1) {
#endif
                            Atm  = AtmIniz;
                            Iniz = Atm -> ResSeq.L;
                            if ((fwrite(&Atm -> ResSeq.C, sizeof(VG_QCHAR), 1, OUT) == 1) &&
                                (fwrite(&Atm -> ResName.C, sizeof(VG_QCHAR), 1, OUT) == 1)) {
                              do {
                                if ((VG_ULONG)Atm -> ResSeq.L != Iniz) {
                                  if ((fwrite(&Atm -> ResSeq.C, sizeof(VG_QCHAR), 1, OUT) == 1) &&
                                      (fwrite(&Atm -> ResName.C, sizeof(VG_QCHAR), 1, OUT) == 1)) {
                                    Iniz = Atm -> ResSeq.L;
                                    ++NumRes;
                                  } else Ret = PrintDosErr();
                                }
                                Atm = Atm -> Ptr;
                              } while((Ret) && (Atm));
                            } else Ret = PrintDosErr();
                          } else Ret = PrintDosErr();
                        }
*/

                        /**** Name of molecules chunk ****/

                        if (Ret) {
                          Count    = 1;
                          Iniz     = 1;
                          if ((Ret = IffWrtHdr(OUT, IFF_MOLN, IFF_MOLN_SIZE))) {
#ifdef LITTLE_ENDIAN
                            SwapW(&NMol);
#endif
                            if (fwrite(&NMol, sizeof(VG_UWORD), 1, OUT) == 1) {
                              NMol = 1;
                              for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
                                if (Atm -> Flags & VG_ATMF_MOLEND) {
#ifdef LITTLE_ENDIAN
                                  Tmp  = SwapI(Iniz);
                                  Tmp2 = SwapI(Count);
                                  if ((fwrite(&Tmp , sizeof(VG_ULONG), 1, OUT) == 1) &&
                                      (fwrite(&Tmp2, sizeof(VG_ULONG), 1, OUT) == 1) &&
                                      (fprintf(OUT, "Molecule%4d%c", NMol, 0) >= 0)) {
#else
                                  if ((fwrite(&Iniz, sizeof(VG_ULONG), 1, OUT) == 1) &&
                                      (fwrite(&Count, sizeof(VG_ULONG), 1, OUT) == 1) &&
                                      (fprintf(OUT, "Molecule%4d%c", NMol, 0) >= 0)) {
#endif
                                    Iniz  += Count;
                                    Count  = 1;
                                    ++NMol;
                                  } else Ret = PrintDosErr();
                                } else ++Count;
                              }
                            } else Ret = PrintDosErr();
                          }

                          /**** Segments chunk ****/

                          if ((Ret) && (TotSeg)) {
                            if ((Ret = IffWrtHdr(OUT, IFF_SEGM, IFF_SEGM_SIZE))) {
                              for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
                                if (Atm -> Flags & VG_ATMF_SEGEND) {
#ifdef LITTLE_ENDIAN
                                  Tmp = Atm -> Num;
                                  Swap(&Tmp);
                                  if (fwrite(&Tmp, sizeof(VG_ULONG), 1, OUT) != 1)
#else
                                  if (fwrite(&Atm -> Num, sizeof(VG_ULONG), 1, OUT) != 1)
#endif
                                    Ret = PrintDosErr();
                                }
                              } /* End of for */
                            }
                          }

                          /**** Constraints chunk ****/

                          if ((Ret) && (GLOBSW_FIXSAV)) {
                            if ((Ret = IffWrtHdr(OUT, IFF_FIXA, IFF_FIXA_SIZE))) {
                              for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
#ifdef LITTLE_ENDIAN
                                TmpF = Atm -> Fix;
                                Swap(&TmpF);
                                if (fwrite(&TmpF, sizeof(float), 1, OUT) != 1)
#else
                                if (fwrite(&Atm -> Fix, sizeof(float), 1, OUT) != 1)
#endif
                                  Ret = PrintDosErr();
                              } /* End of for */
                            }
                          }

                          /***** Surface chunks ****/

                          if ((Ret) && (BegSrf))
                            Ret = IffWrtSrf(OUT, BegSrf, TotalSrf);

                          /**** Comment chunk ****/

                          if ((Ret) && (Ret = IffWrtHdr(OUT, IFF_COMM, IFF_COMM_SIZE))) {
                            if (fwrite(VEGAHdr, IFF_COMM_SIZE, 1, OUT) != 1)
                              Ret = PrintDosErr();
                          }

                          /**** VEGA atom color chunk ****/

                          if ((Ret) && (Ret = IffWrtHdr(OUT, IFF_VGCL, IFF_VGCL_SIZE))) {
                            for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                              if (fwrite(&Atm -> ColorID, 1, 1, OUT) != 1)
                                Ret = PrintDosErr();
                          }

                          /**** VEGA atom label chunk ****/

                          if ((Ret) && (Ret = IffWrtHdr(OUT, IFF_VGLB, IFF_VGCL_SIZE))) {
                            for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                              if (fwrite(&Atm -> Label, 1, 1, OUT) != 1)
                                Ret = PrintDosErr();
                          }

                          /**** VEGA atom activity chunk ****/

                          if ((Ret) && (Ret = IffWrtHdr(OUT, IFF_VGAC, IFF_VGAC_SIZE))) {
                            for(Atm  = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr)
                              if (fwrite(&Atm -> Active, 1, 1, OUT) != 1)
                                Ret = PrintDosErr();
                          }

                          /**** VEGA transformations ****/

#ifdef __VG_OPENGL
                          if ((GLOBSW_OPENGL) && (Ret) &&
                              (Ret = IffWrtHdr(OUT, IFF_VGTR, IFF_VGTR_SIZE)) &&
                              (Ret = IffWrtXyz(OUT, &ViewPrefs.ViewCenter)) &&
                              (Ret = IffWrtXyz(OUT, &ViewPrefs.StepRot)) &&
                              (Ret = IffWrtXyz(OUT, &ViewPrefs.StepTrans))) {
                            TmpF = ViewPrefs.Scale;
#ifdef LITTLE_ENDIAN
                            Swap(&TmpF);
#endif
                            if (fwrite(&TmpF, sizeof(float), 1, OUT) == 1) {
#ifdef LITTLE_ENDIAN
                              GL_Mat4SwapEndian(RotMat);
#endif
                              if (fwrite(&RotMat, sizeof(RotMat), 1, OUT) != 1)
                                Ret = PrintDosErr();
#ifdef LITTLE_ENDIAN
                              GL_Mat4SwapEndian(RotMat);
#endif
                            } else Ret = PrintDosErr();
                          }
#endif
                        }
                      }
                    }
                  }
                }
              }
            }
          }
        }
      } else Ret = PrintDosErr();
    }
  } else Ret = PrintDosErr();

  return Ret;
}


/**** Open an IFF file for read ****/
/**** and eventually check it   ****/

FILE *OpenIff(char *Name, VG_ULONG Comp)
{
  FILE           *FH = NULL;
  IFFHDR         Hdr;

  if ((FH = fopen(Name, "rb"))) {
    if (ReadIffHdr(FH, &Hdr, IFF_FORM)) {
      if (fread(&Hdr, sizeof(VG_ULONG), 1, FH) == 1) {

#ifdef LITTLE_ENDIAN
        Swap(&Hdr.ID);
#endif

        if (Comp != Hdr.ID) {
          CatErr(MSG_ERR_IFF_NOTIFF, (char *)&Hdr.ID);
          fclose(FH);
          FH = NULL;
        }
      } else {
        PrintDosErr();
        fclose(FH);
        FH = NULL;
      }
    }
  }

  return FH;
}


/**** Read an IFF vector ****/

static VG_BOOL IffRdXyz(FILE *FH, XYZ *Ptr)
{
  if (fread(Ptr, sizeof(XYZ), 1, FH) != 1)
    return PrintDosErr();

#ifdef LITTLE_ENDIAN
  Swap(&Ptr -> x);
  Swap(&Ptr -> y);
  Swap(&Ptr -> z);
#endif

  return TRUE;
}


/**** Write the IFF Header ****/

static VG_BOOL IffWrtHdr(FILE *FH, VG_LONG ID, VG_ULONG Size)
{
  VG_BOOL            Ret = TRUE;

#ifdef LITTLE_ENDIAN
  Swap(&ID);
  Swap(&Size);
#endif

  if ((fwrite(&ID  , sizeof(VG_LONG) , 1, FH) != 1) ||
      (fwrite(&Size, sizeof(VG_ULONG), 1, FH) != 1))
    Ret = PrintDosErr();

  return Ret;
}


/**** Write the surface ****/

static VG_BOOL IffWrtSrf(FILE *FH, VG_SURFACE *InizSrf, VG_ULONG TotSrf)
{
#ifdef LITTLE_ENDIAN
  float         TmpF;
  VG_ULONG      TmpI;
#endif

  VG_BOOL       Ret;
  VG_SURFACE    *Srf;
  VG_UBYTE      Col[4];

  /**** SURF chunk ****/

  if ((Ret = IffWrtHdr(FH, IFF_SURF, IFF_SURF_SIZE))) {
    for(Srf = InizSrf;
        (Srf) && (Ret = IffWrtXyz(FH, (XYZ *)&Srf -> x));
        Srf = Srf -> Next);

    /**** SRFV chunk ****/

    if ((Ret) && ((Ret = IffWrtHdr(FH, IFF_SRFV, IFF_SRFV_SIZE)))) {
      for(Srf = InizSrf; Srf; Srf = Srf -> Next) {
#ifdef LITTLE_ENDIAN
        TmpF = Srf -> Val;
        Swap(&TmpF);
        if (fwrite(&TmpF, sizeof(float), 1, FH) != 1) {
#else
        if (fwrite(&Srf -> Val, sizeof(float), 1, FH) != 1) {
#endif
          PrintDosErr();
          Ret = FALSE;
          break;
        }
      } /* End of for */

      /**** SRFA chunk ****/

      if ((Ret) && ((Ret = IffWrtHdr(FH, IFF_SRFA, IFF_SRFA_SIZE)))) {
        for(Srf = InizSrf; Srf; Srf = Srf -> Next) {
#ifdef LITTLE_ENDIAN
          TmpI = SwapI(Srf -> AtmNum);
          if (fwrite(&TmpI, sizeof(VG_ULONG), 1, FH) != 1) {
#else
          if (fwrite(&Srf -> AtmNum, sizeof(VG_ULONG), 1, FH) != 1) {
#endif
            PrintDosErr();
            Ret = FALSE;
            break;
          }
        } /* End of for */

        /**** SRFC chunk ****/

        if ((Ret) && ((Ret = IffWrtHdr(FH, IFF_SRFC, IFF_SRFC_SIZE)))) {
          Col[0] = 0;
          for(Srf = InizSrf; Srf; Srf = Srf -> Next) {
            Col[1] = Srf -> Color[0];
            Col[2] = Srf -> Color[1];
            Col[3] = Srf -> Color[2];
            if (fwrite(Col, 4, 1, FH) != 1) {
              PrintDosErr();
              Ret = FALSE;
              break;
            }
          } /* End of for */
        }
      }
    }
  }

  return Ret;
}


/**** Write an IFF vector ****/

static VG_BOOL IffWrtXyz(FILE *FH, XYZ *Ptr)
{
  XYZ           Temp = *Ptr;


#ifdef LITTLE_ENDIAN
  Swap(&Temp.x);
  Swap(&Temp.y);
  Swap(&Temp.z);
#endif
  if (fwrite(&Temp, sizeof(XYZ), 1, FH) != 1)
    return PrintDosErr();

  return TRUE;
}


/**** Read an IFF header and eventually check it with ****/
/**** the Comp value                                  ****/

VG_BOOL ReadIffHdr(FILE *FH, IFFHDR *Hdr, VG_ULONG Comp)
{
  VG_BOOL          Ret;

  /**** Strip the padding byte ****/

  if (!feof(FH)) {
    if (ftell(FH) & 1) fgetc(FH);
    if (fread(Hdr, sizeof(IFFHDR), 1, FH) == 1) {

#ifdef LITTLE_ENDIAN
      Swap(&Hdr -> ID);
      Swap(&Hdr -> Size);
#endif

      Ret = TRUE;
      if ((Comp != IFF_READ_NOCHK) && (Comp != IFF_READ_NODOSERR) &&
          (Comp != Hdr -> ID))
        Ret = CatErr(MSG_ERR_IFF_CORRUPTED);
    } else {
      if (Comp != IFF_READ_NODOSERR) PrintDosErr();
      Ret = FALSE;
    }
  } else Ret = FALSE;

  return Ret;
}


/**** Read to NULL character ****/

VG_UWORD ReadNull(FILE *IN, register char *Str, VG_UWORD Lim)
{
  register VG_UWORD  Count;

  for(Count = 1;((*Str = getc(IN)) != 0) && (--Lim); ++Count) ++Str;
  if (!Lim) while(getc(IN)) ++Count;

  return Count;
}


