
/*************************************************
****            VEGA - LogP Module            ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * Calculates the logP with atomic template methods
 * Broto/Moreau or Ghose/Crippen) and its lipole.
 */


#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "count.h"
#include "formats.h"
#include "logp.h"


/**** LogP initialization ****/

float *LogPInit(ATOMO *InizAtm, VG_ULONG TotAtm, float *Contrib, VG_ULONG Mode)
{
  ATOMO                 *Atm;
  char                  Field[16], FieldName[VG_LOGP_FIELDLEN], Tmp[32];
  FILE                  *IN;
  float                 Cont, *Ptr, *NewContrib;
  VG_OCHAR              Type;
  VG_ULONG              k;

  static const char     *ModeStr[] = {"BROTO", "CRIPPEN" };
  VG_BOOL               Ret        = TRUE;
  VG_ULONG              Lin        = 0;
  VG_ULONG              Tot        = 0;


  /**** Assign the force field ****/

  GetPrgPath(FieldName, TRUE);
  strcat(FieldName, ModeStr[Mode]);
  strcat(FieldName, ".tem");

  if (AssignFF(FieldName, InizAtm, TotAtm, FALSE, TRUE)) {

    /**** Read the parameter file ****/

    GetPrgPath(FieldName, TRUE);
    strcat(FieldName,  ModeStr[Mode]);
    strcat(FieldName, ".par");
    if (!Contrib) {
      if ((NewContrib = (float *)Alloca(TotAtm * sizeof(float))) != NULL)
        Contrib = NewContrib;
      else return FALSE;
    }


    if ((IN = fopen(FieldName, "r")) != NULL) {
      strcpy(Tmp, "#FF_Param ");
      strcat(Tmp, ModeStr[Mode]);
      while(fgets(FieldName, VG_LOGP_FIELDLEN, IN)) {
        if ((!Lin) && (strncmp(FieldName, Tmp, strlen(Tmp)))) {
          Ret = CatErr(MSG_ERR_LOGP_UNRECFILE, ModeStr[Mode]);
          break;
        }
        if ((*FieldName) && (*FieldName != '\n') &&
            (*FieldName != ';') && (*FieldName != '#')) {
          *Field = 0;
          Cont   = 0.0;
          sscanf(FieldName, "%12s %f", Field, &Cont);
          if (*Field) {
            if ((k = strlen(Field)) <= 8) {
              Type.DL = 0;
              memcpy(Type.C, Field, k);

              /**** Assign the the parameter to each atom ****/

              for(Atm = InizAtm, Ptr = Contrib; Atm; Atm = Atm -> Ptr, ++Ptr) {
                if (Atm -> Pot.DL == Type.DL) {
                  *Ptr = Cont;
                  ++Tot;
                }
              } /* End of for */
            }
          }
        }
        ++Lin;
      } /* End of while */
      fclose(IN);
    } else Ret = CatErr(MSG_ERR_LOGP_NOTFOUND, ModeStr[Mode]);
  }

  if (Tot != CountAtms(InizAtm, FALSE)) Ret = CatErr(MSG_ERR_LOGP_NOTALLPAR, ModeStr[Mode]);

  if ((!Ret) && (NewContrib)) {
    FREE(NewContrib);
    Contrib = NULL;
  }

  return Contrib;
}


/**** LogP calculation ****/

float LogPCalc(ATOMO *Atm, float *Lipole, float *Contrib, VG_ULONG Flags)
{
  float                 ContN = 0.0;
  float                 ContP = 0.0;
  float                 LogP  = 0.0;
  float                 Xn    = 0.0;
  float                 Yn    = 0.0;
  float                 Zn    = 0.0;
  float                 Xp    = 0.0;
  float                 Yp    = 0.0;
  float                 Zp    = 0.0;

  for(; Atm; Atm = Atm -> Ptr, ++Contrib) {

    /**** Skip waters (for trajectory analysis) ****/

    if (((Flags & SRF_FLG_NOWAT) &&
#ifdef LITTLE_ENDIAN
       (Atm -> ResName.L == 0x00484F48)) ||
#else
       (Atm -> ResName.L == 0x484F4800)) ||
#endif
       ((Flags & SRF_FLG_SELONLY) && (!Atm -> Active)) ||
       (Atm -> Flags & VG_ATMF_CENTROID)) {
      continue;
    }

    if (Lipole) {
      if (*Contrib > 0.0) {
        Xp    += Atm -> x * *Contrib;
        Yp    += Atm -> y * *Contrib;
        Zp    += Atm -> z * *Contrib;
        ContP += *Contrib;
      } else {
        Xn += Atm -> x * *Contrib;
        Yn += Atm -> y * *Contrib;
        Zn += Atm -> z * *Contrib;
        ContN += *Contrib;
      }
    }
    LogP += *Contrib;
  } /* End of for */

  if (Lipole) {
    if (ContP) {
      Xp /= ContP;
      Yp /= ContP;
      Zp /= ContP;
    }
    if (ContN) {
      Xn /= ContN;
      Yn /= ContN;
      Zn /= ContN;
    }
    *Lipole = SQR(Quad(Xp - Xn) + Quad(Yp - Yn) + Quad(Zp - Zn));
  }

  return LogP;
}
