
/*************************************************
****          VEGA V1.5.1 - Main code         ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <signal.h>
#include <malloc.h>
#include <stdarg.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#define  __MAIN
#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#include "addhyd.h"
#include "connect.h"
#include "comerr.h"
#include "formats.h"
#include "traj.h"
#include "../LocaleLib/locale.h"
#include "surface.h"

#include "main.h"


/**** Local variables ****/

static char             **LastArg;

/**** Local prototypes ****/

static void     Abort(void);
static void     Sintassi(void);

#ifdef WIN32
int     ConPrintf(FILE *FH, char *, va_list);
#endif


/**** Main code ****/

int main(int argc, char *argv[])
{
  ATOMO         *LastSeg, *TmpIniz;
  char          Buf[256], FField[256], FCharge[256], Shell[256], Opt[100];
  char          **Arg, **CurFile, **Files, *Ptr;
  FILE          *TEST;
  FILEFORMAT    *OutFormat;
  TRJPAR        TrjPar;
  VG_LONG       FormatID;
  VG_SURFACE    *LastSrfSeg;
  VG_ULONG      j, k, TmpTotAtm, TotSrf;


  VG_BOOL       Ret          = TRUE;
  VG_BYTE       Switch[]     = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
  VG_LONG       AddhFlags    = VG_ADDH_FLG_NONE;
  VG_SURFACE    *InizSrf     = NULL;
  VG_ULONG      TotFiles     = 0;
  VG_ULONG      PackMode     = 0;
  VG_UWORD      CharFormat   = 0;

  const char    *CharType[]  = { "FRAGMENTS", "GASTEIGER", "" };
  const char    *AddhModes[] = { "GEN", "PROT", "NA", NULL };

  /**** Trap for CTRL+C ****/

  signal(SIGINT, (fptr)Abort);

  /**** Check the VEGADIR environment variable ****/

  if (!GetPrgPath(FField, TRUE)) exit(0);

  strcpy(FCharge, FField);
  strcpy(Shell,   FField);
  strcat(Shell,   CLUSTER_DIR);
  strcat(FCharge, PREFS_FILE);

  /**** Load the preference (prefs) file ****/

  LoadPrefs(FCharge);
  strcpy(FCharge, FField);
  OutFormat      = Prefs.OUTFORMAT;
  Region.Par     = NULL;
  GLOBSW_CONNSAV = TRUE;           /* Save  connectivity         */
  GLOBSW_IFFCALC = FALSE;          /* Don't save  IFF CALC chunk */
  *OutFile       = 0;

  strcpy(FFName, "UNKNOWN");

  /**** Initialize the localization system ****/

  CatInit();

#ifndef WIN32
  if ((argc == 1) || (*argv[1] == '?'))
     Sintassi();
#else
  if (argc == 1) {
    GLOBSW_CONNCALC = TRUE;
    GLOBSW_OPENGL   = TRUE;
    GL_Start(NULL, 0);
    CloseAll();
    return 0;
  }
  if (*argv[1] == '?') Sintassi();
#endif

  if ((Files = (char **)Alloca(argc * sizeof(char *)))) {
    Arg            = &argv[0];
    CurFile        = Files;
    LastArg        = &argv[argc];

    do {
      ++Arg;
      if (**Arg == '-') {
        if (strlen(*Arg) > 2) {
          CatErr(MSG_ERR_MAIN_UNKOPT);
          Sintassi();
        }
        switch(tolower(*(*Arg + 1))) {

        case 'a':     /* Renumber residues          */
          Switch[SW_RENRES] = TRUE;
          if (SubArg(Arg + 1, SUBA_POSINT, FALSE)) {
            sscanf(*++Arg, "%d", &Prefs.RENSTART);
            if (Prefs.RENSTART < 1) {
              CatErr(MSG_ERR_MAIN_ILLSTRRES);
              Ret = FALSE;
            }
          }
          break;

        case 'b':     /* Don't save connectivity    */
          GLOBSW_CONNSAV       = FALSE;
          Switch[SW_NOCONNECT] = TRUE;
          break;

        case 'c':     /* Assign template of charges */
          if ((Ret = SubArg(++Arg, SUBA_TOUP, TRUE))) {
            strcat(FCharge, *Arg);
            strcat(FCharge, ".tem");
            Ret = FALSE;
            if ((TEST = fopen(FCharge, "r"))) {
              fgets(Buf, 255, TEST);
              if (!strncasecmp(Buf, "#TemplateCharge", 15)) {
                sscanf(Buf, "%*s %s", Opt);
                for(CharFormat = 0; *CharType[CharFormat]; ++CharFormat)
                  if (!strcasecmp(CharType[CharFormat], Opt)) {
                    Ret                = TRUE;
                    Switch[SW_CHARGES] = TRUE;
                    if (CharFormat != CHARGE_FRAG) Switch[SW_FINDRINGS] = TRUE;
                    break;
                  }
              }
              if (!Ret) CatErr(MSG_ERR_MAIN_UNSUPCHGTEMP);
              fclose(TEST);
            } else Ret = CatErr(MSG_ERR_MAIN_CHGFILENOTFOUND);
          }
          break;

        case 'd':     /* Dielectric constant        */
          if ((Ret = SubArg(++Arg, SUBA_POSFLOAT, TRUE))) {
            sscanf(*Arg, "%f", &Prefs.ENDIEL);
            if (Prefs.ENDIEL <= 0) Ret = CatErr(MSG_ERR_MAIN_ILLDIEL);
          }
          break;

        case 'e':     /* Residue name for energy    */
          if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE)) &&
              (!GetPrefsRes(*Arg))) Ret = CatErr(MSG_ERR_MAIN_ILLRESNAME);
          break;

        case 'f':     /* Output format              */
          if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE))) {
            strcpy(Buf, *Arg);
            if ((Ptr = strchr(Buf, '.'))) *Ptr++ = 0;
            for(OutFormat = FileInfo; (OutFormat -> Com) && (strcasecmp(Buf, OutFormat -> Com)); ++OutFormat);
            if (OutFormat -> Flags & FORM_FLAG_INONLY) OutFormat -> Id = FORM_UNK;

            switch (OutFormat -> Id) {
            case 0:            /* Unknown format   */
               Ret = CatErr(MSG_ERR_MAIN_UNKOUTFOR);
               break;

            case FORM_OPENGL:  /* OpenGL           */

#ifdef WIN32
               GLOBSW_OPENGL = TRUE;
#else
               Ret = CatErr(MSG_MAIN_ERR_OPENGL);
#endif
               break;

            default:           /* Compression mode */
              if (Ptr) {
                for(PackMode = 1; (PackExt[PackMode]) &&
                    (strcasecmp(Ptr, PackExt[PackMode]));++PackMode);
                if (!PackExt[PackMode])
                  Ret = CatErr(MSG_ERR_MAIN_UNKPACKMET);
                else if (OutFormat -> Id == FORM_QUANTA_MSF)
                  Ret = CatErr(MSG_ERR_MAIN_CANTPACK);
              }
            } /* End of switch */
          }
          break;

        case 'g':     /* Probe radius for SAS       */
          if ((Ret = SubArg(++Arg, SUBA_POSFLOAT, TRUE))) {
            sscanf(*Arg, "%f", &Prefs.SAS_PROBERAD);
            if (Prefs.SAS_PROBERAD < 0) Ret = CatErr(MSG_ERR_MAIN_ILLPROBERAD);
          }
          break;

        case 'i':     /* Solvate the molecule       */
          Switch[SW_SOLVATE] = TRUE;
          if ((Ret = SubArg(++Arg, SUBA_TOUP, TRUE))) {
            strcat(Shell, *Arg);
            if ((TEST = fopen(Shell, "r"))) {
              fclose(TEST);
              if ((SubArg(++Arg, SUBA_POSFLOAT, FALSE))) {
                sscanf(*Arg, "%f", &Prefs.SOL_RAD);
                if (Prefs.SOL_RAD == 0)
                  Ret = CatErr(MSG_ERR_MAIN_ILLSOLRAD);
                else if ((SubArg(++Arg, SUBA_TOUP, FALSE))) {
                  Ret = FALSE;
                  for(k = 0; ShapeType[k]; ++k)
                    if (!strcmp(ShapeType[k], *Arg)) {
                      Prefs.SOL_SHAPE = k;
                      Ret = TRUE;
                    }
                  if (!Ret) CatErr(MSG_ERR_MAIN_UNKSHAPE);
                } else --Arg;
              } else --Arg;
            } else Ret = CatErr(MSG_ERR_MAIN_SHELLFILE);
          }
          break;

        case 'k':     /* Keywords for Mopac         */
          if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE))) {
            strncpy(Prefs.MOPAC_DEF, *Arg, 80);
            Prefs.MOPAC_FLAG = TRUE;
          }
          break;

        case 'l':                       /* Add hydrogens              */
          if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE))) {
            for(k = 0; (AddhModes[k]) && (strcasecmp(*Arg, AddhModes[k])); ++k);
            if (AddhModes[k]) {
              Switch[SW_FINDRINGS] = TRUE;
              AddhFlags = VG_ADDH_FLG_HPOSENDRES|VG_ADDH_FLG_IUPACNAMES;
              switch(k) {
              case 1:
                AddhFlags |= VG_ADDH_FLG_PROT;
                break;

              case 2:
                AddhFlags |= VG_ADDH_FLG_NA;
                break;
              } /* End of switch */
            } else Ret = CatErr(MSG_ERR_MAIN_MOLTYPE);
          }
          break;

        case 'm':                       /* Analysis options           */
         if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE))) {
           TrjPar.Flags = SRF_FLG_ALLATM;
           Region.Par   = &TrjPar;
           TrjPar.Arg   = Arg;
           TrjPar.Num   = 1;
     if (*Arg != (char *)argv[argc - 1]) {
             do {
         ++Arg;
               if (**Arg == '-') {
                 --Arg;
                 break;
               } else ++TrjPar.Num;
             } while(*Arg != (char *)argv[argc - 1]);
     }
           Ret = TrjParser(&TrjPar);
           GLOBSW_MESTRJ = TRUE;
         }
         break;

        case 'n':     /* Normalize coordinates      */
          Switch[SW_NORMALIZE] = TRUE;
          break;

        case 'o':     /* Output file name           */
          if ((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE)))
            strcpy(OutFile, *Arg);
          break;

        case 'p':     /* Force field name           */
          if (((Ret = SubArg(++Arg, SUBA_NOCHK, TRUE))) &&
              ((Ret = ChkTempFF(*Arg, FField)))) {
                Switch[SW_FFIELD]    = TRUE;
                Switch[SW_FINDRINGS] = TRUE;
                GLOBSW_IFFCALC       = TRUE;
          }
          break;

        case 'r':     /* Remove hydrogens           */
          Switch[SW_RMHYDROG] = TRUE;
          break;

        case 's':     /* Dot density for SAS        */
          if ((Ret = SubArg(++Arg, SUBA_POSINT, TRUE))) {
            sscanf(*Arg, "%d", &Prefs.SAS_POINTS);
            if (Prefs.SAS_POINTS == 0) Ret = CatErr(MSG_ERR_MAIN_ILLSASDEN);
          }
          break;

        case 't':                       /* Port number                */
#ifdef WIN32
          if ((Ret = SubArg(++Arg, SUBA_POSINT, TRUE))) {
            VegaPort = 0;
            sscanf(*Arg, "%d", &VegaPort);
            if (VegaPort > 0) {
              sprintf(Buf, VG_CLASS_NAME, VegaPort);
              if (FindWindow(Buf, NULL))
                Ret = CatErr(MSG_ERR_GL_RUNNING, VegaPort);
              else
               GLOBSW_OPENGL = TRUE;
               OutFormat -> Id = FORM_OPENGL;
            } else Ret = CatErr(MSG_ERR_MAIN_INVALIDPORT);
          }
#else
          Ret = CatErr(MSG_ERR_MAIN_WIN32ONLY);
#endif
          break;

        case 'w':     /* Remove waters              */
          Switch[SW_RMWATER]  = TRUE;
          TrjPar.Flags       |= SRF_FLG_NOWAT;
          break;

        case 'h':     /* Help                       */
        default:
          Sintassi();
        }
      } else {
        ++TotFiles;
        *CurFile = *Arg;
        ++CurFile;
      }
    } while ((Ret) && (*Arg != argv[argc - 1]));

    if (Ret) {
      if (TotFiles) {
        if ((!*OutFile) && (OutFormat -> Id != FORM_OPENGL)) {
          GLOBSW_STDOUT = TRUE;
          if (!(OutFormat -> Flags & FORM_FLAG_STDOUT))
            Ret = CatErr(MSG_ERR_MAIN_NOSTDOUT);
          if (PackMode && !*OutFile)
            Ret = CatErr(MSG_ERR_MAIN_PKNOSTDOUT);
        }

        /**** Check if the output format support the selected options ****/

        if (Ret) {
          if ((Switch[SW_FFIELD]) && (!(OutFormat -> Flags & FORM_FLAG_FFCALC)))
            CatErr(MSG_ERR_MAIN_NOFFTAG);
          else {
            if ((Switch[SW_CHARGES]) &&
                (!(OutFormat -> Flags & FORM_FLAG_CHARGE)))
              CatErr(MSG_ERR_MAIN_NOCHARGE);
            else {

              if ((!Switch[SW_FFIELD]) && (!Switch[SW_RMHYDROG]) &&
                  (!Switch[SW_RMWATER]) && (!Switch[SW_CHARGES])) {
                if ((!Switch[SW_NOCONNECT]) ||
                    (OutFormat -> Flags & FORM_FLAG_CONNAL) ||
                    (TrjPar.Com == MS_VLOGP)) {
                  if (OutFormat -> Flags & FORM_FLAG_CONN)
                    GLOBSW_CONNCALC = TRUE;
                  else GLOBSW_CONNCALC = FALSE;
                }
              } else GLOBSW_CONNCALC = TRUE;

              /**** Multiple loads ****/

              TotalAtm  = 0;
              CurFile   = Files;
              j         = GLOBSW_CONNCALC;
              for(k = 1; k <= TotFiles; ++k) {
                LastSeg    = LastAtm;
                LastSrfSeg = LastSrf;
                TmpIniz    = Loader(*CurFile, &TmpTotAtm, &FormatID, NULL, k, &InizSrf, &TotSrf);
                if (InizSrf) {
                  TotalSrf += TotSrf;
                  if (BegSrf) {
                    LastSrfSeg -> Next = InizSrf;
                    InizSrf    -> Prev = LastSrfSeg;
                  } else BegSrf = InizSrf;
                }

                if (TmpIniz) {
                  if (TmpIniz != (ATOMO *)-2) {
                    TotalAtm += TmpTotAtm;
                    if (BegAtm) {                 /* Binding */
                      LastSeg -> Ptr  = TmpIniz;
                      TmpIniz -> Prev = LastSeg;
                    } else BegAtm = TmpIniz;
                    if ((GLOBSW_CONNCALC) && (!Connect(TmpIniz, TmpTotAtm, Prefs.CONNTOL, FALSE))) {
                      Switch[SW_SAVMOL] = TRUE;
                      Ret               = FALSE;
                      break;
                    }
                    if (OutFormat -> Flags & FORM_FLAG_VDW) {
                      PrintProg(MSG_MAIN_ASSVDW);
                      AssignVdwNew(BegAtm, 0, NULL);
                    }
                  }
                } else {
                  Ret = FALSE;
                  break;
                }
                GLOBSW_CONNCALC = j;
                ++CurFile;
              } /* End of for */

              if ((Ret) && (BegAtm)) {
                if (((OutFormat -> Id == FORM_CSSR) ||
                     (OutFormat  -> Id == FORM_QMC )) && (TotalAtm > 9999)) {
                  Ret = CatErr(MSG_ERR_MAIN_CSSR);
                } else {
                  if (Switch[SW_NORMALIZE]) Normalize(BegAtm);
                  if (Switch[SW_RENRES]) RenRes(BegAtm, Prefs.RENSTART, FALSE);
                  if ((Switch[SW_RMWATER])&& (!Prefs.DCD_FILE)) {
                    RemoveH2O(&BegAtm, &TotalAtm);
                    if (!TotalAtm) Ret = FALSE;
                  }

                  if (Ret) {

                    /**** Solvate the molecule ****/

                    if (Switch[SW_SOLVATE])
                      Ret = Solvate(Shell, BegAtm, &TotalAtm, NULL);

                    /**** Find the rings ****/

                    if ((Switch[SW_FINDRINGS]) ||
                        (OutFormat -> Flags & FORM_FLAG_RINGS))
                      FindRing(BegAtm, FALSE);

                    /**** Remove the hydrogens ****/

                    if (Switch[SW_RMHYDROG]) {
                      PrintProg(MSG_UTILS_RMHPROG);
                      RemoveH(&BegAtm, &TotalAtm, FALSE);
                    }

                    /**** Add the hydrogens ****/

                    if (AddhFlags != VG_ADDH_FLG_NONE)
                      AddHyd(&BegAtm, &TotalAtm, AddhFlags);

                    /**** Compute the atomic charges ****/

                    if ((Switch[SW_CHARGES]) &&
                        (!Charge(FCharge, CharFormat, BegAtm, TotalAtm, FALSE)))
                      Ret = FALSE;

                    /**** Fix the atom types ****/

                    else if (Switch[SW_FFIELD])
                      AssignFF(FField, BegAtm, TotalAtm, FALSE, FALSE);
                  }
                }

                if (Ret) {
                  if ((!GLOBSW_OPENGL) && (Prefs.DCD_FILE)) {
                    PrintProg(MSG_MAIN_DCDPROG);
                    TrjPar.OutFile   = OutFile;
                    TrjPar.OutForm   = OutFormat;
                    Ret = TrjAnalyzer(Prefs.DCD_FILE, BegAtm, TotalAtm, &TrjPar);
                  } else if (OutFormat -> Id != FORM_OPENGL) {
                    Ret = Saver(OutFile, OutFormat, PackMode, BegAtm, TotalAtm);
#ifdef __VG_OPENGL
                  } else { /* Activate the OpenGL output */
                    FindRing(BegAtm, FALSE);
                    PrintProg(MSG_MAIN_OPENGLSTART);
                    Ret = GL_Start(*--CurFile, FormatID);
                  }
#else
                  }
#endif
                  if (Ret) PrintProg(MSG_MAIN_DONEPROG);
                } else PrintProg(MSG_MAIN_NOTSAVPROG);
              } else if (Region.Points) CatErr(MSG_ERR_MAIN_RGNNOMOL);
            }
          }
        }
#ifdef __VG_OPENGL
      } else if (OutFormat -> Id == FORM_OPENGL) {
        GLOBSW_CONNCALC = TRUE;
        Ret = GL_Start(NULL, 0);
      } else CatErr(MSG_ERR_MAIN_FILEMISS);
#else
      }
#endif
    }
    FREE(Files);
  }
  FreeAtm(BegAtm);
  CloseAll();

  return 0;
}


/**** Abort code ****/

void Abort(void)
{
  LocPrintf(stderr, "\n%s\n\n", GetStr(MSG_ERR_MAIN_ABORT));
  GlobErr = VG_PARET_ABORT;
  CloseAll();
  exit(0);
}


/**** Close all ****/

void CloseAll(void)
{
  SrfFree(BegSrf);
  SelRemAll();
  BegSrf    = NULL;
  TotalSrf  = 0;
  AtmLoaded = 0;


  MyCloseAll();                 /* Close all opened files */
  CatClose();                   /* Close the catalog      */

#ifdef AMIGA
  if (PPBase)     CloseLibrary(PPBase);
#endif

#ifdef WIN32
  PluginFree();                 /* Close all plugins      */

  if (ReturnStr) FREE(ReturnStr);
  if (hFmod)     FreeLibrary(hFmod);
  if (hLibBz2)   FreeLibrary(hLibBz2);
  if (hZ32)      FreeLibrary(hZ32);
  if (hZip32)    FreeLibrary(hZip32);
  if (hZipLib)   FreeLibrary(hZipLib);
  if (hZlib32)   FreeLibrary(hZlib32);
#ifdef USE_XDRF_DLL
  if (hXdrf)     FreeLibrary(hXdrf);
#endif
  if (hIl) {
    FreeLibrary(hIl);
    FreeLibrary(hIlu);
    FreeLibrary(hIlut);
  }
#endif
}

/**** Print copyright ****/

void PrtCopyright(void)
{
#ifdef WIN32
  ConPrintf(stdout, "\nVEGA V" VEGA_VERSTR " - (c) " VEGA_COPYRIGHT \
                    ", Alessandro Pedretti & Giulio Vistoli\n", NULL);
#else
  printf("\n\033[33mVEGA V" VEGA_VERSTR "\033[0m - (c) " VEGA_COPYRIGHT \
                  ", Alessandro Pedretti & Giulio Vistoli\n");
#endif

#ifdef VLOGP
  CatPrintf(stdout, MSG_MAIN_VLOGP);
#endif

  if (Catalog)
    LocPrintf(stdout, "%s\n", GetStr(MSG_TRANSLATION));

#ifdef AMIGA
#  define  OS_VERSION     "AmigaDOS"
#  ifdef mc68040
#    define  CPUOPT         " 68040"
#  else
#    ifdef mc68030
#      ifdef __HAVE_68881__
#        define  CPUOPT         " 68030/68881"
#      else
#        define  CPUOPT         " 68030"
#      endif
#    else
#      ifdef mc68020
#        ifdef __HAVE_68881__
#          define  CPUOPT         " 68020/68881"
#        else
#          define  CPUOPT         " 68020"
#        endif
#      else
#        ifdef mc68000
#          define  CPUOPT         " 68000"
#        endif
#      endif
#    endif
#  endif
#else
#  ifdef WIN32
#    define  OS_VERSION     "Windows 9x/ME/NT/2000/XP"
#    ifdef _M_IX86
#      if _M_IX86 == 300
#        define  CPUOPT         " 386"
#      else
#        if _M_IX86 == 400
#          define  CPUOPT         " 486"
#        else
#          if _M_IX86 == 500
#            define  CPUOPT         " Pentium"
#          else
#            if _M_IX86 == 600
#              define  CPUOPT         " Pentium Pro"
#            endif
#          endif
#        endif
#      endif
#    else
#     define  CPUOPT         ""
#    endif
#  else
#    ifdef  LINUX
#      define  OS_VERSION     "Linux"
#      define  CPUOPT         " x86"
#    else
#      ifdef  _SGI_SOURCE
#        define  OS_VERSION     "Silicon Graphics"
#        define  CPUOPT         ""
#      else
#        define  OS_VERSION     "UNIX"
#        define  CPUOPT         ""
#      endif
#    endif
#  endif
#endif

  CatPrintf(stdout, MSG_MAIN_OSCPU, OS_VERSION, CPUOPT);
}


/**** Show the correct syntax ****/

static void Sintassi(void)
{
  PrtCopyright();
  CatPrintf(stdout, MSG_MAIN_SYNOPSIS);
  CloseAll();
  exit(0);
}


/**** Extract arg options ****/

VG_BOOL SubArg(char **PtrArg, VG_WORD Opt, VG_BOOL Mod)
{
  VG_BOOL   Ret  = TRUE;
  register char   *Arg = *PtrArg;

  if ((PtrArg != LastArg) && (*Arg != '-')) {
    switch(Opt) {
    case SUBA_TOUP:         /* To upper                   */
      Upper(Arg);
      break;
    case SUBA_POSFLOAT:     /* Is positive float number   */
      do {
        if (((*Arg <  '0') || (*Arg > '9')) && (*Arg != '+') && (*Arg != '.')) {
          Ret = FALSE;
          if (Mod) CatErr(MSG_ERR_MAIN_FLOATNUM);
        }
      } while((Ret) && (*++Arg));
      break;

    case SUBA_POSINT:      /* Is positive integer number */
      do {
        if (((*Arg <  '0') || (*Arg > '9')) && (*Arg != '+')) {
          Ret = FALSE;
          if (Mod) CatErr(MSG_ERR_MAIN_INTNUM);
        }
      } while((Ret) && (*++Arg));
      break;

    default:               /* Nothing                    */
      break;
    }
  } else {
    if (Mod) CatErr(MSG_ERR_MAIN_SUBARG);
    Ret = FALSE;
  }

  return Ret;
}

