
/*************************************************
****      VEGA - Molecule superimposition     ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <malloc.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "overlay.h"
#include "selmol.h"

/**** Global variables ****/

VG_MATRIX       MySpos      = {0, 0, NULL };


/**** Calculate the RMSD ****/

VG_BOOL CalcRmsd(float *Rms, ATOMO **AtmX, ATOMO **AtmY, VG_BOOL ActOnly)
{
  ATOMO         *AtmTmp;
  double        Rmsd;
  VG_BOOL       Ret;

  if ((Ret = SuperimposeAll(&Rmsd, *AtmX, *AtmY, ActOnly))) {
    AtmTmp = *AtmX;
    *AtmX  = *AtmY;
    *AtmY  = AtmTmp;
    *Rms = (float)Rmsd;
  }

  return Ret;
}


/**** Calculate the superposition matrix ****/

VG_BOOL CalcSuperposMtx(VG_ULONG Natoms,  ATOMO **BegXMol, ATOMO **BegYMol,
                        VG_MATRIX *spos, XYZ *cgx, XYZ *cgy, VG_BOOL ActOnly)
{
  ATOMO         **XMol, **YMol, *AtmX, *AtmY;
  float         oneovern, sum, sigma, gamma, dist, qk, rk;
  VG_BOOL       rotated;
  VG_MATRIX     corr;
  VG_ULONG      i, j, k, p, q, r, itno;

  if (!MatrixAlloc(&corr, 3, 3)) return FALSE;

  /**** Calc cgx and cgy ****/

  cgx -> x = cgx -> y = cgx -> z = 0.0f;
  cgy -> x = cgy -> y = cgy -> z = 0.0f;

  /* If Natoms is 0 the two molecules are separated and are identical */

  if (Natoms) {
    XMol = BegXMol;
    YMol = BegYMol;
    for(i = 0; i < Natoms; i++) {
      cgx -> x += (*XMol) -> x;
      cgx -> y += (*XMol) -> y;
      cgx -> z += (*XMol) -> z;
      cgy -> x += (*YMol) -> x;
      cgy -> y += (*YMol) -> y;
      cgy -> z += (*YMol) -> z;
      ++XMol;
      ++YMol;
    } /* End of for */
    oneovern  = 1.0f / (float)Natoms;
  } else {
    for(AtmX = *BegXMol, AtmY = *BegYMol; (AtmX) && (AtmY);
        AtmX = AtmX -> Ptr, AtmY = AtmY -> Ptr) {
      if ((!ActOnly) || (AtmX -> Active)) {
        cgx -> x += AtmX -> x;
        cgx -> y += AtmX -> y;
        cgx -> z += AtmX -> z;
        cgy -> x += AtmY -> x;
        cgy -> y += AtmY -> y;
        cgy -> z += AtmY -> z;
      }
    } /* End of for */
    oneovern  = 1.0f / (float)TotalAtm;
  }

  cgx -> x *= oneovern;
  cgx -> y *= oneovern;
  cgx -> z *= oneovern;
  cgy -> x *= oneovern;
  cgy -> y *= oneovern;
  cgy -> z *= oneovern;

  /**** Initialise the correlation matrix ****/

  if (Natoms) {
    for(j = XAXIS; j <= ZAXIS; j++) {
      for(k = XAXIS; k <= ZAXIS; k++) {
        sum  = 0.0f;
        XMol = BegXMol;
        YMol = BegYMol;
        for(i = 0; i < Natoms; i++) {
          sum += (*((float *)&((*XMol) -> x) + j) - *((float *)cgx + j)) *
                 (*((float *)&((*YMol) -> x) + k) - *((float *)cgy + k));
          ++XMol;
          ++YMol;
        } /* End of for */
        MatrixSetElement(&corr, k, j, sum);
      } /* End of for */
    } /* End of for */
  } else {
    for(j = XAXIS; j <= ZAXIS; j++) {
      for(k = XAXIS; k <= ZAXIS; k++) {
        sum  = 0.0f;
        for(AtmX = *BegXMol, AtmY = *BegYMol; (AtmX) && (AtmY);
            AtmX = AtmX -> Ptr, AtmY = AtmY -> Ptr) {
          if ((!ActOnly) || (AtmX -> Active)) {
            sum += (*((float *)&(AtmX -> x) + j) - *((float *)cgx + j)) *
                   (*((float *)&(AtmY -> x) + k) - *((float *)cgy + k));
          }
        } /* End of for */
        MatrixSetElement(&corr, k, j, sum);
      } /* End of for */
    } /* End of for */
  }
  MatrixMakeUnit(spos);

  itno = 0;
  do {

    /**** Calc the rotation needed about each axis ****/

    rotated = FALSE;
    for(p = XAXIS; p <= ZAXIS; p++) {
      q = (p + 1 <= ZAXIS) ? p + 1 : XAXIS;
      r = (q + 1 <= ZAXIS) ? q + 1 : XAXIS;

      /**** Calc the rotation needed, angle = arctan(sigma/gamma) ****/

      sigma = MatrixGetElement(&corr, r, q) - MatrixGetElement(&corr, q, r);
      gamma = MatrixGetElement(&corr, q, q) + MatrixGetElement(&corr, r, r);

      /**** If the angle is not too small update the two matrices ****/

      if ((Natoms == 2) && (sigma == 0.0f)) sigma = M_PI;

      dist = sigma / gamma;
      FastAbs(dist);
      if (dist > VG_OVER_TOLERANCE) {
        dist = (float)sqrt(gamma * gamma + sigma * sigma);
        if (dist != 0.0f) {
          for(k = XAXIS; k <= ZAXIS; k++) {

            /**** The superposition matrix ****/

            qk = MatrixGetElement(spos, q, k);
            rk = MatrixGetElement(spos, r, k);
            MatrixSetElement(spos, q, k, ( gamma * qk + sigma * rk) / dist);
            MatrixSetElement(spos, r, k, (-sigma * qk + gamma * rk) / dist);

            /**** and the correlation matrix ****/

            qk = MatrixGetElement(&corr, q, k);
            rk = MatrixGetElement(&corr, r, k);
            MatrixSetElement(&corr, q, k, ( gamma * qk + sigma * rk) / dist);
            MatrixSetElement(&corr, r, k, (-sigma * qk + gamma * rk) / dist);
          } /* End of for */
        }
        rotated = TRUE;
      }
    }
    itno++;
  } while(rotated && itno < VG_OVER_MAXITERS);

  MatrixForget(&corr);

  return TRUE;
}


/**** Superimpose two molecules with the same number of atoms ****/

VG_BOOL SuperimposeAll(double *Rms, ATOMO *BegXMol, ATOMO *BegYMol, VG_BOOL ActOnly)
{
  ATOMO         *AtmX, *AtmY;
  float         x, y, z;
  VG_MATRIX     Ypos, Result, Spos;
  XYZ           Cgx, Cgy, *Ptr;
  VG_ULONG      Num;

  VG_BOOL       Ret  = FALSE;

  if (MatrixAlloc(&Spos, 3, 3)) {
    if (MatrixAlloc(&Ypos, 3, 1)) {
      if (MatrixAlloc(&Result, 3, 1)) {
        if (CalcSuperposMtx(0, &BegXMol, &BegYMol, &Spos, &Cgx, &Cgy, ActOnly)) {
          if (Rms) {
            Num  = 0;
            *Rms = 0.0;
          }
          for(AtmX = BegXMol, AtmY = BegYMol; (AtmX) && (AtmY);
              AtmX = AtmX -> Ptr, AtmY = AtmY -> Ptr) {
            if ((!ActOnly) || (AtmY -> Active)) {
              MatrixSetElement(&Ypos, 0, 0, AtmY -> x - Cgy.x);
              MatrixSetElement(&Ypos, 1, 0, AtmY -> y - Cgy.y);
              MatrixSetElement(&Ypos, 2, 0, AtmY -> z - Cgy.z);
              MatrixMult(&Spos, &Ypos, &Result);
              x = Cgx.x + MatrixGetElement(&Result, 0, 0);
              y = Cgx.y + MatrixGetElement(&Result, 1, 0);
              z = Cgx.z + MatrixGetElement(&Result, 2, 0);
              if (Rms) {
                if (!(AtmX -> Flags & VG_ATMF_CENTROID)) {
                  *Rms += (double)(Quad(AtmX -> x - x) + Quad(AtmX -> y - y) +
                          Quad(AtmX -> z - z));
                  ++Num;
                }
              } else {
                AtmY -> x = x;
                AtmY -> y = y;
                AtmY -> z = z;
              }
            }
          }  /* End of for */
          if (Rms) *Rms = sqrt(*Rms / (double)Num);
          Ret = TRUE;
        }
        MatrixForget(&Result);
      }
      MatrixForget(&Ypos);
    }
    MatrixForget(&Spos);
  }

  return Ret;
}


/**** Alloc a matrix ****/

VG_BOOL MatrixAlloc(VG_MATRIX *m, int rows, int cols)
{
   if((m -> element = (float *)Alloca(rows * cols * sizeof(float))) != NULL) {
     m -> rows = rows;
     m -> cols = cols;
     return TRUE;
   }

   return FALSE;
}


/**** Free the matrix ****/

void MatrixForget(VG_MATRIX *m)
{
  FREE(m -> element);
  m -> cols    = 0;
  m -> element = NULL;
  m -> rows    = 0;
}


/**** Make unit ****/

VG_MATRIX *MatrixMakeUnit(VG_MATRIX *m)
{
  VG_ULONG      i, j;

  for(i = 0; i < m -> rows; i++)
    for(j = 0; j < m -> cols; j++)
      MatrixSetElement(m, i, j, (i==j) ? 1.0 : 0.0);

   return m;
}


/**** Multiply two matrix ****/

VG_MATRIX *MatrixMult(VG_MATRIX *m1, VG_MATRIX *m2, VG_MATRIX *r)
{
  VG_ULONG      i, j, k;
  float         e;


  for(i = 0; i < m1 -> rows; i++) {
    for(j = 0; j < m2 -> cols; j++) {
      e = 0;
      for(k = 0; k < m1 -> cols; k++)
        e += MatrixGetElement(m1, i, k) * MatrixGetElement(m2, k, j);
      MatrixSetElement(r, i, j, e);
    } /* End of for */
  } /* End of for */

  return r;
}


/*************************
**** OpenGL functions ****
*************************/

#ifdef __VG_OPENGL

/**** Calculate the superimposition RMS ****/

float CalcSuperposRms(VG_ULONG Natoms,  ATOMO **XMol, ATOMO **YMol)
{
  VG_ULONG      k;
  float         Rms = 0.0f;

  for(k = 0; k < Natoms; ++k) {
    Rms += Quad((*XMol) -> x - (*YMol) -> x) +
           Quad((*XMol) -> y - (*YMol) -> y) +
           Quad((*XMol) -> z - (*YMol) -> z);
    ++XMol;
    ++YMol;
  } /* End of for */

  return SQR(Rms / Natoms);
}


/**** Superimpose two molecules ****/

VG_BOOL SuperimposeMol(VG_ULONG Natoms, ATOMO **BegXMol, ATOMO **BegYMol,
                       VG_ULONG MolY, XYZ **BegUndo)
{
  ATOMO         *Atm;
  VG_MATRIX     ypos, result;
  XYZ           Cgx, Cgy, *Ptr;
  VG_ULONG      k;

  VG_BOOL       Ret  = FALSE;
  VG_ULONG      NMol = 1;

  if ((MySpos.element == NULL) && (!MatrixAlloc(&MySpos, 3, 3)))
    return FALSE;

  if (MatrixAlloc(&ypos, 3, 1)) {
    if (MatrixAlloc(&result, 3, 1)) {
      if ((BegUndo) && ((k = MolListGet(MolY, NULL, NULL)) != 0) &&
          ((*BegUndo = (XYZ *)Alloca(k * sizeof(XYZ))) != NULL))
        Ptr = *BegUndo;
      else
        Ptr = NULL;

      if (CalcSuperposMtx(Natoms, BegXMol, BegYMol, &MySpos, &Cgx, &Cgy, FALSE)) {
        Ret = TRUE;
        for(Atm = BegAtm; (Atm) && (NMol <= MolY); Atm = Atm -> Ptr) {
          if (NMol == MolY) {
            if (Ptr) {
              Ptr -> x = Atm -> x;
              Ptr -> y = Atm -> y;
              Ptr -> z = Atm -> z;
              ++Ptr;
            }
            MatrixSetElement(&ypos, 0, 0, Atm -> x - Cgy.x);
            MatrixSetElement(&ypos, 1, 0, Atm -> y - Cgy.y);
            MatrixSetElement(&ypos, 2, 0, Atm -> z - Cgy.z);
            MatrixMult(&MySpos, &ypos, &result);
            Atm -> x = Cgx.x + MatrixGetElement(&result, 0, 0);
            Atm -> y = Cgx.y + MatrixGetElement(&result, 1, 0);
            Atm -> z = Cgx.z + MatrixGetElement(&result, 2, 0);
          }
          if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
        }  /* End of for */
      }
      MatrixForget(&result);
    }
    MatrixForget(&ypos);
  }

  return Ret;
}
#endif
