
/*************************************************
****                VEGA - PDB                ****
****   Loader, saver and trajectory routines  ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * Supported PDB formats:
 *
 * PDB 2.2               read & write
 * PDB standard          read & write
 * PDB non standard      read & write
 * PDBQ                  write only
 * PDB Orac              read only
 * PDBF (Fat subformat)  read & write
 * PDBA (Atdl subformat) read & write
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "bond.h"
#include "count.h"
#include "formats.h"
#include "traj.h"

/**** Local definitions ****/

struct _PdbExt {
  VG_UWORD      Type;
  VG_ULONG      FoundStr;
  char          *RecStr;
};

static const struct _PdbExt     PdbExt[] = {{PDB_ATDL, MSG_PDB_ATDLFOUND, "REMARK  78"},
                                            {PDB_FAT , MSG_PDB_FATFOUND , "REMARK  77 EXTRA"},
                                            {PDB_ORAC, MSG_PDB_ORACFOUND, "REMARK   1 Configuration at time step"},
                                            {0       , 0                , NULL }
                                           };

/**** Local prototypes ****/

static ATOMO     **PDBAllocLoadTable(ATOMO *, VG_ULONG);
static VG_ULONG  *PDBAllocSaveTable(ATOMO *, VG_ULONG);


/**** Allocate the pdb atom table for random access to the list ****/

static ATOMO **PDBAllocLoadTable(ATOMO *Atm, VG_ULONG MaxSerial)
{
  ATOMO         **AtmTable;

  if ((AtmTable = (ATOMO **)Alloca(sizeof(ATOMO *) * (MaxSerial + 1))) != NULL) {
    while(Atm) {
      AtmTable[Atm -> Num] = Atm;
      Atm    = Atm -> Ptr;
    } /* End of while */
  }

  return AtmTable;
}

/**** Allocate the pdb atom table number for random ****/

static VG_ULONG *PDBAllocSaveTable(ATOMO *Atm, VG_ULONG TotAtm)
{
  VG_ULONG         *AtmTable, *Ptr;

  VG_ULONG         NumTer = 0;

  if ((AtmTable = (VG_ULONG *)Alloca(sizeof(VG_ULONG) * (TotAtm + 1))) != NULL) {
    Ptr = AtmTable + 1;
    while(Atm) {
      *Ptr = Atm -> Num + NumTer;
      if (Atm -> Flags & VG_ATMF_SEGEND) ++NumTer;
      Atm = Atm -> Ptr;
      ++Ptr;
    } /* End of while */
  }

  return AtmTable;
}


/**** Close the trajectory file ****/

void PDBClose(TRJINFO *Info)
{
  if (Info -> FH)      fclose(Info -> FH);
  if (Info -> SeekTab) FREE(Info -> SeekTab);
}


/**** Universal PDB Loader ****/

ATOMO *PDBLoad(FILE *PDBIN, RECORD *PDBLin, VG_ULONG *TotAtomi, VG_LONG *FormatID, VG_BOOL *IsMulti)
{
  ATOMO                         *Bnd;
  char                          *j, ResSeq[20];
  struct _PdbExt                *Ptr;
  VG_ULONG                      ConNum[5];
  VG_LONG                       k;
  VG_ULONG                      StartNum, MaxSerial;
  VG_UWORD                      l;

  ATOMO                         **ConMtx  = NULL;
  ATOMO                         *InizAtm  = NULL;
  ATOMO                         *PrecAtm  = NULL;
  ATOMO                         *Atm      = NULL;
  VG_BOOL                       FirstAtm  = FALSE;
  VG_ULONG                      CountConn = 0;
  VG_UWORD                      IsPdbf    = 0;

  *FormatID = FORM_PDB;
  fseek(PDBIN, 0, SEEK_SET);
  while(fgets(PDBLin -> Line, VG_LINELEN, PDBIN)) {

    /**** PDBA/PDBF recognition ****/

    if (PDBLin -> Hdr == *(VG_LONG *)"REMA") {
      for(Ptr = (struct _PdbExt *)PdbExt; Ptr -> RecStr; ++Ptr) {
        if (!strncmp(PDBLin -> Line, Ptr -> RecStr, strlen(Ptr -> RecStr))) {
          if (!IsPdbf)
            PrintProg(Ptr -> FoundStr);
          IsPdbf = Ptr -> Type;
          break;
        }
      } /* End of for */

      switch(IsPdbf) {
      case PDB_ATDL:
        *FormatID = FORM_PDB_ATDL;
        if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
      	  sscanf(PDBLin -> Line + 16, "%f %8s", &Atm -> Charge, Atm -> Pot.C);
        } else break;
        break;

      case PDB_FAT:
        *FormatID = FORM_PDB_FAT;
        if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
      	  sscanf(PDBLin -> Line + 26, "%8s %f", Atm -> Pot.C, &Atm -> Charge);
        } else break;
        break;
      } /* End of switch */

    /**** ATOM/HETATM records ****/

    } else if ((PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[0]) ||
               (PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[1])) {
      if (((IsPdbf == PDB_ATDL) || (IsPdbf == PDB_FAT)) && (!FirstAtm)) {
        Atm       = InizAtm;
        PrecAtm   = InizAtm;
      }

      if ((IsPdbf != PDB_ATDL) && (IsPdbf != PDB_FAT))
        Atm = AllocAtm(&InizAtm, TotAtomi);
      if (Atm) {

        /**** Atom name & element ****/

        if (!FirstAtm) StartNum  = Atm -> Num;

      	sscanf(PDBLin -> Line + 6, "%d %4s", &Atm -> Num, Atm -> Name.C);

        if (!FirstAtm) {
          FirstAtm  = TRUE;
          MaxSerial = Atm -> Num;
        } else if (MaxSerial < Atm -> Num) MaxSerial = Atm -> Num;

        if (IsPdbf != PDB_ORAC) {
      	  l = *Atm -> Name.C;
      	  if (isdigit(l)) {
            for(k = 0; (k < 4) && (Atm -> Name.C[k]); ++k);
            for(j = Atm -> Name.C;j < (Atm -> Name.C + sizeof(Atm -> Name.C) - 1); ++j)
              *j = j[1];
            Atm -> Name.C[--k] = l;
      	  }
      	  *Atm -> Elem.C = toupper(*Atm -> Name.C);
      	  if ((isalpha(PDBLin -> Line[12])) && (!isdigit(Atm -> Name.C[1])))
      	    Atm -> Elem.C[1] = tolower(Atm -> Name.C[1]);

          /**** if present, decode the element name ****/

          if ((strlen(PDBLin -> Line) >= 79) && (PDBLin -> Line[77] != ' ') &&
              (!isdigit(PDBLin -> Line[76])) && (!isdigit(PDBLin -> Line[77]))) {
            *FormatID = FORM_PDB_2;
            if (PDBLin -> Line[76] != ' ') {
              Atm -> Elem.C[0] = PDBLin -> Line[76];
              Atm -> Elem.C[1] = PDBLin -> Line[77];
            } else {
              Atm -> Elem.C[0] = PDBLin -> Line[77];
              Atm -> Elem.C[1] = 0;
            }
          }
        } else *Atm -> Elem.C = *Atm -> Name.C;

        /**** Residue name and residue number ****/

        Atm -> ResName.L = 0;
      	sscanf(PDBLin -> Line + 17, "%3s", Atm -> ResName.C);
      	Atm -> ChainID = PDBLin -> Line[21];
      	sscanf(PDBLin -> Line + 22, "%4s", ResSeq);
      	Str2Qchar(&Atm -> ResSeq, ResSeq);
      	sscanf(PDBLin -> Line + 30, "%8f%8f%8f", &Atm -> x, &Atm -> y, &Atm -> z);

        /**** Adjust the ATOM/HETATM flags ****/

      	if (PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[0])
          Atm -> Flags = VG_ATMF_NONE;
      	else
          Atm -> Flags = VG_ATMF_HETATM;

        PrecAtm = Atm;
        if ((IsPdbf == PDB_ATDL) || (IsPdbf == PDB_FAT))
          Atm = Atm -> Ptr;
      } else break;

    /**** CONECT record ****/

    } else if (PDBLin -> Hdr == *(VG_LONG *)"CONE") {
      if (MaxSerial >= *TotAtomi) {
        if ((!ConMtx) &&
            ((ConMtx = PDBAllocLoadTable(InizAtm, MaxSerial)) == NULL)) break;
        PDBLin -> Line[31] = 0;
        j = PDBLin -> Line + 6;
        for(k = 0; k < 5; ++k) {
          if ((ConNum[k] = AnToI(j, 5)) == 0)
            break;
          j += 5;
        } /* End of for */

        if (ConNum[0] <= MaxSerial) {
          if ((Atm = ConMtx[ConNum[0]]) != NULL) {
            for(k = 1; (k < 5) && (ConNum[k]); ++k) {
              if ((ConNum[k] <= MaxSerial) &&
                  ((Bnd = ConMtx[ConNum[k]]) != NULL) &&
                  (!BndCheck(Atm, Bnd, NULL))) {
                BndMake(Atm, Bnd, VG_BOND_SINGLE);
              } else l = 0;
            } /* End of for */
          }
        }
        ++CountConn;
      }

    /**** TER record ****/

    } else if (PDBLin -> Hdr == *(VG_LONG *)"TER ") {
      PrecAtm -> Flags |= VG_ATMF_SEGEND;
      if (IsPdbf == PDB_ORAC) break;

    /**** ENDMODEL record ****/

    } else if (PDBLin -> Hdr == *(VG_LONG *)"ENDM") {
      if (IsMulti) *IsMulti = TRUE;
      break;
    }
  } /* End of while */

  if (PrecAtm) {
    PrecAtm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
    if (CountConn >= *TotAtomi) GLOBSW_CONNCALC = FALSE;
    RenAtm(InizAtm, StartNum);
  }
  if (ConMtx) FREE(ConMtx);
  LastAtm = PrecAtm;

  return InizAtm;
}


/**** Open the trajectory file ****/

VG_BOOL PDBOpen(char *FileName, TRJINFO *Info)
{
  RECORD        PDBLin;
  VG_ULONG      *Ptr;

  VG_BOOL       Ret = TRUE;

  if ((Info -> FH = fopen(FileName, "rb")) != NULL) {

    /**** Count the number of frames ****/

    while(fgets(PDBLin.Line, VG_LINELEN, Info -> FH)) {
      if (PDBLin.Hdr == *(VG_LONG *)"MODE")
        ++Info -> Frames;
    } /* End of while */

    if (Info -> Frames) {
      fseek(Info -> FH, 0, SEEK_SET);

      /**** Create the seek table ****/

      if ((Info -> SeekTab = (VG_ULONG *)Alloca(Info -> Frames * sizeof(VG_ULONG))) != NULL) {
        Ptr = Info -> SeekTab;
        while(fgets(PDBLin.Line, VG_LINELEN, Info -> FH)) {
          if (PDBLin.Hdr == *(VG_LONG *)"MODE") {
            *Ptr++ = ftell(Info -> FH);
          }
        } /* End of while */
        Info -> MolAtm = TotalAtm;
      } else Ret = FALSE;
    }
  } else Ret = PrintDosErr();

  if (!Ret) PDBClose(Info);

  return Ret;
}


/**** Read one trajectory frame ****/

VG_BOOL PDBReadFrm(TRJINFO *Info, ATOMO *Atm)
{
  RECORD        PDBLin;

  VG_ULONG      Num    = 0;
  VG_ULONG      TotAtm = CountAtms(Atm, FALSE);

  while((fgets(PDBLin.Line, VG_LINELEN, Info -> FH)) &&
        (PDBLin.Hdr != *(VG_LONG *)"ENDM")) {
    if ((PDBLin.Hdr == *(VG_LONG *)PDBAtmRec[0]) ||
        (PDBLin.Hdr == *(VG_LONG *)PDBAtmRec[1])) {

     if (!(Atm -> Flags & VG_ATMF_CENTROID)) {
       if (Num++ < TotAtm)
         sscanf(PDBLin.Line + 30, "%8f%8f%8f", &Atm -> x, &Atm -> y, &Atm -> z);
       else
         return FALSE;
     }

     Atm = Atm -> Ptr;
    }
  } /* End of while */

  if (Num != TotAtm) return FALSE;

  return TRUE;
}


/**** PDB Saver ****/

VG_BOOL PDBSave(FILE *OUT, ATOMO *AtmIniz, VG_ULONG Tot, VG_WORD Type)
{
  char              AtdlStr[10], *j;
  register ATOMO    *Atm;
  VG_QCHAR          Name;
  VG_ULONG          *NumMtx;
  VG_UWORD          l;
  VG_WORD           Hdr;

  VG_BOOL           ConSav    = GLOBSW_CONNSAV;
  VG_BOOL           Ret       = TRUE;
  float             Charge    = 0;
  VG_ULONG          k         = 1;
  VG_ULONG          NumConect = 0;
  VG_ULONG          NumRem    = 3;
  VG_UWORD          NumTer    = 0;


  /**** Check if the connectivity can be saved ****/

  if ((ConSav) && (Tot > 99999)) {
    PrintWarn(GetStr(MSG_WARN_PDB_NOCONN));
    ConSav = FALSE;
  }

  /**** Remarks ****/

  if (fprintf(OUT, "REMARK   4\n" \
                   "REMARK   4 %s\n"\
                   "REMARK   4\n", VEGAHdr) >= 0) {

    switch(Type) {
    case PDB_ATDL:
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
        AtmToAtdl(AtdlStr, Atm);
        if (fprintf(OUT, "REMARK  78 %5d %8.4f %-8.8s %s",
                    Atm -> Num, Atm -> Charge, Atm -> Pot.C, AtdlStr) > 0) {
          for(l = 0; (Ret) && (l < Atm -> NSost); ++l) {
            if (l) j = " ";
            else j = " (";
            if (fprintf(OUT, j) <= 0) {
              Ret = PrintDosErr();
              break;
            }
            AtmToAtdl(AtdlStr, Atm -> Conn[l]);
            if (fprintf(OUT, AtdlStr) <= 0)
              Ret = PrintDosErr();
          } /* End of for */
          if (Ret) {
            if (l) j = ")\n";
            else j = "\n";
            if (fprintf(OUT, j) < 0)
              Ret = PrintDosErr();
          }
        } else Ret = PrintDosErr();
      } /* End of atom loop */
      NumRem += Tot;
      break;

    case PDB_FAT:
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
        if (fprintf(OUT, "REMARK  77 EXTRA %5d %-2.2s %-8.8s %8.4f\n",
                    Atm -> Num, Atm -> Elem.C, Atm -> Pot.C, Atm -> Charge) < 0)
          Ret = PrintDosErr();
      } /* End of atom loop */
      NumRem += Tot;
      break;
    } /* End of switch */

    /**** Write ATOM/HETATM records ****/

    if (Ret) {
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {

        if (Type == PDB_NONSTD) Hdr = 0;
        else Hdr = Atm -> Flags & VG_ATMF_HETATM;

        if (fprintf(OUT, "%-6s%5d ", PDBAtmRec[Hdr], k) >= 0) {
          l = Atm -> Name.C[3];
          if (Atm -> Elem.C[1]) {
            if (fprintf(OUT, "%-4.3s", Atm -> Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (isdigit(l)) {
            Name.L = Atm -> Name.L;
            for(j = Name.C + 3;j > Name.C; --j)
              *j = *(j - 1);
            *Name.C = l;
            if (fprintf(OUT, "%-4.4s", Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (fprintf(OUT, " %-3.3s", Atm -> Name.C) < 0) {
            Ret = PrintDosErr();
            break;
          }

          if (Type == PDB_PDBQ) Charge = Atm -> Charge;
          else if (GLOBSW_FIXSAV) Charge = Atm -> Fix;

          if (fprintf(OUT, " %-3.3s %c%4.4s    %8.3f%8.3f%8.3f  1.00 %5.2f",
                      Atm -> ResName.C, Atm -> ChainID, Atm -> ResSeq.C,
                      Atm -> x, Atm -> y, Atm -> z, Charge) >= 0) {
            if (Type == PDB_2) {
              if (Atm -> ChainID != ' ') {
                sprintf(AtdlStr, "%c%-3d", Atm -> ChainID, NumTer + 1);
              } else *AtdlStr = 0;
              if (fprintf(OUT, "      %4.4s%2.2s\n", AtdlStr, Atm -> Elem.C) <= 0)
                Ret = PrintDosErr();
            } else if (fprintf(OUT, "\n") <= 0) Ret = PrintDosErr();

            if ((Ret) && (Type != PDB_NONSTD) && (Atm -> Flags & VG_ATMF_SEGEND)) {
              ++k;
              if (fprintf(OUT, "TER   %5d      %-3.3s %c%4.4s\n",
                          k, Atm -> ResName.C, Atm -> ChainID, Atm -> ResSeq.C) > 0) {
                ++NumTer;
              } else Ret = PrintDosErr();
            }
            ++k;
          } else Ret = PrintDosErr();
        } else Ret = PrintDosErr();
      }  /* End of atom loop */

      if (Ret) {

        /**** Write the connectivity ****/

        if (ConSav) {
          if ((NumMtx = PDBAllocSaveTable(AtmIniz, Tot)) != NULL) {
            for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
              Hdr = 0;
              for(k = 0; (Ret) && (Atm -> Conn[k]) && (k < MAXBOND); ++k) {
                if (k == 4) {
                  if (fprintf(OUT, "\n") <= 0) Ret = PrintDosErr();
                  else Hdr = 0;
                }
                if (!Hdr) {
                  if (fprintf(OUT, "CONECT%5d", NumMtx[Atm -> Num]) > 0) {
                    Hdr = 1;
                    ++NumConect;
                  } else Ret = PrintDosErr();
                }
                if ((Ret) && (fprintf(OUT, "%5d", NumMtx[Atm -> Conn[k] -> Num]) < 0))
                  Ret = PrintDosErr();
              } /* End of bond order loop */
              if ((Ret) && (Hdr) && (fprintf(OUT, "\n") <= 0))
                Ret = PrintDosErr();
            } /* End of atom loop */
            FREE(NumMtx);
          }
        }

        /**** Writes MASTER record ****/

        if ((Ret) && (Type != PDB_NONSTD) &&
            (fprintf(OUT,"MASTER %8d    0    0    0    0    0    0    0%5d%5d%5d    0\n",
                     NumRem, Tot, NumTer, NumConect) < 0))
          Ret = PrintDosErr();
        if ((Ret) && (fprintf(OUT, "END\n") < 0))
          Ret = PrintDosErr();
      }
    }
  } else Ret = PrintDosErr();

  return Ret;
}


/**** Seek a trajectory frame ****/

VG_BOOL PDBSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  VG_LONG       Offset;

  switch(Mode) {
  case SEEK_CUR:
    Offset = Info -> FrmCur + Frames;
    break;

  case SEEK_END:
    Offset = Info -> Frames - Frames;
    break;

  case SEEK_SET:
    Offset = Frames;
    break;
  } /* End of switch */

  if (Offset < 0)                       Offset = 0;
  if (Offset > (VG_LONG)Info -> Frames) Offset = Info -> Frames - 1;

  if (fseek(Info -> FH, Info -> SeekTab[Offset], SEEK_SET) == 0)
    return TRUE;

  return FALSE;
}

