
/*************************************************
****         VEGA - Interface to libbz2       ****
**** Copyright 1996-2001, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include "../Bzip2/bzlib.h"

#ifdef __WIN32__
#include <windows.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globstr.h"

#define  FALSE                  0
#define  TRUE                   1
#define  BZ2_BUFFER_SIZE        65536


/**** Local prototypes ****/

/* Some VEGA definitions are incompatible with the standard windows.h */
/* file and thus can't be used in the following prototypes            */

void         *Alloca(unsigned int);
short        CatErr(unsigned int, ...);
short        BZ2_DepackFile(char *, char *);
static short BZ2_Error(BZFILE *FH);
short        BZ2_PackFile(char *, char *);
short        PrintDosErr(void);

#ifdef WIN32
static short BZ2_Init(void);
short        GetProcAddresses(HINSTANCE *, LPCSTR, INT, ... );

/**** Library function pointers ****/

void          (WINAPI *VG_BZ2_bzclose)(BZFILE*)                   = NULL;
const char *  (WINAPI *VG_BZ2_bzerror)(BZFILE *, int *)           = NULL;
BZFILE *      (WINAPI *VG_BZ2_bzopen)(const char *, const char *) = NULL;
int           (WINAPI *VG_BZ2_bzread)(BZFILE*, void *, int)       = NULL;
int           (WINAPI *VG_BZ2_bzwrite)(BZFILE *, void *, int)     = NULL;

/**** Global variables ****/

extern HINSTANCE               hLibBz2;
#else
#  define  VG_BZ2_bzclose       BZ2_bzclose
#  define  VG_BZ2_bzerror       BZ2_bzerror
#  define  VG_BZ2_bzopen        BZ2_bzopen
#  define  VG_BZ2_bzread        BZ2_bzread
#  define  VG_BZ2_bzwrite       BZ2_bzwrite
#endif


/**** Decompress a file into another ****/

short BZ2_DepackFile(char *Src, char *Dest)
{
  char          *Buf;
  FILE          *FO;
  int           Rd;

  BZFILE       *FI = NULL;
  short        Ret = TRUE;

#ifdef WIN32
  if (!BZ2_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(BZ2_BUFFER_SIZE))) {
    if ((FI = VG_BZ2_bzopen(Src, "rb"))) {
      if ((FO = fopen(Dest, "wb"))) {
        while((Ret) && (Rd = VG_BZ2_bzread(FI, Buf, BZ2_BUFFER_SIZE))) {
          if (Rd == -1)
            Ret = BZ2_Error(FI);
          else if (fwrite(Buf, Rd, 1, FO) != 1)
           Ret = PrintDosErr();
        }
        fclose(FO);
      } else Ret = PrintDosErr();
      VG_BZ2_bzclose(FI);
    } else Ret = FALSE;
  } else Ret = BZ2_Error(FI);

  return Ret;
}


/**** Compress a file into another ****/

short BZ2_PackFile(char *Src, char *Dest)
{
  BZFILE        *FO;
  char          *Buf;
  FILE          *FI;
  int           Rd;

  short        Ret = TRUE;

#ifdef WIN32
  if (!BZ2_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(BZ2_BUFFER_SIZE))) {
    if ((FI = fopen(Src, "rb"))) {
      if ((FO = VG_BZ2_bzopen(Dest, "wb"))) {
        while ((Ret) && (Rd = fread(Buf, 1, BZ2_BUFFER_SIZE, FI))) {
          if (VG_BZ2_bzwrite(FO, Buf, Rd) != Rd)
            Ret = BZ2_Error(FO);
        }
        if (ferror(FI)) Ret = PrintDosErr();
        VG_BZ2_bzclose(FO);
      } else Ret = BZ2_Error(FO);
      fclose(FI);
    } else Ret = PrintDosErr();
  } else Ret = FALSE;

  return Ret;
}


/**** Show a bzip2 error ****/

static short BZ2_Error(BZFILE *FH)
{
  int           err;

  return CatErr(MSG_ERR_BZIP2_ERROR, VG_BZ2_bzerror(FH, &err));
}


#ifdef WIN32

/**** Library initialization ****/

static short BZ2_Init(void)
{
  if (hLibBz2)
    return TRUE;

  if (GetProcAddresses(&hLibBz2, "libbz2.dll", 5,
                   &VG_BZ2_bzclose, "BZ2_bzclose",
                   &VG_BZ2_bzerror, "BZ2_bzerror",
                   &VG_BZ2_bzopen , "BZ2_bzopen",
                   &VG_BZ2_bzread , "BZ2_bzread",
                   &VG_BZ2_bzwrite, "BZ2_bzwrite"
                   ) == FALSE)
    return FALSE;

  return TRUE;
}
#endif
