
/*************************************************
****        VEGA - un/packing utilities       ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/

/*
 * Universal data de/compressor engine for VEGA.
 */

#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "pk_def.h"

#define  FILESIZE            512


/**** Local prototypes ****/

static FILE *MyOpen(char *, char *, char *, VG_UWORD, VG_BOOL);


/**** Local definitions ****/

typedef struct __PkFileInfo {
  struct __PkFileInfo   *Ptr;             /* Pointer to next list element */
  FILE                  *FH;              /* File handle                  */
  char                  Src[FILESIZE];    /* File name with path          */
  char                  Dest[FILESIZE];   /* Temp file name               */
  VG_WORD               PackMode;         /* Packing mode                 */
  VG_BOOL               NoRemove;         /* Don't remove the file        */
} PKFILEINFO;


/**** Local variables ****/

static PKFILEINFO       *InizPkInfo    = NULL;

/**** Global variables ****/

const char              *PackerNames[] = { "NONE", "BZIP2", "GZIP", "POWERPACKER",
                                           "ZCOMPRESS", NULL};


/**** Check the packer format *****/

VG_LONG ChkPackerName(char *Packer, VG_BOOL Verbose)
{
  register VG_LONG  k;

  for(k = 0;((PackerNames[k]) && (strcasecmp(PackerNames[k], Packer))); ++k);
  if (PackerNames[k] == NULL) {
    if (Verbose) CatErr(MSG_ERR_MAIN_UNKPACKMET);
    k = -1;
  }

  return k;
}


/**** Get the temporary directory ****/

void GetTmpDir(char *Str, unsigned int Size)
{
#ifdef WIN32
  unsigned int          Len;

  Len = GetTempPath(Size, Str);
  if (Len < Size) {
    if (Str[Len - 1] != '\\') strcat(Str, "\\");
  } else strcpy(Str, ".\\");

#else
  char                  *Ptr;

  if (((Ptr = getenv("TEMP")) != NULL) ||
      ((Ptr = getenv("TMP" )) != NULL)) {
    strcpy(Str, Ptr);
    Ptr = Str + strlen(Str) - 1;
#ifdef AMIGA
    if ((*Ptr != '/') && (*Ptr != ':')) strcat(Str, "/");
#else
    if (*Ptr != '/') strcat(Str, "/");
#endif
  } else *Str = 0;
#endif
}


/**** Get a temporary file name with path ****/

void GetTmpFileName(char *Str)
{
#ifdef WIN32
  char          Tmp[MAX_PATH];

  GetTmpDir(Tmp, MAX_PATH);
  GetTempFileName(Tmp, "Vg", 0, Str);

#else
  mkstemp(Str);
#endif
}


/**** Close a temporary file and free all resources ****/

VG_BOOL MyClose(FILE *FH)
{
  PKFILEINFO            *PkInfo;

  PKFILEINFO            *PrecPkInfo = NULL;
  VG_BOOL               Ret         = TRUE;

  if (FH) {

    /**** Find the entry into the list ****/

    PkInfo     = InizPkInfo;
    while((PkInfo) && (PkInfo -> FH != FH)) {
      PrecPkInfo = PkInfo;
      PkInfo     = PkInfo -> Ptr;
    }
    fclose(FH);
    if (PkInfo) {

      /**** Compress the file ****/

      if (PkInfo -> PackMode) {
        PrintProg(MSG_PK_PROG);

        switch (PkInfo -> PackMode) {
        case PK_BZ2:
          Ret = BZ2_PackFile(PkInfo -> Src, PkInfo -> Dest);
          break;
        case PK_GZ:
          Ret = GZ_PackFile(PkInfo -> Src, PkInfo -> Dest);
          break;
        case PK_PP:
          Ret = PP_PackFile(PkInfo -> Src, PkInfo -> Dest);
          break;
        case PK_Z:
          Ret = Z_PackFile(PkInfo -> Src, PkInfo -> Dest);
          break;
        }
      }
      if ((!PkInfo -> NoRemove) && (remove(PkInfo -> Src)))
        Ret = PrintDosErr();                          /* Remove the file         */
      if (PkInfo == InizPkInfo) InizPkInfo = NULL;    /* Remake the list binding */
      else PrecPkInfo -> Ptr = PkInfo -> Ptr;
      FREE(PkInfo);
    }
  }

  return Ret;
}


/**** Close all opened file and remove the temporary files ****/

void MyCloseAll(void)
{
  while(InizPkInfo)
    MyClose(InizPkInfo -> FH);
}


/**** Open a temporary file and store all infos ****/

FILE *MyOpen(char *FileName, char *TmpFileName, char *Mode,
             VG_UWORD PackMode, VG_BOOL NoRemove)
{
  FILE          *FH;
  PKFILEINFO    *New, *PkInfo;

  if ((FH = fopen(FileName, Mode))) {
    if ((New = (PKFILEINFO *)Alloca(sizeof(PKFILEINFO)))) {
      strcpy(New -> Src, FileName);
      if (TmpFileName) strcpy(New -> Dest, TmpFileName);
      New -> FH       = FH;
      New -> NoRemove = NoRemove;
      New -> PackMode = PackMode;
      if (InizPkInfo) {

        /**** Add the entry into the list ****/

        for(PkInfo = InizPkInfo; PkInfo -> Ptr; PkInfo = PkInfo -> Ptr);
        PkInfo -> Ptr = New;
      } else InizPkInfo = New;
    } else {
      fclose(FH);
      FH = NULL;
    }
  } else PrintDosErr();

  return FH;
}


/**** Open & decrunch a packed file ****/

/*
 * FileName -> File name (with path) to open.
 * Mode     -> Access mode (e.g. r, w, rb, etc).
 * PackMode -> Packing mode only for write operations (see pk_def.h)
 * Verbose  -> If TRUE shows error messages
 */

FILE *PkOpen(char *FileName, char *Mode, VG_UWORD PackMode, VG_BOOL Verbose)
{
  char          TmpFileName[FILESIZE];
  VG_LONG       Hdr;

  VG_BOOL       Ret   = FALSE;
  FILE          *FH   = NULL;


  /**** Read a packed file ****/

  if (*Mode == 'r') {
    if ((FH = fopen(FileName, "rb"))) {
      if (fread(&Hdr, sizeof(VG_LONG), 1, FH) == 1) {
        fclose(FH);

#ifdef LITTLE_ENDIAN
        Swap(&Hdr);
#endif

        /**** Packer recognition ****/

        switch(Hdr) {
        case PK_HDR_POWERPACKER:   /* Amiga PowerPacker */
          GetTmpFileName(TmpFileName);
          Ret = PP_DepackFile(FileName, TmpFileName);
          break;

        default:

          switch (Hdr & 0xFFFF0000) {
          case PK_HDR_GZIP:        /* GZip */
            GetTmpFileName(TmpFileName);
            Ret = GZ_DepackFile(FileName, TmpFileName);
            break;

          case PK_HDR_COMPRESS:    /* Unix compress */
            GetTmpFileName(TmpFileName);
            Ret = Z_DepackFile(FileName, TmpFileName);
            break;

          default:
            if ((Hdr & 0xFFFFFFF0) == PK_HDR_BZIP2) { /* BZip2 */
              GetTmpFileName(TmpFileName);
              Ret = BZ2_DepackFile(FileName, TmpFileName);
            }
          } /* End of switch */
        } /* End of switch */

        /**** Open the file ****/

        if (Ret) FH = MyOpen(TmpFileName, NULL, Mode, PK_NONE, FALSE);
        else FH = MyOpen(FileName, NULL, Mode, PK_NONE, TRUE);
      } else if (Verbose) PrintDosErr();
    } else if (Verbose) PrintDosErr();
  } else {

    /**** Write a packed file ****/

    if (PackMode == PK_NONE)
      FH = MyOpen(FileName, NULL, Mode, PK_NONE, TRUE);
    else {
      GetTmpFileName(TmpFileName);
      FH = MyOpen(TmpFileName, FileName, Mode, PackMode, FALSE);
    }
  }

  return FH;
}


/**** Re-open a file without depacking ****/
/**** Useful to chenge the stream mode ****/

FILE *PkReOpen(FILE *FH, char *Mode)
{
  PKFILEINFO            *PkInfo     = NULL;
  PKFILEINFO            *PrecPkInfo = NULL;

  if (FH) {
    PkInfo     = InizPkInfo;
    while((PkInfo) && (PkInfo -> FH != FH)) {
      PrecPkInfo = PkInfo;
      PkInfo     = PkInfo -> Ptr;
    }
    if (PkInfo) {
      fclose(FH);
      if ((PkInfo -> FH = fopen(PkInfo -> Src, Mode)) == NULL) {
        PrintDosErr();
        if (!PkInfo -> NoRemove) remove(PkInfo -> Src); /* Remove the file         */
        if (PkInfo == InizPkInfo) InizPkInfo = NULL;    /* Remake the list binding */
        else PrecPkInfo -> Ptr = PkInfo -> Ptr;
        FREE(PkInfo);                                   /* Free the memory         */
      }
    }
  }

  return PkInfo -> FH;
}

