
/*************************************************
****     VEGA - Interface to Unix compress    ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>

#include "globstr.h"
#include "../Z32/z32.h"

#ifdef __WIN32__
#include <windows.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#define  FALSE                     0
#define  TRUE                      1
#define  Z_BUFFER_SIZE         65536


/**** Local prototypes ****/

/* Some VEGA definitions are incompatible with the standard windows.h */
/* file and thus can't be used in the following prototypes            */

void         *Alloca(unsigned int);
short        CatErr(unsigned int, ...);
short        Z_DepackFile(char *, char *);
static short Z_Error(ZSTREAM *FH);
short        Z_PackFile(char *, char *);
short        PrintDosErr(void);

#ifdef WIN32
static short Z32_Init(void);
short        GetProcAddresses(HINSTANCE *, LPCSTR, INT, ... );

/**** Library function pointers ****/

int           (WINAPI *VG_zclose)(ZSTREAM *)                      = NULL;
char    *     (WINAPI *VG_zerror)(ZSTREAM *, int *)               = NULL;
ZSTREAM *     (WINAPI *VG_zopen)(const char *, const char *, int) = NULL;
int           (WINAPI *VG_zread)(ZSTREAM *, char *, int)          = NULL;
int           (WINAPI *VG_zwrite)(ZSTREAM *, const char *, int)   = NULL;

/**** Global variables ****/

extern HINSTANCE                hZ32;

#else
#  define  VG_zclose           zclose
#  define  VG_zerror           zerror
#  define  VG_zopen            zopen
#  define  VG_zread            zread
#  define  VG_zwrite           zwrite
#endif


/**** Decompress a file into another ****/

short Z_DepackFile(char *Src, char *Dest)
{
  char          *Buf;
  FILE          *FO;
  int           Rd;
  ZSTREAM       *FI;

  short        Ret = TRUE;

#ifdef WIN32
  if (!Z32_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(Z_BUFFER_SIZE))) {
    if ((FI = VG_zopen(Src, "r", 0))) {
      if ((FO = fopen(Dest, "wb"))) {
        while((Ret) && (Rd = VG_zread(FI, Buf, Z_BUFFER_SIZE))) {
          if (Rd == -1)
            Ret = Z_Error(FI);
          else if (fwrite(Buf, Rd, 1, FO) != 1)
            Ret = PrintDosErr();
        }
        fclose(FO);
      } else Ret = PrintDosErr();
      VG_zclose(FI);
    } else Ret = Z_Error(FI);
  } else Ret = FALSE;

  return Ret;
}


/**** Compress a file into another ****/

short Z_PackFile(char *Src, char *Dest)
{
  char          *Buf;
  FILE          *FI;
  int           Rd;
  ZSTREAM       *FO;

  short        Ret = TRUE;

#ifdef WIN32
  if (!Z32_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(Z_BUFFER_SIZE))) {
    if ((FI = fopen(Src, "rb"))) {
      if ((FO = VG_zopen(Dest, "w", 0))) {
        while ((Ret) && (Rd = fread(Buf, 1, Z_BUFFER_SIZE, FI))) {
          if (VG_zwrite(FO, Buf, Rd) != Rd)
            Ret = Z_Error(FO);
        }
        if (ferror(FI)) Ret = PrintDosErr();
        VG_zclose(FO);
      } else Ret = Z_Error(FO);
      fclose(FI);
    } else Ret = PrintDosErr();
  } else Ret = FALSE;

  return Ret;
}


/**** Show a Z error ****/

static short Z_Error(ZSTREAM *FH)
{
  int           err;

  return CatErr(MSG_ERR_Z32_ERROR, VG_zerror(FH, &err));
}


#ifdef WIN32

/**** Library initialization ****/

static short Z32_Init(void)
{
  if (hZ32)
    return TRUE;

  if (GetProcAddresses(&hZ32, "z32.dll", 5,
                   &VG_zclose, "zclose",
                   &VG_zerror, "zerror",
                   &VG_zopen , "zopen",
                   &VG_zread , "zread",
                   &VG_zwrite, "zwrite"
                   ) == FALSE)
    return FALSE;

  return TRUE;
}
#endif
