
/*************************************************
****   VEGA - High level trajectory analyzer  ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#include <windows.h>
#include <GL\gl.h>
#include <conio.h>
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "atom.h"
#include "count.h"
#include "formats.h"
#include "logp.h"
#include "overlay.h"
#include "progbar.h"
#include "surface.h"
#include "traj.h"

#ifdef __VG_OPENGL
#include "gl_global.h"
#endif

/**** Local prototypes ****/

static VG_UWORD TrjFindFormat(char *);


/**** Open a plot file and recognize the format ****/

FILE *OpenPlot(char *FileName, VG_UWORD *Format)
{
  char          *Ptr1, *Ptr2, Buf[256], Str[6];
  FILE          *FH;
  VG_BOOL       Chk;

  VG_BOOL       Cont     = TRUE;
  char          *CsvFilt = " 1234567890.,;+-\t\012\015";
  VG_ULONG      Line     = 0;

  *Format = FPLOT_UNK;
  if (FH = PkOpen(FileName, "r", 0, FALSE)) {
    while((Cont) && (fgets(Buf, 256, FH))) {

      /**** NAMD out ****/

      if (!strncmp(Buf, "Info: NAMD", 10)) {
        *Format = FPLOT_NAMDOUT;
        break;
      } else {
        sscanf(Buf, "%4s", Str);
        ++Line;
        switch(Line) {
        case 2:
          if (!strncmp(Buf + 25, ":-)  G  R  O", 12)) {
            *Format = FPLOT_GROMACSLOG;
            Cont = FALSE;
          }
          break;

        case 10:
          Cont = FALSE;
          break;

        default:
          if (*Str) {

            /**** CHARMm ENE ****/

            if ((strlen(Buf) == 81) && (Buf[27] == '.') && (Buf[43] == '.') &&
                (Buf[59] == '.') && (Buf[75] == '.')) {
              fseek(FH, 0, SEEK_SET);
              *Format = FPLOT_CHARMMENE;
              Cont = FALSE;

            /**** CSV ****/

            } else if ((*Str >= '0') && (*Str <= '9')) {
              Chk = TRUE;
              for(Ptr1 = Buf; (*Ptr1) && (Chk); ++Ptr1) {
                Chk = FALSE;
                for(Ptr2 = CsvFilt; *Ptr2; ++Ptr2)
                  if (*Ptr2 == *Ptr1) {
                    Chk = TRUE;
                    break;
                  }
              } /* End of for */
              if (Chk) {
                fseek(FH, 0, SEEK_SET);
                *Format = FPLOT_CSV;
                Cont = FALSE;
              }
              break;
            }
          }
        } /* End of switch */
      }
    } /* End of while */

    if (*Format == FPLOT_UNK) {
      CatErr(MSG_ERR_LOADER_UNKFORMAT);
      fclose(FH);
      FH = NULL;
    }
  }

  return FH;
}



/**** Trajectory analyzer ****/

VG_BOOL TrjAnalyzer(char *FileName, ATOMO *InizAtm, VG_ULONG MolAtm, TRJPAR *Par)
{
  float                 *Result;

  VG_BOOL               Ret     = TRUE;

  if (TrjOpen(FileName, &TrjInfo, FTRJ_NONE)) {
    if (TrjInfo.MolAtm == CountAtms(InizAtm, FALSE)) {
      if ((Result = TrjCalc(&TrjInfo, Par, InizAtm, MolAtm)) != NULL) {
        if ((*Par -> OutFile) && (!strchr(Par -> OutFile, '.'))) {
          strcat(Par -> OutFile, ".");
          strcat(Par -> OutFile, Par -> OutForm -> Ext);
        }
        Ret = TrjPlotSave(Par -> OutFile, Par -> OutForm -> Id, Result, TrjInfo.Frames);
        FREE(Result);
      }
    } else Ret = CatErr(MSG_ERR_DCD_UNKTRAJ);
    TrjClose(&TrjInfo);
  }

  return Ret;
}


/**** Calculate a property ****/

float *TrjCalc(TRJINFO *Info, TRJPAR *Par, ATOMO *InizAtm, VG_ULONG MolAtm)
{
  ATOMO                 *Frm[MAXTRJATM];
  char                  *ProcFrmTxt;
  float                 *Iniz, *Ene;
  float                 EneMin, EneMax, IlmCutOff, MaxRad;
  VG_ULONG              Nu, YPos;
  XYZ                   BoxMin, BoxMax;
  register ATOMO        *Atm;
  register VG_ULONG     k;
  VG_BOOL               UpdateCentr;

  ATOMO                 *PrecBegAtm = NULL;
  float                 *Contrib    = NULL;
  float                 *VdwRad     = NULL;
  double                Media       = 0.0;
  double                Media2      = 0.0;
  VG_BOOL               Ret         = TRUE;
  VG_ULONG              FrmMin      = 1;
  VG_ULONG              FrmMax      = 1;
  XYZ                   *SfUn       = NULL;

#ifdef WIN32
  char                  Buf[80];
#endif

  if (Par -> Flags & SRF_FLG_NOWAT) RenameWat(InizAtm);

  if ((Iniz = (float *)Alloca(Info -> Frames * sizeof(float))) &&
      ((Ret = TrjReadFrm(Info, 1)))) {

    UpdateCentr = CentroidUpdateAll(BegAtm);

    /**** Initialization ****/

    k = VG_LOGP_CRIPPEN;

    switch (Par -> Com) {
    case MS_LIPBROTO:
      k = VG_LOGP_BROTO;

    case MS_LIPCRIPPEN:
      if (!GLOBSW_OPENGL) FindRing(InizAtm, FALSE);
      PrintProg(MSG_DCD_ASSATMTYPES);
      if ((Contrib = LogPInit(InizAtm, MolAtm, NULL, k)) != NULL) {
        if (!GLOBSW_STDOUT) {
          LogPCalc(InizAtm, &EneMax, Contrib, Par -> Flags);
          CatPrintf(stdout, MSG_DCD_STARTLIPOLE, EneMax);
        }
      } else Ret = FALSE;
      break;

    case MS_PSA:
    case MS_SURFACE:
    case MS_SURFDIA:
      if ((VdwRad = AssignVdwRad(InizAtm, MolAtm, Prefs.SAS_PROBERAD, &MaxRad))) {
        if ((SfUn = CalcSfUn(Prefs.SAS_POINTS, MaxRad, &Nu))) {
          if (!GLOBSW_STDOUT) {
            EneMin = CalcSrfArea(InizAtm, VdwRad, SfUn, Nu, Par, MaxRad, NULL, Par -> Flags);
            switch (Par -> Com) {
            case MS_SURFACE:
              k = MSG_DCD_STARTAREA;
              break;
            case MS_SURFDIA:
              k      = MSG_DCD_STARTSURFDIA;
              EneMin = SQR(EneMin / PI);
              break;
            case MS_PSA:
              k = MSG_DCD_STARTPSA;
              break;
            } /* End of switch */
            CatPrintf(stdout, k, EneMin);
          }
        } else Ret = FALSE;
      } else Ret = FALSE;
      break;

    case MS_VLOGP:
#ifdef VLOGP
      if ((VdwRad = AssignVdwRad(InizAtm, MolAtm, VLOGP_PROBE_RAD, &MaxRad))) {
        if ((SfUn = CalcSfUn(VLOGP_DOT_DENSITY, MaxRad, &Nu))) {
          if (!GLOBSW_OPENGL) FindRing(InizAtm, FALSE);
          PrintProg(MSG_DCD_ASSATMTYPES);
          if ((Contrib = LogPInit(InizAtm, MolAtm, NULL, VG_LOGP_BROTO)) != NULL) {
            if (!GLOBSW_STDOUT) {
              CatPrintf(stdout, MSG_DCD_STARTVLOGP,
                        CalcVLogP(InizAtm, VdwRad, MaxRad, SfUn,
                                  Nu, Contrib, Par -> Flags));
            }
          } else Ret = FALSE;
        } else Ret = FALSE;
      } else Ret = FALSE;
#else
      Ret = PrintVlogPErr();
#endif
      break;

    case MS_VOLUME:
    case MS_VOLDIA:
        if ((VdwRad = AssignVdwRad(InizAtm, MolAtm, 0.0, &MaxRad))) {
          CalcBox(InizAtm, VdwRad, &BoxMin, &BoxMax);
          if (!GLOBSW_STDOUT) {
            EneMin = CalcVolume(InizAtm, VdwRad, Prefs.VOL_DEN, &BoxMin,
                                &BoxMax, Par -> Flags);
            if (Par -> Com == MS_VOLUME)
              CatPrintf(stdout, MSG_DCD_STARTVOL, EneMin);
            else
              CatPrintf(stdout, MSG_DCD_STARTVOLDIA, pow(6.0 * EneMin / PI, 1.0 / 3.0));
          }
        } else Ret = FALSE;
      break;

    case MS_DIPOLE:
      if (!ChkCharges(InizAtm, MolAtm, NULL))
        Ret = CatErr(MSG_ERR_DCD_UNCHARGED);
      break;

    case MS_ILM:
      if (RenameWat(InizAtm)) {
        IlmCutOff = IlmClacCutOff(InizAtm);
        IlmCutOff *= IlmCutOff;
      } else Ret = CatErr(MSG_ERR_SURFACE_ILM);
      break;

    case MS_RMSD:
      if ((PrecBegAtm = CloneAtms(BegAtm, TotalAtm)) == NULL)
        Ret = FALSE;
      break;
    } /* End of switch */

    if (Ret) {

      /**** Select atoms ****/

      for(k = 0; k < (VG_ULONG)Par -> Num; ++k) {
        for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
          if (Atm -> Num == Par -> Atm[k].Num) {
            Frm[k] = Atm;
            break;
          }
        } /* End of atom loop */
      } /* End of selection loop */

      /**** Make the calculations ****/

      ProcFrmTxt = GetStr(MSG_DCD_PROCFRAME);

#ifdef WIN32
      if (GLOBSW_OPENGL)
        ProgBarInit(GetStr(MSG_PRGBAR_TITLE), Info -> Frames);
      else
        YPos = wherey();
#else
      YPos = 0;
#endif

      Ene = Iniz;
      for(k = 1;(Ret) &&
                (k <= Info -> Frames) &&
                (Ret = TrjReadFrm(Info, k)); ++k) {
        if (UpdateCentr) CentroidUpdateAll(BegAtm);

#ifdef WIN32
        if (GLOBSW_OPENGL) GL_DrawScene(&ViewPrefs, TRUE);
#endif


        switch(Par -> Com) {
        case MS_DISTANCE:
          *Ene = SQR(Quad(Frm[0] -> x - Frm[1] -> x) + Quad(Frm[0] -> y - Frm[1] -> y) +
                     Quad(Frm[0] -> z - Frm[1] -> z));
          break;

        case MS_LIPBROTO:
        case MS_LIPCRIPPEN:
          LogPCalc(InizAtm, Ene, Contrib, Par -> Flags);
          break;

        case MS_ANGLE:
          *Ene = NormAnglef(BondAnglef(Frm[0], Frm[1], Frm[2]));
          break;

        case MS_TORSION:
          *Ene = (float)NormAngle(Torsion(Frm[0], Frm[1], Frm[2], Frm[3]));
          break;

        case MS_PLANEANG:
          *Ene = (float)NormAngle(PlaneAngAtm(Frm) * RAD_TO_DEG);
          break;

        case MS_RMSD:
          Ret = CalcRmsd(Ene, &PrecBegAtm, &BegAtm, Par -> Flags & SRF_FLG_SELONLY);
          break;

#ifdef VLOGP
        case MS_VLOGP:
          *Ene = (float)CalcVLogP(InizAtm, VdwRad, MaxRad, SfUn, Nu,
                                  Contrib, Par -> Flags);
          break;
#endif
        case MS_VOLUME:
        case MS_VOLDIA:
          CalcBox(InizAtm, VdwRad, &BoxMin, &BoxMax);
          *Ene = (float)CalcVolume(InizAtm, VdwRad, Prefs.VOL_DEN, &BoxMin,
                                   &BoxMax, Par -> Flags);
          if (Par -> Com == MS_VOLDIA)
            *Ene = pow(6.0 * *Ene / PI, 1.0 / 3.0);
          break;

        case MS_DIPOLE:
          *Ene = (float)CalcDipole(InizAtm, Par -> Flags);
          break;

        case MS_SURFACE:
        case MS_SURFDIA:
          *Ene = (float)CalcSrfArea(InizAtm, VdwRad, SfUn, Nu, Par, MaxRad, NULL, Par -> Flags);
          if (Par -> Com == MS_SURFDIA) *Ene = SQR(*Ene / PI);
          break;

        case MS_PSA:
          CalcSrfArea(InizAtm, VdwRad, SfUn, Nu, Par, MaxRad, Ene, Par -> Flags);
          break;

        case MS_ILM:
          *Ene = CalcIlm(InizAtm, IlmCutOff, Par -> Flags);
          break;
        } /* End of switch */

        /**** Statistical values ****/

        Media  += *Ene;
        Media2 += Quad(*Ene);
        if (k == 1) {
          EneMin = *Ene;
          EneMax = *Ene;
        } else {
          if (*Ene < EneMin) {
            EneMin = *Ene;
            FrmMin = k;
          }
          if (*Ene > EneMax) {
            EneMax = *Ene;
            FrmMax = k;
          }
        }


        /**** Progress informations ****/

        if (!GLOBSW_STDOUT) {
#ifdef WIN32
          if (GLOBSW_OPENGL) {
            sprintf(Buf, ProcFrmTxt, k, Info -> Frames, *Ene);
            Ret = ProgBarUpd(Buf, k);
          } else {
            gotoxy(1, YPos);
            clreol();
            cprintf(ProcFrmTxt, k, Info -> Frames, *Ene);
          }
#else
          printf(ProcFrmTxt, k, Info -> Frames, *Ene);
          printf("          \n\033M");
#endif
        }
        ++Ene;
      } /* End of frame loop */


      if (!GLOBSW_STDOUT) {
#ifdef WIN32
        gotoxy(1, YPos);
        clreol();
#else
        printf("%40s\n\033M", " ");
#endif

        if (Ret)
          CatPrintf(stdout, MSG_DCD_RESULTS,
                EneMin, FrmMin, EneMax, FrmMax,
                            Media / Info -> Frames,
                            SQR((Media2 - (Quad(Media) / Info -> Frames)) / (Info -> Frames - 1)));
#ifdef WIN32
        if (GLOBSW_OPENGL) {
          ProgBarClose();
          LocPrintf(stdout, "\n");
        }
#endif
      }
    }
  }

  if (Contrib)    FREE(Contrib);
  if (PrecBegAtm) FreeAtm(PrecBegAtm);
  if (SfUn)       FREE(SfUn);
  if (VdwRad)     FREE(VdwRad);

  if (!Ret) {
    FREE(Iniz);
    Iniz = NULL;
  }

  return Iniz;
}


/**** Calculate the energy properties ****/

void TrjCalcEne(TRJINFO *Info)
{
  float        *Ene    = Info -> Energy;
  VG_ULONG     Frm     = 1;
  VG_ULONG     TrjFrm  = 1;

  if (Ene) {
    Info -> ConvStep = 0;
    if ((Info -> Frames) && (Info -> Frames != Info -> EneFrames)) {
      if (!(Info -> EneFrames % Info -> Frames))
        Info -> ConvStep = (VG_ULONG)(Info -> EneFrames / Info -> Frames);
      else if (!((Info -> EneFrames - 1) % (Info -> Frames - 1)))
        Info -> ConvStep = (VG_ULONG)((Info -> EneFrames - 1) / (Info -> Frames - 1));
      else return;
    } else Info -> ConvStep = 1;

    while(Frm <= Info -> EneFrames) {
      if (Frm == 1) {
        Info -> EneMax = *Ene;
        Info -> EneMin = *Ene;
        Info -> FrmMax = 1;
        Info -> FrmMin = 1;
      } else {
        if (*Ene > Info -> EneMax) {
          Info -> EneMax = *Ene;
          Info -> FrmMax = TrjFrm;
        }
        if (*Ene < Info -> EneMin) {
          Info -> EneMin = *Ene;
          Info -> FrmMin = TrjFrm;
        }
      }
      Ene += Info -> ConvStep;
      Frm += Info -> ConvStep;
      ++TrjFrm;
    } /* End of while */
  }
}


/**** Open a trajectory file ****/

void TrjClose(TRJINFO *Info)
{
  switch(Info -> Format) {

  case FTRJ_ARC:       /* Accelrys archive */
    CarClose(Info);
    break;

  case FTRJ_BIODOCK:   /* BioDock Output */
    BioDockClose(Info);
    break;

  case FTRJ_CSR:       /* Quanta CSR */
    CsrClose(Info);
    break;

  case FTRJ_DCD:       /* CHARMm DCD */
    DCDClose(Info);
    break;

  case FTRJ_ESCHERNG:  /* ESCHER Next Generation */
    EscherClose(Info);
    break;

  case FTRJ_PDB:       /* PDB multi-model */
    PDBClose(Info);
    break;

  case FTRJ_XTC:      /* Gromacs XTC */
    XtcClose(Info);
    break;
  } /* End of switch */

  if (Info -> Energy) FREE(Info -> Energy);

#ifdef WIN32
  ZeroMemory(Info, sizeof(TRJINFO));
#else
  memset(Info, 0, sizeof(TRJINFO));
#endif
}


/**** Detect the trajectory format ****/

static VG_UWORD TrjFindFormat(char *FileName)
{
  char          Buf[256];
  FILE          *FH;
  VG_LONG       Data;
  VG_ULONG      k;

  VG_UWORD      Format = FTRJ_NONE;

  /**** Binary file ****/

  if ((FH = fopen(FileName, "rb")) != NULL) {
    for(k = 0; (k < 2) && (fread(&Data, sizeof(VG_LONG), 1, FH) == 1); ++k) {
#ifdef LITTLE_ENDIAN
      Swap(&Data);
#endif
      switch(k) {
      case 0:   /* First long word */
        if (Data == 1995) Format = FTRJ_XTC;
        break;

      case 1:   /* Second long word */
        switch(Data) {
        case 0x434f5244:
        case 0x44524f43:
          Format = FTRJ_DCD;
          break;

        case 0x56333362:        /* V33b */
        case 0x62333356:
        case 0x56333367:        /* V33g */
        case 0x67333356:
        case 0x5633333f:        /* V33? */
        case 0x3f333356:
          Format = FTRJ_CSR;
          break;

        } /* End of switch */
        break;
      } /* End of switch */
    } /* End of for */
    fclose(FH);
  }

  /**** Ascii file ****/

  if ((Format == FTRJ_NONE) && ((FH = fopen(FileName, "r")) != NULL))  {
    while((Format == FTRJ_NONE) && (fgets(Buf, 255, FH))) {
      if (!strncmp(Buf, "MODEL        1", 14))
        Format = FTRJ_PDB;
      else if (!strncmp("!BIOSYM archive", Buf, 15))
        Format = FTRJ_ARC;
    } /* End of while */
    fclose(FH);
  }

  return Format;
}


/**** Open a trajectory file ****/

VG_BOOL TrjOpen(char *FileName, TRJINFO *Info, VG_UWORD Format)
{
  VG_BOOL       Ret;

  if ((!TotalAtm) && (Format != FTRJ_BIODOCK) &&
      (Format != FTRJ_ESCHERNG)) {
    CatErr(MSG_ERR_DCD_NOMOL);
    return FALSE;
  }

  memset(Info, 0, sizeof(TRJINFO));

  if ((Format == FTRJ_NONE) &&
      ((Format = TrjFindFormat(FileName)) == FTRJ_NONE))
    return CatErr(MSG_ERR_DCD_UNKTRAJ);

  switch(Format) {
  case FTRJ_ARC:
    Ret = CarOpen(FileName, Info);
    break;

  case FTRJ_BIODOCK:
    Ret = BioDockOpen(FileName, Info);
    break;

  case FTRJ_CSR:
    Ret = CsrOpen(FileName, Info);
    break;

  case FTRJ_DCD:
    Ret = DCDOpen(FileName, Info);
    break;

  case FTRJ_ESCHERNG:
    Ret = EscherOpen(FileName, Info);
    break;

  case FTRJ_PDB:
    Ret = PDBOpen(FileName, Info);
    break;

  case FTRJ_XTC:
    Ret = XtcOpen(FileName, Info);
    break;
  } /* End of switch */

  if (Ret) {
    Info -> Format = Format;
    strcpy(Info -> FileName, FileName);
    TrjCalcEne(Info);
  }

  return Ret;
}


/**** Save the data ****/

VG_BOOL TrjPlotSave(char *FileName, VG_ULONG Format, float *Data, VG_ULONG Tot)
{
  char          *OutMode;
  FILE          *FH;
  float         Ene, x;
  VG_ULONG      k;

  VG_BOOL       Ret = TRUE;


  /**** Set the write mode ****/

  switch (Format) {
  case FORM_PLOT_CSV:
  case FORM_PLOT_QUANTA:
    OutMode = "w";
    break;

  case FORM_PLOT_BIN:
    OutMode = "wb";
    break;

  default :
      return CatErr(MSG_ERR_DCD_UNSUPPOUT);
  }

  if (*FileName)
    FH = fopen(FileName, OutMode);
  else
    FH = stdout;

  if (FH != NULL) {
    if (!GLOBSW_OPENGL) PrintProg(MSG_ECD_SAVPLOT);
      for(k = 1; k <= Tot; ++k) {
        switch(Format) {
        case FORM_PLOT_CSV:           /* Comma separted values */
          if (fprintf(FH, "%5d; %14.6f\n", k, *Data) < 0)
            Ret = PrintDosErr();
          break;

        case FORM_PLOT_QUANTA:         /* Quanta plot          */
          if (fprintf(FH, "%5d.000000,%14.6f\n", k, *Data) < 0)
            Ret = PrintDosErr();
          break;

        case FORM_PLOT_BIN:            /* Binary Plot          */
          x = (float)k;
          Ene = *Data;
#ifdef LITTLE_ENDIAN
          Swap(&x);
          Swap(&Ene);
#endif
          if ((fwrite(&x   , sizeof(float), 1, FH) != 1) ||
              (fwrite(&Ene , sizeof(float), 1, FH) != 1))
            Ret = PrintDosErr();
          break;
        } /* End of switch */
        ++Data;
      } /* End of for */
    fclose(FH);
  } else Ret = PrintDosErr();

  return Ret;
}


/**** Read a specified frame ****/

VG_BOOL TrjReadFrm(TRJINFO *Info, VG_ULONG Frame)
{
  VG_BOOL       Ret = FALSE;

  if ((Info -> FH) &&
      (Frame <= Info -> Frames) &&
      (Ret = TrjSeekFrm(Info, Frame - 1, SEEK_SET))) {
    switch(Info -> Format) {
    case FTRJ_ARC:      /* Accelrys archive */
      Ret = CarReadFrm(Info, BegAtm);
      break;

    case FTRJ_BIODOCK:  /* BioDock output */
      Ret = BioDockReadFrm(Info);
      break;

    case FTRJ_CSR:      /* Quanta CSR */
      Ret = CsrReadFrm(Info, BegAtm);
      break;

    case FTRJ_DCD:      /* CHARMm DCD */
      Ret = DCDReadFrm(Info -> FH, BegAtm);
      break;

    case FTRJ_ESCHERNG: /* ESCHER Next Generation */
      Ret = EscherReadFrm(Info);
      break;

    case FTRJ_PDB:      /* PDB multi-model */
      Ret = PDBReadFrm(Info, BegAtm);
      break;

    case FTRJ_XTC:      /* Gromacs XTC */
      Ret = XtcReadFrm(Info, BegAtm);
      break;

    default:
      Ret = FALSE;
    } /* End of switch */
    if (Ret) ++Info -> FrmCur;
  }

  return Ret;
}


/**** Seek a trajectory frame ****/

VG_BOOL TrjSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  VG_BOOL       Ret;

  if (CountAtms(BegAtm, FALSE)!= Info -> MolAtm)
    return CatErr(MSG_ERR_DCD_UNKTRAJ);

  switch(Info -> Format) {
  case FTRJ_ARC:        /* Accelrys archive */
    Ret = CarSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_BIODOCK:    /* BioDock output */
    Ret = BioDockSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_CSR:        /* Quanta CSR*/
    Ret = CsrSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_DCD:        /* CHARMm DCD */
    Ret = DCDSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_ESCHERNG:   /* ESCHER NG solutions */
    Ret = EscherSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_PDB:        /* PDB multi-model */
    Ret = PDBSeekFrm(Info, Frames, Mode);
    break;

  case FTRJ_XTC:        /* Gromacs XTC */
    Ret = XtcSeekFrm(Info, Frames, Mode);
    break;

  default:
    Ret = FALSE;
  } /* End of switch */

  if (Ret) {
    switch(Mode) {
    case SEEK_CUR:
      Info -> FrmCur += Frames;
      break;

    case SEEK_END:
      Info -> FrmCur = Info -> Frames - Frames;
      break;

    case SEEK_SET:
      Info -> FrmCur = Frames;
      break;
    } /* End of switch */
  }

  return Ret;
}
