
/*************************************************
****           VEGA - Add hydrogens           ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * The atom type routines are based on the original code included into
 * Babel 1.6 (Copyright 1992-96, W. Patrick Walters, Matthew T. Stahl).
 * The powerful ATDL engine can't be used because the atom valences are
 * incomplete due to the missing hydrogens.
 * The "Nim" atom type was introduted to recognize the nitrogens in five
 * member aromatic rings and thus to add correctly the hydrogens to these
 * rings.
 * Some protein structures have got unallowed geometries and so the atom types
 * can't be correctly attributed. A special template system was introduced to
 * check the hybridization of each protein atom and to correct the wrong atom
 * types fond in the normal way.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#ifdef __VG_OPENGL
#  ifdef WIN32
#    include <windows.h>
#  endif
#  include "gl_global.h"
#endif

#include "addhyd.h"
#include "bond.h"
#include "count.h"
#include "hybtem.h"

/**** Local variables ****/

static VG_BYTE          *Redo;
static VG_ULONG         AddedH;


/**** Add a single hydrogen ****/

ATOMO *AddH(ATOMO *Pos, ATOMO *Dest, VG_ULONG *TotAtm, float Len, VG_LONG Geo,
            VG_LONG Flags)
{
  ATOMO         *Atm;

  if ((Atm = InsertAtm(Pos, TotAtm)) != NULL) {
    Atm -> Elem.S = VG_ELEM_H;
    Atm -> Rad    = VG_ADDH_HVDWRAD;

    if (Dest -> Elem.S == VG_ELEM_C) Atm -> Pot.DL = VG_ADDH_FF_HC;
    else Atm -> Pot.DL = VG_ADDH_FF_H;

    BndMake(Atm, Dest, 1);

    /**** Set the atom name ****/

    if (Flags & VG_ADDH_FLG_IUPACNAMES)
      AddhSetAtmNameProt(Atm, Dest);
    else
      SetAtmName(Atm);

#ifdef __VG_OPENGL
    if (GLOBSW_OPENGL) GL_ColorAtm(Atm);
#endif

    switch(Geo) {
    case VG_ADDH_GEO_SP:    /* Sp */
      AtmSetGeoSp(Atm, Dest, Len);
      break;

    case VG_ADDH_GEO_SP2:   /* Sp2 */
      AtmSetGeoSp2(Atm, Dest, Len);
      break;

    case VG_ADDH_GEO_SP3:   /* Sp3 */
      AtmSetGeoSp3(Atm, Dest, Len);
      break;
    } /* End of switch */
    ++AddedH;
  }

  return Atm;
}


/**** Assign the atom types without ATDL ****/

VG_BOOL AddhAssignTypes(ATOMO *InizAtm, VG_ULONG *TotAtm, VG_LONG Flags)
{
  ATOMO         *Atm1, *Atm2;
  char          *TemFile, TemPath[VG_MAX_PATH];
  VG_BOOL       Flag;
  VG_BYTE       i;
  VG_OCHAR      CorPot;

  VG_LIST       *BegTem = NULL;

  if ((Redo = (VG_BYTE *)Alloca(*TotAtm)) == NULL)
    return FALSE;

  /**** If required, load the hybridization template ****/

  if (Flags & VG_ADDH_FLG_NA) TemFile = VG_HYB_TEM_NA;
  else if (Flags & VG_ADDH_FLG_PROT) TemFile = VG_HYB_TEM_PROT;
  else TemFile = NULL;

  if (TemFile) {
    GetPrgPath(TemPath, TRUE);
    strcat(TemPath, TemFile);
    if ((BegTem = HybTemLoad(TemPath)) == NULL)
      return FALSE;
  }

  /**** Fix generic atom types ****/

  for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
      Atm1 -> Pot.DL   = 0;
      Atm1 -> Pot.C[0] = Atm1 -> Elem.C[0];
      Atm1 -> Pot.C[1] = Atm1 -> Elem.C[1];
    }
  } /* End of for */

  AddhTagOrganic(InizAtm, Flags);

  /**** Phase 1 ****/

  AddhTypeH(InizAtm, Flags);

  /**** Phase 2 ****/

  AddhValence4(InizAtm, Flags);
  AddhValence3(InizAtm, Flags);
  AddhValence2(InizAtm, Flags);

  /**** Phase 3 ****/

  AddhValence1(InizAtm, Flags);

  /**** Phase 4 ****/

  AddhPhase4(InizAtm, Flags);

  /**** Phase 5 ****/

  for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) &&
        (Atm1 -> Pot.DL == VG_ADDH_FF_C2)) {
      Flag = FALSE;
      for(i = 0; i < Atm1 -> NSost; i++) {
        Atm2 = Atm1 -> Conn[i];
        if ((Atm2 -> Pot.DL != VG_ADDH_FF_C3) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_DC) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_HC) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_N3) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_N3p) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_N3) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_Pac) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_Sac) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_Sox) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_C1) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_S3) &&
            (Atm2 -> Pot.DL != VG_ADDH_FF_Cac)) {
          Flag = TRUE;
        }
      } /* End of for */
      if (!Flag) Atm1 -> Pot.DL = VG_ADDH_FF_C3;
    }
  } /* End of for */

  /**** Phase 6 ****/

  AddhPhase6(InizAtm, Flags);

  /**** Check for amides ****/

  for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
      if (Atm1 -> Pot.DL == VG_ADDH_FF_Npl) {
        for(i = 0; i < Atm1 -> NSost; i++) {
          Atm2 = Atm1 -> Conn[i];
          if ((Atm2 -> Pot.DL == VG_ADDH_FF_Cac) ||
              (Atm2 -> Pot.DL == VG_ADDH_FF_Sox) ||
              (Atm2 -> Pot.DL == VG_ADDH_FF_So2)) {
            Atm1 -> Pot.DL = VG_ADDH_FF_Nam;
            break;
          }

          if ((Atm2 -> Pot.DL == VG_ADDH_FF_C2) &&
              (AddhChkCarbonyl(Atm2) == 3)) {
            Atm1 -> Pot.DL = VG_ADDH_FF_Nam;
            break;
          }
        } /* End of for */
      }
    }
  } /* End of for */

  /**** Check the atom types ****/

  if (BegTem) {
    for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
      if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
        if (HybTemAtmCheck(Atm1, BegTem, &CorPot)) {

          /**** The atom has unusual geometry ****/

          CatPrintf(stdout, MSG_ADDH_WARN, Atm2Str(TemPath, Atm1));
          Atm1 -> Pot.DL = CorPot.DL;
        }
      }
    } /* End of for */
    ListClear(BegTem);
  }
  FREE(Redo);

  return TRUE;
}


/**** Check for carbonyl ****/

VG_LONG AddhChkCarbonyl(ATOMO *Atm1)
{
  ATOMO         *Atm2;
  VG_BYTE       i;

  for (i = 0; i < Atm1 -> NSost; ++i) {
    Atm2 = Atm1 -> Conn[i];
    if ((Atm2 -> Pot.DL == VG_ADDH_FF_O2) || (Atm2 -> Pot.DL == VG_ADDH_FF_S2))
      return 3;
  } /* End of for */

  return 2;
}


/**** Fix the protein atom names ****/

void AddhFixAtmNameProt(ATOMO *Atm1, VG_LONG Flags)
{
  VG_UBYTE       *Ptr;

  while(Atm1) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) &&
        (Atm1 -> Elem.S == VG_ELEM_H) && (CountBndHyd(Atm1 -> Conn[0]) == 1)) {
      for(Ptr = Atm1 -> Name.C + 3; Ptr != (VG_UBYTE *)Atm1 -> Name.C; --Ptr) {
        if ((*Ptr >= '0') && (*Ptr <= '9')) {
          *Ptr = 0;
          break;
        }
      } /* End of for */
    }
    Atm1 = Atm1 -> Ptr;
  } /* End of while */
}


/**** Phase 4 of potential assignment ****/

void AddhPhase4(ATOMO *Atm1, VG_LONG Flags)
{
  ATOMO         *Atm2;
  float         Len;
  VG_BOOL       Flag;
  VG_BYTE       i;

  VG_BYTE       *Ptr = Redo;


  while(Atm1) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
      switch(*Ptr) {
      case 1:
        for (i = 0; i < Atm1 -> NSost; ++i) {
          Atm2 = Atm1 -> Conn[i];
          Len = QuadDist(Atm1, Atm2);
          Len = SQR(Len);
    if ((Len <= V2_C2_C_CUTOFF) && (*Atm2 -> Pot.C == 'C'))
            Atm1 -> Pot.DL = VG_ADDH_FF_C2;
    else if ((Len <= V2_C2_N_CUTOFF) && (*Atm2 -> Pot.C == 'N'))
            Atm1 -> Pot.DL = VG_ADDH_FF_C2;
        } /* End of for */

        for (i = 0; i < Atm1 -> NSost; ++i) {
          Atm2 = Atm1 -> Conn[i];
          Len = QuadDist(Atm1, Atm2);
          Len = SQR(Len);
    if ((Len > V2_C3_C_CUTOFF) && (*Atm2 -> Pot.C == 'C'))
      Atm1 -> Pot.DL = VG_ADDH_FF_C3;
    else if ((Len > V2_C3_N_CUTOFF) && (*Atm2 -> Pot.C == 'N'))
            Atm1 -> Pot.DL = VG_ADDH_FF_C3;
    else if ((Len > V2_C3_O_CUTOFF) && (*Atm2 -> Pot.C == 'O'))
      Atm1 -> Pot.DL = VG_ADDH_FF_C3;
        } /* End of for */
        break;

      case 2:
        for (i = 0; i < Atm1 -> NSost; ++i) {
          Atm2 = Atm1 -> Conn[i];
          Len = QuadDist(Atm1, Atm2);
          Len = SQR(Len);
    if ((Len <= V2_N2_C_CUTOFF) && (*Atm2 -> Pot.C == 'C')) {
            if (Atm2 -> Ring < 0) {
              switch(Atm1 -> Ring) {
              case -5:
                Atm1 -> Pot.DL = VG_ADDH_FF_Nim;
                break;

              case -6:
                Atm1 -> Pot.DL = VG_ADDH_FF_Npy;
                break;

              default:
                Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
              } /* End of switch */
            } else Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
    } else if ((Len <= V2_N2_N_CUTOFF) && (*Atm2 -> Pot.C == 'N'))
            Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
        } /* End of for */
        break;

      case 3:
  Flag = FALSE;
        for (i = 0; i < Atm1 -> NSost; ++i) {
          Atm2 = Atm1 -> Conn[i];
          Len = QuadDist(Atm1, Atm2);
          Len = SQR(Len);
    if ((Len <= V2_C2_C_CUTOFF) && (*Atm2 -> Pot.C == 'C')) {
            Atm1 -> Pot.DL = VG_ADDH_FF_C2;
      Flag = TRUE;
    } else if ((Len <= V2_C2_N_CUTOFF) && (*Atm2 -> Pot.C == 'N')) {
            Atm1 -> Pot.DL = VG_ADDH_FF_C2;
      Flag = TRUE;
          }
  } /* End of for */

  if (!Flag) {
          for (i = 0; i < Atm1 -> NSost; ++i) {
            Atm2 = Atm1 -> Conn[i];
            Len = QuadDist(Atm1, Atm2);
            Len = SQR(Len);
      if ((Len > V2_C3_C_CUTOFF) && (*Atm2 -> Pot.C == 'C')) {
              Atm1 -> Pot.DL = VG_ADDH_FF_C3;
        Flag = TRUE;
      } else if ((Len > V2_C3_N_CUTOFF) && (*Atm2 -> Pot.C == 'N')) {
              Atm1 -> Pot.DL = VG_ADDH_FF_C3;
        Flag = TRUE;
            } else if ((Len > V2_C3_O_CUTOFF) && (*Atm2 -> Pot.C == 'O')) {
              Atm1 -> Pot.DL = VG_ADDH_FF_C3;
        Flag = TRUE;
            } else if (!Flag) {
              if ((Len > GEN_C3_C_CUTOFF) && (*Atm2 -> Pot.C == 'C')) {
                Atm1 -> Pot.DL = VG_ADDH_FF_C3;
          Flag = TRUE;
              }
      }
          } /* End of for */
        }
        break;
      } /* End of switch */
    }
    Atm1 = Atm1 -> Ptr;
    ++Ptr;
  } /* End of while */
}


/**** Phase 6 of potential assignment ****/

void AddhPhase6(ATOMO *Atm1, VG_LONG Flags)
{
  ATOMO         *Atm2, *Atm3;
  VG_BOOL       Protonated;
  VG_BYTE       j, k;
  VG_LONG       m;

  while(Atm1) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
      Protonated = TRUE;
      if (Atm1 -> Pot.DL == VG_ADDH_FF_N3) {
        for(j = 0; j < Atm1 -> NSost; j++) {
          Atm2 = Atm1 -> Conn[j];
    if ((Atm1 -> NSost == 2) &&
              ((Atm2 -> Pot.DL == VG_ADDH_FF_Car) ||
               (Atm2 -> Pot.DL == VG_ADDH_FF_C2) ||
               (Atm2 -> Pot.DL == VG_ADDH_FF_Sox) ||
               (Atm2 -> Pot.DL == VG_ADDH_FF_Sac) ||
               (Atm2 -> Pot.DL == VG_ADDH_FF_Pac) ||
               (Atm2 -> Pot.DL == VG_ADDH_FF_So2))) {
      Protonated = FALSE;
            if (Atm1 -> Ring == -5)
              Atm1 -> Pot.DL = VG_ADDH_FF_Nim;
            else
              Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
      break;
    }

          if ((Atm2 -> Pot.DL != VG_ADDH_FF_C3) && (Atm2 -> Elem.S != VG_ELEM_H) &&
              (Atm2 -> Elem.S != VG_ELEM_D) && (Atm2 -> Elem.S != VG_ELEM_T))
      Protonated = FALSE;
        } /* End of for */

        if (Protonated) Atm1 -> Pot.DL = VG_ADDH_FF_N3p;

#if 0
      } else if (Atm1 -> Pot.DL == VG_ADDH_FF_Npy) {

        /**** Look for guanine ring ****/

        m = 0;
        for(j = 0; j < Atm1 -> NSost; j++) {
          Atm2 = Atm1 -> Conn[j];
          if (Atm2 -> Pot.DL == VG_ADDH_FF_C2) {
            for(k = 0; k < Atm2 -> NSost; k++) {
              Atm3 = Atm2 -> Conn[k];
              if (((Atm3 -> Pot.DL == VG_ADDH_FF_O2) ||
                   (Atm3 -> Pot.DL == VG_ADDH_FF_Npl)) &&
                  (!Atm3 -> Ring)) {
                ++m;
              }
            } /* End of for */
          }
        } /* End of for */
        if (m == 2) Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
#endif
      } else if (Atm1 -> Pot.DL == VG_ADDH_FF_C2) {

        /**** Look for guanadinium nitrogens ****/

  m = 0;
        for(j = 0; j < Atm1 -> NSost; ++j) {
          Atm2 = Atm1 -> Conn[j];
          if ((Atm2 -> Pot.DL == VG_ADDH_FF_Npl) ||
              (Atm2 -> Pot.DL == VG_ADDH_FF_N2) ||
              (Atm2 -> Pot.DL == VG_ADDH_FF_Ngp))
      m++;
  } /* End of for */

  if (m == 3) {
          Atm1 -> Pot.DL = VG_ADDH_FF_Cp;
    for(j = 0; j < Atm1 -> NSost; ++j) {
            Atm1 -> Conn[j] -> Pot.DL = VG_ADDH_FF_Ngp;
    } /* End of for */
  }
      } else if (Atm1 -> Pot.DL == VG_ADDH_FF_Cac) {
        for(j = 0; j < Atm1 -> NSost; ++j) {
          Atm2 = Atm1 -> Conn[j];
          if ((*Atm2 -> Pot.C == 'O') && (CountBndHeavyAtms(Atm2) == 1))
            Atm2 -> Pot.DL = VG_ADDH_FF_Om;
  } /* End of for */
      }
    }
Esci:
    Atm1 = Atm1 -> Ptr;
  } /* End of while */
}


/**** Place the hydrogens ****/

void AddhPlaceHyd(ATOMO *InizAtm, VG_ULONG *TotAtm, VG_LONG Flags)
{
  ATOMO         *Atm1, *Atm2, *Atm3, *InsertPos;
  float         Dist;
  VG_BYTE       NSost, j, k;
  VG_LONG       Geo;

  VG_BYTE       PrecChainID = 0;
  VG_LONG       PrecResName = 0;
  VG_LONG       PrecResSeq  = 0;


  for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) &&
        (Atm1 -> Elem.S != VG_ELEM_H)) {
      if (Flags & VG_ADDH_FLG_HPOSENDRES) {
        if ((PrecResName != Atm1 -> ResName.L) ||
            (PrecResSeq  != Atm1 -> ResSeq.L) ||
            (PrecChainID != Atm1 -> ChainID)) {
          PrecResName = Atm1 -> ResName.L;
          PrecResSeq  = Atm1 -> ResSeq.L;
          PrecChainID = Atm1 -> ChainID;
          if (Atm1 -> Ptr)
            FindResByAtm(Atm1, &InsertPos, NULL);
          else
            InsertPos = Atm1;
        }
      } else InsertPos = Atm1;

      switch(Atm1 -> Pot.DL) {
      case VG_ADDH_FF_C:
      case VG_ADDH_FF_C3:
        NSost = 4;
        Dist  = VG_ADDH_DIST_C_SP3;
        Geo   = VG_ADDH_GEO_SP3;
        break;

      case VG_ADDH_FF_C2:
      case VG_ADDH_FF_Car:
        NSost = 3;
        Dist  = VG_ADDH_DIST_C_SP2;
        Geo   = VG_ADDH_GEO_SP2;
        break;

      case VG_ADDH_FF_C1:
        NSost = 2;
        Dist  = VG_ADDH_DIST_C_SP;
        Geo   = VG_ADDH_GEO_SP;
        break;

      case VG_ADDH_FF_N3:
        NSost = 3;
        Dist  = VG_ADDH_DIST_N_SP3;
        Geo   = VG_ADDH_GEO_SP3;
        break;

      case VG_ADDH_FF_N3p:
        NSost = 4;
        Dist  = VG_ADDH_DIST_N_SP3;
        Geo   = VG_ADDH_GEO_SP3;
        break;

      case VG_ADDH_FF_Nam:
      case VG_ADDH_FF_Ngp:
      case VG_ADDH_FF_Npl:
        NSost = 3;
        Dist  = VG_ADDH_DIST_N_SP2;
        Geo   = VG_ADDH_GEO_SP2;
        break;

      case VG_ADDH_FF_Nim:
        NSost = 3;
        Dist  = VG_ADDH_DIST_N_SP2;
        Geo   = VG_ADDH_GEO_SP2;
        for(j = 0; j < Atm1 -> NSost; ++j) {
          Atm2 = Atm1 -> Conn[j];

          /**** Isoxazole and isotiazole ****/

          if ((Atm2 -> Pot.DL == VG_ADDH_FF_O3) ||
              (Atm2 -> Pot.DL == VG_ADDH_FF_S3) ||

              /**** Pirazole ****/

              (((Atm2 -> Pot.DL == VG_ADDH_FF_Nim) ||
                (Atm2 -> Pot.DL == VG_ADDH_FF_Npl)) &&
               (Atm2 -> NSost == 3))) {
            NSost = 0;
            break;
          } else if (Atm2 -> Pot.DL == VG_ADDH_FF_C2) {
            for(k = 0; k < Atm2 -> NSost; ++k) {
              Atm3 = Atm2 -> Conn[k];

              /**** Oxazole and tiazole ****/

              if ((Atm3 -> Pot.DL == VG_ADDH_FF_O3) ||
                  (Atm3 -> Pot.DL == VG_ADDH_FF_S3) ||

                  /**** Imidazole ****/

                  (((Atm3 -> Pot.DL == VG_ADDH_FF_Nim) ||
                    (Atm3 -> Pot.DL == VG_ADDH_FF_Npl)) &&
                   (Atm3 -> NSost == 3))) {
                NSost = 0;
                break;
              }
            } /* End of for */
          }
        } /* End of for */
        break;

      case VG_ADDH_FF_O:
      case VG_ADDH_FF_O3:
        NSost = 2;
        Dist  = VG_ADDH_DIST_O_SP3;
        Geo   = VG_ADDH_GEO_SP3;
        break;

      case VG_ADDH_FF_S:
      case VG_ADDH_FF_S3:
        NSost = 2;
        Dist  = VG_ADDH_DIST_S_SP3;
        Geo   = VG_ADDH_GEO_SP3;
        break;

      default:
        NSost = 0;
      } /* End of switch */

      if (NSost) {
        for(k = Atm1 -> NSost; k < NSost; ++k) {
          AddH(InsertPos, Atm1, TotAtm, Dist, Geo, Flags);
          InsertPos = InsertPos -> Ptr;
        } /* End of for */
      }
    }

    if (Flags & VG_ADDH_FLG_HPOSENDRES) {
      if ((Atm1 -> Flags & VG_ATMF_SEGEND) ||
          (Atm1 -> Flags & VG_ATMF_MOLEND)) {
        PrecResName = Atm1 -> ResName.L;
        PrecResSeq  = Atm1 -> ResSeq.L;
        PrecChainID = Atm1 -> ChainID;
        if (Atm1 -> Ptr)
          FindResByAtm(Atm1, &InsertPos, NULL);
        else
          InsertPos = Atm1;
      }
    }
  } /* End of for */

  if (Flags & VG_ADDH_FLG_IUPACNAMES) AddhFixAtmNameProt(InizAtm, Flags);
}


/**** Set the atom name for proteins ****/

void AddhSetAtmNameProt(ATOMO *Atm, ATOMO *Conn)
{
  VG_BYTE       k;
  VG_LONG       Pos;

  VG_UBYTE      Num = 1;

#if 0
  /**** Alpha carbon (CA) ****/

#ifdef LITTLE_ENDIAN
  if (Conn -> Name.L == 0x00004143) {
    Atm -> Name.L = 0x00004148;
#else
  if (Conn -> Name.L == 0x43410000) {
    Atm -> Name.L = 0x48410000;
#endif
    return;
  }

  /**** Amidic nitrogen ****/

  if (Conn -> Pot.DL == VG_ADDH_FF_Nam) {
#ifdef LITTLE_ENDIAN
    Atm -> Name.L = 0x00000048;
#else
    Atm -> Name.L = 0x48000000;
#endif
    return;
  }
#endif

  if (!Conn -> Name.C[3]) {
    Atm -> Name.L  = Conn -> Name.L;
    *Atm -> Name.C = *Atm -> Elem.C;
    if (!Atm -> Name.C[1]) Pos = 1;
    else if (!Atm -> Name.C[2]) Pos = 2;
      else Pos = 3;

    do {
      Atm -> Name.C[Pos] = '0' + Num;
      for(k = 0; k < Conn -> NSost; ++k) {
        if ((Conn -> Conn[k] != Atm) &&
            (Atm -> Name.L == Conn -> Conn[k] -> Name.L))
          break;
      } /* End of for */
      ++Num;
    } while(k != Conn -> NSost);

  } else SetAtmName(Atm);
}


/**** Tag the organic atoms ****/

void AddhTagOrganic(register ATOMO *Atm, VG_LONG Flags)
{
  while(Atm) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm -> Active)) {
      if ((Atm -> Elem.S == VG_ELEM_H) ||
          (Atm -> Elem.S == VG_ELEM_C) ||
          (Atm -> Elem.S == VG_ELEM_N) ||
          (Atm -> Elem.S == VG_ELEM_O) ||
          (Atm -> Elem.S == VG_ELEM_S) ||
          (Atm -> Elem.S == VG_ELEM_P))
        Atm -> Flags |= VG_ATMF_ORGANIC;
      else
        Atm -> Flags = Atm -> Flags &~ VG_ATMF_ORGANIC;
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Type the hydrogens ****/

void AddhTypeH(register ATOMO *Atm, VG_LONG Flags)
{
  while(Atm) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm -> Active)) {
      if ((Atm -> Elem.S == VG_ELEM_H) ||
          (Atm -> Elem.S == VG_ELEM_D) ||
          (Atm -> Elem.S == VG_ELEM_T)) {
        Atm -> Pot.DL = 0;
        *Atm -> Pot.C = *Atm -> Elem.C;
        if ((Atm -> NSost) && (Atm -> Conn[0] -> Elem.S == VG_ELEM_C))
          Atm -> Pot.C[1] = 'C';
      }
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Add hydrogens ****/

VG_ULONG AddHyd(ATOMO **InizAtm, VG_ULONG *TotAtm, VG_LONG Flags)
{
  AddedH = 0;

  PrintProg(MSG_ADDH_PROG);
  RemoveH(InizAtm, TotAtm, Flags & VG_ADDH_FLG_ACTONLY);
  FindRing(*InizAtm, TRUE);
  if ((AddhAssignTypes(*InizAtm, TotAtm, Flags))) {
    AddhPlaceHyd(*InizAtm, TotAtm, Flags);
    if (AddedH) {
      RenAtm(*InizAtm, 1);
    }
  }

  return AddedH;
}


/**** Valence 1 ****/

void AddhValence1(ATOMO *Atm1, VG_LONG Flags)
{
  ATOMO         *Atm2;
  float         Len;

  while(Atm1) {
    if ((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm1 -> Active)) {
      Atm2 = Atm1 -> Conn[0];
      if ((Atm1 -> NSost == 1) && (Atm1 -> Flags & VG_ATMF_ORGANIC)) {
        Len  = QuadDist(Atm1, Atm2);
        Len  = SQR(Len);
        switch(Atm1 -> Elem.S) {
        case VG_ELEM_C:
          if (Atm1 -> Pot.DL == VG_ADDH_FF_C) {
            if ((Atm2 -> Pot.DL == VG_ADDH_FF_C1) &&
                (Len <= V1_C1_C1_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_C1;
            else if ((*Atm2 -> Pot.C == 'C') && (Len <= V1_C2_C_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_C2;
      else Atm1 -> Pot.DL = VG_ADDH_FF_C3;
    }
    if (*Atm2 -> Pot.C == 'N') {
      if (Len <= V1_C2_N_CUTOFF)
              Atm1 -> Pot.DL = VG_ADDH_FF_C2;
            else
              Atm1 -> Pot.DL = VG_ADDH_FF_C3;
          }
          break;

        case VG_ELEM_N:
          if (Atm1 -> Pot.DL == VG_ADDH_FF_N) {
            if ((Atm2 -> Pot.DL == VG_ADDH_FF_C1) && (Len <= V1_N1_C1_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_N1;
      else if (((Atm2 -> Pot.DL == VG_ADDH_FF_C2) ||
                      (Atm2 -> Pot.DL == VG_ADDH_FF_C3)) &&
                     (Len > V1_N3_C_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_N3;
      else if (((Atm2 -> Pot.DL == VG_ADDH_FF_N3) ||
                      (Atm2 -> Pot.DL == VG_ADDH_FF_N3p)) &&
                     (Len > V1_N3_N3_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_N3;
            else if ((Atm2 -> Pot.DL == VG_ADDH_FF_Npl) &&
         (Len > V1_N3_N2_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_N3;
            else Atm1 -> Pot.DL = VG_ADDH_FF_Npl;
          }
    break;

        case VG_ELEM_O:
    if (Atm1 -> Pot.DL == VG_ADDH_FF_O) {
            if ((Atm2 -> Pot.DL == VG_ADDH_FF_Cac) ||
                (Atm2 -> Pot.DL == VG_ADDH_FF_Pac) ||
                (Atm2 -> Pot.DL == VG_ADDH_FF_Sac) ||
                (Atm2 -> Pot.DL == VG_ADDH_FF_Ntr))
              Atm1 -> Pot.DL = VG_ADDH_FF_Om;
            else if ((Atm2 -> Pot.DL == VG_ADDH_FF_Nox) ||
                     (Atm2 -> Pot.DL == VG_ADDH_FF_Pox) ||
                     (Atm2 -> Pot.DL == VG_ADDH_FF_Sox))
              Atm1 -> Pot.DL = VG_ADDH_FF_O2;
      else if ((Atm2 -> Pot.C[0] == 'C') && (Len <= V1_O2_C2_CUTOFF)) {
              Atm1 -> Pot.DL = VG_ADDH_FF_O2;
              Atm2 -> Pot.DL = VG_ADDH_FF_C2;
              Redo[Atm2 -> Num - 1] = 0;
            } else if ((Atm2 -> Pot.DL == VG_ADDH_FF_As) && (Len <= V1_O2_AS_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_O2;
            else
              Atm1 -> Pot.DL = VG_ADDH_FF_O3;
          }
          break;

        case VG_ELEM_S:
    if (*Atm1 -> Pot.C == 'S') {
            if (*Atm2 -> Pot.C == 'P')
              Atm1 -> Pot.DL = VG_ADDH_FF_S2;
            else if ((*Atm2 -> Pot.C == 'C') && (Len <= V1_S2_C2_CUTOFF)) {
              Atm1 -> Pot.DL = VG_ADDH_FF_S2;
              Atm2 -> Pot.DL = VG_ADDH_FF_C2;
              Redo[Atm1 -> Num - 1] = 0;
      } else if ((Atm2 -> Pot.DL == VG_ADDH_FF_As) && (Len <= V1_S2_AS_CUTOFF))
              Atm1 -> Pot.DL = VG_ADDH_FF_S2;
      else
              Atm1 -> Pot.DL = VG_ADDH_FF_S3;
          }
    break;
        } /* End of switch */
      }
    }
    Atm1 = Atm1 -> Ptr;
  } /* End of while */
}


/**** Valence 2 ****/

void AddhValence2(ATOMO *Atm, VG_LONG Flags)
{
  float         Ang;

  while(Atm) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm -> Active)) &&
        (Atm -> NSost == 2) && (Atm -> Flags & VG_ATMF_ORGANIC)) {
      Ang  = BondAnglef(Atm -> Conn[0], Atm, Atm -> Conn[1]);
      switch(Atm -> Elem.S) {
      case VG_ELEM_C:
  if (Atm -> Pot.DL == VG_ADDH_FF_C) {
    if (Ang < VG_ADDH_ANG_SP3_MAX) {
            Atm -> Pot.DL = VG_ADDH_FF_C3;
      Redo[Atm -> Num - 1] = 1;
    } else if (Ang < VG_ADDH_ANG_SP_MIN) {
            Atm -> Pot.DL = VG_ADDH_FF_C2;
            if (Ang < VG_ADDH_ANG_SP2_MAX)
              Redo[Atm -> Num - 1] = 3;
          } else Atm -> Pot.DL = VG_ADDH_FF_C1;
  }
        break;

      case VG_ELEM_N:
  if (Ang <= VG_ADDH_ANG_SP3_MAX) {
          Atm -> Pot.DL = VG_ADDH_FF_N3;
    Redo[Atm -> Num - 1] = 2;
  } else if (Ang <= VG_ADDH_ANG_SP_MIN) {
          if (Atm -> Ring == -6)
            Atm -> Pot.DL = VG_ADDH_FF_Npy;
          else
            Atm -> Pot.DL = VG_ADDH_FF_Npl;
        } else Atm -> Pot.DL = VG_ADDH_FF_N1;
  break;

      case VG_ELEM_O:
        Atm -> Pot.DL = VG_ADDH_FF_O3;
  break;

      case VG_ELEM_S:
        if (Atm -> Pot.DL == VG_ADDH_FF_S)
          Atm -> Pot.DL = VG_ADDH_FF_S3;
  break;
      } /* End of switch */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Valence 3 ****/

void AddhValence3(ATOMO *Atm, VG_LONG Flags)
{
  float         Ang;

  while(Atm) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm -> Active)) &&
        (Atm -> NSost == 3) && (Atm -> Flags & VG_ATMF_ORGANIC)) {
      Ang  = BondAnglef(Atm -> Conn[0], Atm, Atm -> Conn[1]);
      Ang += BondAnglef(Atm -> Conn[0], Atm, Atm -> Conn[2]);
      Ang += BondAnglef(Atm -> Conn[1], Atm, Atm -> Conn[2]);
      Ang /= 3.0f;

      switch(Atm -> Elem.S) {
      case VG_ELEM_C:
  if (Ang < VG_ADDH_ANG_SP3_MAX)
          Atm -> Pot.DL = VG_ADDH_FF_C3;
  else if (CountBndFreeOx(Atm) >= 2)
          Atm -> Pot.DL = VG_ADDH_FF_Cac;
        else Atm -> Pot.DL = VG_ADDH_FF_C2;
        break;

      case VG_ELEM_N:
  if (Ang < VG_ADDH_ANG_SP3_MAX)
          Atm -> Pot.DL = VG_ADDH_FF_N3;
  else if (CountBndFreeOx(Atm) >= 2)
          Atm -> Pot.DL = VG_ADDH_FF_Ntr;
        else if (Atm -> Ring == -5)
          Atm -> Pot.DL = VG_ADDH_FF_Nim;
        else Atm -> Pot.DL = VG_ADDH_FF_Npl;
  break;

      case VG_ELEM_S:
        if (Atm -> Pot.DL == VG_ADDH_FF_S) {
    if (CountBndFreeOx(Atm) >= 1)
            Atm -> Pot.DL = VG_ADDH_FF_Sox;
    else
            Atm -> Pot.DL = VG_ADDH_FF_S3p;
  }
  break;

      case VG_ELEM_B:
  if (CountBndFreeOx(Atm) >= 1)
          Atm -> Pot.DL = VG_ADDH_FF_Box;
  else
          Atm -> Pot.DL = VG_ADDH_FF_B;
  break;
      } /* End of switch */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Valence 4 ****/

void AddhValence4(register ATOMO *Atm, VG_LONG Flags)
{
  VG_ULONG              k;

  while(Atm) {
    if (((!(Flags & VG_ADDH_FLG_ACTONLY)) || (Atm -> Active)) &&
        (Atm -> NSost == 4) && (Atm -> Flags & VG_ATMF_ORGANIC)) {
      switch(Atm -> Elem.S) {
      case VG_ELEM_C:
        if (Atm -> Pot.DL == VG_ADDH_FF_C)
          Atm -> Pot.DL = VG_ADDH_FF_C3;
        break;

      case VG_ELEM_N:
        if (CountBndFreeOx(Atm) >= 1)
          Atm -> Pot.DL = VG_ADDH_FF_Nox;
        else
          Atm -> Pot.DL = VG_ADDH_FF_N3p;
        break;

      case VG_ELEM_P:
        if (Atm -> Pot.DL == VG_ADDH_FF_P) {
          k = CountBndFreeOx(Atm);
    if (k >= 2)
            Atm -> Pot.DL = VG_ADDH_FF_Pac;
    else if (k == 1)
            Atm -> Pot.DL = VG_ADDH_FF_Pox;
          else
            Atm -> Pot.DL = VG_ADDH_FF_P3p;
        }
        break;

      case VG_ELEM_S:
        if (Atm -> Pot.DL == VG_ADDH_FF_S) {
          k = CountBndFreeOx(Atm);
    if (k >= 3)
            Atm -> Pot.DL = VG_ADDH_FF_Sac;
    else if (k >= 1)
            Atm -> Pot.DL = VG_ADDH_FF_Sox;
          else
            Atm -> Pot.DL = VG_ADDH_FF_S;
        }
        break;

      case VG_ELEM_B:
        k = CountBndFreeOx(Atm);
        if (k >= 3)
          Atm -> Pot.DL = VG_ADDH_FF_Bac;
  else if (k >= 1)
          Atm -> Pot.DL = VG_ADDH_FF_Box;
        else Atm -> Pot.DL = VG_ADDH_FF_B;
        break;
      } /* End of switch */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}

