
/*************************************************
****      VEGA - Alchemy loader & saver       ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "count.h"
#include "bond.h"

/**** Local variables ****/

static const char       *AlcBondStr[] = {"SINGLE", "DOUBLE", "TRIPLE", "AROMATIC", NULL };


/**** Alchemy Loader ****/

ATOMO *AlchemyLoad(FILE *IN, RECORD *Lin, VG_ULONG *TotAtomi)
{
  ATOMO                 **ConMtx;
  char			Str[12];
  VG_UBYTE              Bond;
  VG_ULONG              i, j;

  ATOMO			*InizAtm = NULL;
  register ATOMO	*Atm     = NULL;
  VG_ULONG              NumAtoms = 0;
  VG_ULONG              NumBonds = 0;

  sscanf(Lin -> Line, "%d %*s %d", &NumAtoms, &NumBonds);
  if (!NumAtoms) return NULL;

  /**** Atoms ****/

  i = NumAtoms;
  while((i--) && (fgets(Lin -> Line, VG_LINELEN, IN))) {
    if ((*Lin -> Line) && (*Lin -> Line != '\n')) {
      if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
        sscanf(Lin -> Line, "%*s %8s %f %f %f %f",
               Str, &Atm -> x, &Atm -> y, &Atm -> z, &Atm -> Charge);
        Str2Qchar(&Atm -> Name, Str);
        AtmName2Elem(Atm);
      } else {
        FreeAtm(InizAtm);
        return NULL;
      }
    }
  } /* End of while */

  if (Atm) {
    Atm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
    LastAtm       = Atm;

    /**** Bonds ****/

    if ((NumBonds) && ((ConMtx = AllocAtmTable(InizAtm, NumAtoms, TRUE)) != NULL)) {
      while((NumBonds--) && (fgets(Lin -> Line, VG_LINELEN, IN))) {
        i = j = 0;
        sscanf(Lin -> Line, "%*s %d %d %11s", &i, &j, Str);
        for(Bond = 0; (AlcBondStr[Bond]) && (strcmp(Str, AlcBondStr[Bond])); ++Bond);
        if (AlcBondStr[Bond]) ++Bond;
        else Bond = VG_BOND_SINGLE;

        if ((i > 0) && (i <= NumAtoms) && (j > 0) && (j <= NumAtoms)) {
          BndMake(ConMtx[--i], ConMtx[--j], Bond);
          GLOBSW_CONNCALC = FALSE;
        }
      } /* End of while */
      FREE(ConMtx);
    }
  }

  return InizAtm;
}


/**** Alchemy saver ****/

VG_BOOL AlchemySave(FILE *OUT, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  ATOMO                 *Atm;
  char                  *BondName;
  register VG_UWORD	k;
  VG_ULONG              Count;

  VG_BOOL               Ret = TRUE;

  /**** Header ****/

  if (fprintf(OUT, "%5d ATOMS, %5d BONDS,     0 CHARGES\n",
                TotAtm, CountBond(InizAtm)) > 0) {

    /**** Atoms ****/

    Count = 1;
    for(Atm = InizAtm; (Ret) && (Atm); Atm = Atm -> Ptr) {
      if (fprintf(OUT, "%5d %-6.4s%8.4f %8.4f %8.4f %10.4f\n",
                  Count++, Atm -> Name.C, Atm -> x, Atm -> y,
                  Atm -> z, Atm -> Charge) <= 0) {
        Ret = PrintDosErr();
      }
    } /* End of for */

    /**** Bonds ****/

    if ((Ret) && ((Ret = BndAssignTypes(InizAtm, TotAtm, FALSE)))) {
      Count = 1;
      for(Atm = InizAtm; (Atm) && (Ret); Atm = Atm -> Ptr) {
        for(k = 0; (Ret) && (k < Atm -> NSost); ++k) {
          if (Atm -> Conn[k] -> Num > Atm -> Num) {
            if ((Atm -> Order[k] >= VG_BOND_SINGLE) && (Atm -> Order[k] <= VG_BOND_PARDOUBLE))
              BondName = (char *)AlcBondStr[Atm -> Order[k] - 1];
            else BondName = (char *)AlcBondStr[0];

            if (fprintf(OUT, "%5d  %4d  %4d  %s\n",
                        Count++, Atm -> Num, Atm -> Conn[k] -> Num, BondName) <= 0)
              Ret = PrintDosErr();
          }
        } /* End of for */
      } /* End of for */
    }
  } else Ret = PrintDosErr();

  return Ret;
}




