
/*************************************************
****     VEGA - Accelrys loader & saver       ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * Supported Accelrys formats:
 * - Archive 1     read & write
 * - Archive 3     read & write
 * - Arc traj.     read
 */


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#include "count.h"
#include "traj.h"


/**** Accelrys Universal Loader ****/

ATOMO *CarLoad(FILE *CARIN, register RECORD *CarLin, VG_ULONG *TotAtomi,
               VG_BOOL *IsArc)
{
  char              *FormStr, ResSeq[20];
  VG_LONG            k;

  register ATOMO     *Atm      = NULL;
  ATOMO              *InizAtm  = NULL;
  VG_BOOL            FirstEnd  = FALSE;
  VG_LONG            SubFormat = 0;


  /**** Subformat recognition ****/

  sscanf(CarLin -> Line + 15, "%d", &SubFormat);
  if ((SubFormat < 1) || (SubFormat > 3)) CatErr(MSG_ERR_BIOSYM_NOTSUPPORTED);
  else {
    if ((SubFormat == 1) || (SubFormat == 2)) FormStr = "%s %s %f";
    else FormStr = "%s %s %*s %f";

    /**** Skip the header ****/

    while((fgets(CarLin -> Line, VG_LINELEN, CARIN)) &&
          (CarLin -> Hdr != *(VG_LONG *)"!DAT"));

    while(fgets(CarLin -> Line, VG_LINELEN, CARIN)) {
      if ((*CarLin -> Line != 0x0a) &&
          (*CarLin -> Line != 0x0d)) {
        if ((CarLin -> Line[0] == 'e') &&
            (CarLin -> Line[1] == 'n') &&
            (CarLin -> Line[2] == 'd')) {
          if (!FirstEnd) {
            Atm -> Flags |= VG_ATMF_SEGEND;
            FirstEnd = TRUE;
          } else {
            while(fgets(CarLin -> Line, VG_LINELEN, CARIN)) {
              if (CarLin -> Hdr == *(VG_LONG *)"!DAT") {
                *IsArc = TRUE;
                break;
              }
            } /* End of while */
            break;
          }
        } else if ((CarLin -> Hdr != *(VG_LONG *)"HELI") &&
                   (CarLin -> Hdr != *(VG_LONG *)"PBC ")) {
          if ((Atm = AllocAtm(&InizAtm, TotAtomi)) != NULL) {
            FirstEnd = FALSE;
            Atm -> ResName.L = 0;
            sscanf(CarLin -> Line, "%s %f %f %f %3s",
                   Atm -> Name.C, &Atm -> x, &Atm -> y, &Atm -> z, Atm -> ResName.C);
                   Atm -> ChainID = ' ';

            /**** Convert the residue name ****/

      for(k = 0; *AAEquivTab[k].C; k += 2) {
        if (AAEquivTab[k].L == Atm -> ResName.L) {
          Atm -> ResName.L = AAEquivTab[k + 1].L;
          break;
        }
      } /* End of for */

            /**** Element name ****/

            if ((SubFormat != 3) ||
                ((SubFormat == 3) && CarLin -> Line[71] == '?')) {
              AtmName2Elem(Atm);
            } else {
              *Atm -> Elem.C = CarLin -> Line[71];
              if (CarLin -> Line[72] != ' ')
                Atm -> Elem.C[1] = CarLin -> Line[72];
            }
      if (!strncmp(Atm -> ResName.C, "HOH", 3)) Atm -> Flags = VG_ATMF_HETATM;
      sscanf(CarLin -> Line + 55, FormStr, ResSeq, Atm -> Pot.C, &Atm -> Charge);
            Str2Qchar(&Atm -> ResSeq, ResSeq);
          } else break;
        }
      }
    } /* End of while */

    if (Atm) {
      Atm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
      LastAtm       = Atm;
    }
  }

  return InizAtm;
}


/**** Open the trajectory file in arc format ****/

VG_BOOL CarOpen(char *FileName, TRJINFO *Info)
{
  RECORD        CarLin;
  VG_ULONG      k, *Ptr;

  VG_BOOL       FirstEnd = FALSE;
  VG_BOOL       Ret      = TRUE;
  VG_ULONG      TotAtm   = CountAtms(BegAtm, FALSE);

  if ((Info -> FH = fopen(FileName, "rb")) != NULL) {

    /**** Count the number of frames ****/

    while(fgets(CarLin.Line, VG_LINELEN, Info -> FH)) {
      if (CarLin.Hdr == *(VG_LONG *)"!DAT") {
        ++Info -> Frames;

        /**** Count the atoms ****/

        k = 0;
        while(fgets(CarLin.Line, VG_LINELEN, Info -> FH)) {
          if ((*CarLin.Line != 0x0a) &&
              (*CarLin.Line != 0x0d) &&
              (*CarLin.Line != '!')) {
            if ((CarLin.Line[0] == 'e') &&
                (CarLin.Line[1] == 'n') &&
                (CarLin.Line[2] == 'd')) {
              if (!FirstEnd) FirstEnd = TRUE;
              else break;
            } else if ((CarLin.Hdr != *(VG_LONG *)"HELI") &&
                       (CarLin.Hdr != *(VG_LONG *)"PBC ")) {
              FirstEnd = FALSE;
              ++k;
            }
          }
        } /* End of while */
        if (k != TotAtm) {
          CarClose(Info);
          return FALSE;
        }
      }
    } /* End of while */

    if (Info -> Frames) {
      fseek(Info -> FH, 0, SEEK_SET);

      /**** Create the seek table ****/

      if ((Info -> SeekTab = (VG_ULONG *)Alloca(Info -> Frames * sizeof(VG_ULONG))) != NULL) {
        Ptr = Info -> SeekTab;
        while(fgets(CarLin.Line, VG_LINELEN, Info -> FH)) {
          if (CarLin.Hdr == *(VG_LONG *)"!DAT")
            *Ptr++ = ftell(Info -> FH);
        } /* End of while */
        Info -> MolAtm = TotalAtm;
      } else Ret = FALSE;
    }
  } else Ret = PrintDosErr();

  if (!Ret) CarClose(Info);

  return Ret;
}


/**** Read one trajectory frame ****/

VG_BOOL CarReadFrm(TRJINFO *Info, ATOMO *Atm)
{
  RECORD        CarLin;

  VG_BOOL       FirstEnd = FALSE;
  VG_ULONG      Num      = 0;
  VG_ULONG      TotAtm   = CountAtms(Atm, FALSE);

  while(fgets(CarLin.Line, VG_LINELEN, Info -> FH)) {
    if ((*CarLin.Line != '!') &&
        (*CarLin.Line != 0x0a) &&
        (*CarLin.Line != 0x0d)) {
      if ((CarLin.Line[0] == 'e') &&
          (CarLin.Line[1] == 'n') &&
          (CarLin.Line[2] == 'd')) {
        if (FirstEnd) break;
        else FirstEnd = TRUE;
      } else if ((CarLin.Hdr != *(VG_LONG *)"Fram") &&
                 (CarLin.Hdr != *(VG_LONG *)"HELI") &&
                 (CarLin.Hdr != *(VG_LONG *)"PBC ")) {
        if (!(Atm -> Flags & VG_ATMF_CENTROID)) {
          if (Num++ < TotAtm) {
            sscanf(CarLin.Line, "%*s %f %f %f", &Atm -> x, &Atm -> y, &Atm -> z);
            FirstEnd = FALSE;
          } else return FALSE;
          Atm = Atm -> Ptr;
        }
      }
    }
  } /* End of while */

  if (Num != TotAtm) return FALSE;

  return TRUE;
}


/**** Biosym Saver ****/

VG_BOOL CarSave(FILE *OUT, register ATOMO *Atm, VG_WORD SubFormat)
{
  char      DataStr[28];

  VG_BOOL   Ret   = TRUE;
  VG_ULONG  Count = 1;
  time_t    Data  = time(NULL);

  strcpy(DataStr, ctime(&Data));
  DataStr[24] = 0;

  /**** Write the header ****/

  if (fprintf(OUT, "!BIOSYM archive %d%63s\n" \
                   "%-80s\n" \
                   "%-80s\n" \
                   "!DATE %-74s\n",
                   SubFormat, " ", "PBC=OFF", VEGAHdr, DataStr) >= 0) {
    do {
      switch(SubFormat) {
      case 1:
        if (fprintf(OUT, "%-5.4s%15.9f%15.9f%15.9f %-5.4s%4.4s %-4.4s%9.4f%6d\n",
                         Atm -> Name.C, Atm -> x, Atm -> y, Atm -> z, Atm -> ResName.C,
                         Atm -> ResSeq.C, Atm -> Pot.C, Atm -> Charge, Count) < 0)
          Ret = PrintDosErr();
        break;
      case 3:
        if (fprintf(OUT, "%-5.4s%15.9f%15.9f%15.9f %-5.4s%-7.4s%-8.4s%-2.2s%7.3f\n",
                         Atm -> Name.C, Atm -> x, Atm -> y, Atm -> z, Atm -> ResName.C,
                         Atm -> ResSeq.C, Atm -> Pot.C, Atm -> Elem.C, Atm -> Charge) < 0)
          Ret = PrintDosErr();
        break;
      }
      if ((Ret) && (Atm -> Flags & VG_ATMF_SEGEND) && (fprintf(OUT, "end\n") < 0))
        Ret = PrintDosErr();
      Atm = Atm -> Ptr;
      ++Count;
    } while((Ret) && (Atm));
    if ((Ret) && (fprintf(OUT, "end\n") < 0))
      Ret = PrintDosErr();
  } else Ret = PrintDosErr();

  return Ret;
}
