
/*************************************************
****      VEGA - Bond managment routines      ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <string.h>
#include <malloc.h>
#include <math.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#include "bond.h"

/**** Constants ****/

#define  VG_ONE_OVER_SQRT3              0.577350269f
#define  VG_SQRT_TWO_THIRDS             0.816496581f

/**** Macros ****/

#define  XyzScalar(V, S)                (V).x *= (S); \
                                        (V).y *= (S); \
                                        (V).z *= (S);
#define  XyzSum(S, V1, V2)              (S).x = (V1).x + (V2).x; \
                                        (S).y = (V1).y + (V2).y; \
                                        (S).z = (V1).z + (V2).z;


/**** Matrix 3x3 determinant ****/

float MatDet3x3(float a[3][3])
{
  return (a[0][0] * (a[1][1] * a[2][2] - a[2][1] * a[1][2]))
	  -(a[0][1] * (a[1][0] * a[2][2] - a[2][0] * a[1][2]))
	  +(a[0][2] * (a[1][0] * a[2][1] - a[2][0] * a[1][1]));
}


/**** Matrix 3x3 invert ****/

void MatInv3x3(float m[3][3])
{
  float         t[3][3];
  float         Det = MatDet3x3(m);

  if (Det != 0.0f) {
    t[0][0] = m[1][1] * m[2][2] - m[1][2] * m[2][1];
    t[1][0] = m[1][2] * m[2][0] - m[1][0] * m[2][2];
    t[2][0] = m[1][0] * m[2][1] - m[1][1] * m[2][0];
    t[0][1] = m[2][1] * m[0][2] - m[2][2] * m[0][1];
    t[1][1] = m[2][2] * m[0][0] - m[2][0] * m[0][2];
    t[2][1] = m[2][0] * m[0][1] - m[2][1] * m[0][0];
    t[0][2] = m[0][1] * m[1][2] - m[0][2] * m[1][1];
    t[1][2] = m[0][2] * m[1][0] - m[0][0] * m[1][2];
    t[2][2] = m[0][0] * m[1][1] - m[0][1] * m[1][0];
    
    m[0][0] = t[0][0] / Det;
    m[1][0] = t[1][0] / Det;
    m[2][0] = t[2][0] / Det;
    m[0][1] = t[0][1] / Det;
    m[1][1] = t[1][1] / Det;
    m[2][1] = t[2][1] / Det;
    m[0][2] = t[0][2] / Det;
    m[1][2] = t[1][2] / Det;
    m[2][2] = t[2][2] / Det;
  }
}


/**** Vectorial cross product ****/

void XyzCrossProd(XYZ *Normal, XYZ *Vect1, XYZ *Vect2)
{
  Normal -> x = Vect1 -> y * Vect2 -> z - Vect2 -> y * Vect1 -> z;
  Normal -> y = Vect2 -> x * Vect1 -> z - Vect1 -> x * Vect2 -> z;
  Normal -> z = Vect1 -> x * Vect2 -> y - Vect2 -> x * Vect1 -> y;
}


/**** Magnitude of a single precision vector ****/

float XyzMagnitude(XYZ *V0)
{
  return SQR(Quad(V0 -> x) + Quad(V0 -> y) + Quad(V0 -> z));
}


/**** Normalize the single precision vector ****/

void XyzNormalize(XYZ *V0)
{
  float         Mag = SQR(Quad(V0 -> x) + Quad(V0 -> y) + Quad(V0 -> z));

  if (Mag != 0.0) {
    V0 -> x = V0 -> x / Mag;
    V0 -> y = V0 -> y / Mag;
    V0 -> z = V0 -> z / Mag;
  }
}


/**** Rotate a vector around the Z axis ****/

void XyzRotZ(XYZ *V, float Rot)
{
  float         CosAz, SinAz, T;

  Rot   *= DEG_TO_RAD;
  CosAz  = cos(Rot);
  SinAz  = sin(Rot);

  T       = V -> x * CosAz - V -> y * SinAz;
  V -> y  = V -> x * SinAz + V -> y * CosAz;
  V -> x  = T;
}


/**** Set sp geometry ****/

void AtmSetGeoSp(ATOMO *Dest, ATOMO *Atm1, float BndLen)
{
  float         Dist;
  VG_BYTE       k;
  XYZ           Atm2;

  if (Atm1 -> NSost > 1) {
    for(k = 0; (k < Atm1 -> NSost) && (Atm1 -> Conn[k] == Dest); ++k);
    Atm2.x = Atm1 -> Conn[k] -> x;
    Atm2.y = Atm1 -> Conn[k] -> y;
    Atm2.z = Atm1 -> Conn[k] -> z;
  } else Atm2.x = Atm2.y = Atm2.z = 0.0;

  Dist = SQR(Quad(Atm1 -> x - Atm2.x) + Quad(Atm1 -> y - Atm2.y) + Quad(Atm1 -> z - Atm2.z));
  Dist = (Dist + BndLen) / Dist;
  Dest -> x = (Atm1 -> x - Atm2.x) * Dist + Atm2.x;
  Dest -> y = (Atm1 -> y - Atm2.y) * Dist + Atm2.y;
  Dest -> z = (Atm1 -> z - Atm2.z) * Dist + Atm2.z;
}


/**** Set sp2 geometry ****/

void AtmSetGeoSp2(ATOMO *Dest, ATOMO *Atm1, float BndLen)
{
  ATOMO         *Atm2, *Atm3;
  VG_BYTE       k;
  XYZ           h1vect, v1, v2;

  if (Atm1 -> NSost >= 2) {
    for(k = 0; (k < Atm1 -> NSost) && (Atm1 -> Conn[k] == Dest); ++k);
    Atm2 = Atm1 -> Conn[k];

    v1.x = Atm1 -> x - Atm2 -> x;
    v1.y = Atm1 -> y - Atm2 -> y;
    v1.z = Atm1 -> z - Atm2 -> z;
    XyzNormalize(&v1);

    if (Atm1 -> NSost >= 3) {
      for(k = 0; (k < Atm1 -> NSost) &&
                 ((Atm1 -> Conn[k] == Dest) || (Atm1 -> Conn[k] == Atm2)); ++k);
      Atm3 = Atm1 -> Conn[k];

      v2.x = Atm1 -> x - Atm3 -> x;
      v2.y = Atm1 -> y - Atm3 -> y;
      v2.z = Atm1 -> z - Atm3 -> z;
      XyzNormalize(&v2);
      XyzSum(h1vect, v1, v2);
      XyzNormalize(&h1vect);
    } else {
      for(k = 0; (k < Atm2 -> NSost) && (Atm2 -> Conn[k] == Atm1); ++k);
      if (k != Atm2 -> NSost) {
        Atm3 = Atm2 -> Conn[k];
        h1vect.x = Atm2 -> x - Atm3 -> x;
        h1vect.y = Atm2 -> y - Atm3 -> y;
        h1vect.z = Atm2 -> z - Atm3 -> z;
        XyzNormalize(&h1vect);
      } else {
        h1vect.x = Atm2 -> x - Atm1 -> x;
        h1vect.y = Atm2 -> y - Atm1 -> y;
        h1vect.z = Atm2 -> z - Atm1 -> z;
        XyzRotZ(&h1vect, VG_ANGLE_SP2);
        XyzNormalize(&h1vect);
      }
    }
    Dest -> x = h1vect.x * BndLen + Atm1 -> x;
    Dest -> y = h1vect.y * BndLen + Atm1 -> y;
    Dest -> z = h1vect.z * BndLen + Atm1 -> z;
  } else AtmSetGeoSp(Dest, Atm1, BndLen);
}


/**** Set sp3 geometry ****/

void AtmSetGeoSp3(ATOMO *Dest, ATOMO *Atm1, float BndLen)
{
  ATOMO         *Atm2, *Atm3, *Atm4;
  float         m[3][3];
  VG_BYTE       k;
  XYZ           v1, v2, v3, s, h1vect;
  float         T;

  if (Atm1 -> NSost >= 2) {
    for(k = 0; (k < Atm1 -> NSost) && (Atm1 -> Conn[k] == Dest); ++k);
    Atm2 = Atm1 -> Conn[k];

    v1.x = Atm1 -> x - Atm2 -> x;
    v1.y = Atm1 -> y - Atm2 -> y;
    v1.z = Atm1 -> z - Atm2 -> z;
    XyzNormalize(&v1);

    if (Atm1 -> NSost >= 3) {
      for(k = 0; (k < Atm1 -> NSost) &&
                 ((Atm1 -> Conn[k] == Dest) || (Atm1 -> Conn[k] == Atm2)); ++k);
      Atm3 = Atm1 -> Conn[k];

      v2.x = Atm1 -> x - Atm3 -> x;
      v2.y = Atm1 -> y - Atm3 -> y;
      v2.z = Atm1 -> z - Atm3 -> z;
      XyzNormalize(&v2);

      if (Atm1 -> NSost >= 4) {
        for(k = 0; (k < Atm1 -> NSost) &&
                   ((Atm1 -> Conn[k] == Dest) || (Atm1 -> Conn[k] == Atm2) ||
                    (Atm1 -> Conn[k] == Atm3)); ++k);
        Atm4 = Atm1 -> Conn[k];

        v3.x = Atm1 -> x - Atm4 -> x;
        v3.y = Atm1 -> y - Atm4 -> y;
        v3.z = Atm1 -> z - Atm4 -> z;
        XyzNormalize(&v3);
        m[0][0] = v1.x;  m[1][0] = v1.y; m[2][0] = v1.z;
        m[0][1] = v2.x;  m[1][1] = v2.y; m[2][1] = v2.z;
        m[0][2] = v3.x;  m[1][2] = v3.y; m[2][2] = v3.z;
        MatInv3x3(m);
        h1vect.x = m[0][0] + m[1][0] + m[2][0];
        h1vect.y = m[0][1] + m[1][1] + m[2][1];
        h1vect.z = m[0][2] + m[1][2] + m[2][2];
        XyzNormalize(&h1vect);
      } else {
        XyzSum(s, v1, v2);
        XyzCrossProd(&v3, &v1, &v2);
        XyzScalar(s, VG_ONE_OVER_SQRT3);
        XyzScalar(v3, VG_SQRT_TWO_THIRDS);
        XyzSum(h1vect, s, v3);
        XyzNormalize(&h1vect);
      }
    } else {
      for(k = 0; (k < Atm2 -> NSost) && (Atm2 -> Conn[k] == Atm1); ++k);
      if (k != Atm2 -> NSost) {
        Atm3 = Atm2 -> Conn[k];
        h1vect.x = Atm2 -> x - Atm3 -> x;
        h1vect.y = Atm2 -> y - Atm3 -> y;
        h1vect.z = Atm2 -> z - Atm3 -> z;
        XyzNormalize(&h1vect);
      } else {
        h1vect.x = Atm2 -> x - Atm1 -> x;
        h1vect.y = Atm2 -> y - Atm1 -> y;
        h1vect.z = Atm2 -> z - Atm1 -> z;
        XyzRotZ(&h1vect, VG_ANGLE_SP3);
        XyzNormalize(&h1vect);
      }
    }
    Dest -> x = h1vect.x * BndLen + Atm1 -> x;
    Dest -> y = h1vect.y * BndLen + Atm1 -> y;
    Dest -> z = h1vect.z * BndLen + Atm1 -> z;
  } else AtmSetGeoSp(Dest, Atm1, BndLen);
}


/**** Assign the bond types ****/

VG_BOOL BndAssignTypes(ATOMO *InizAtm, VG_ULONG TotAtm, VG_BOOL ActiveOnly)
{
  ATOMO                 *Atm1, *Atm2;
  char			FieldName[VG_BOND_FIELDLEN];
  VG_UBYTE              Bond, i, j, k, l;
  float                 T;

  VG_BOOL               Ret    = TRUE;

#ifdef __VG_OPENGL
  VG_DLONG              *Store = NULL;

  if (GLOBSW_OPENGL)
    Store = FFPush(InizAtm, TotAtm);
#endif

  /**** Assign BOND atom types ****/

  PrintProg(MSG_BOND_ASSIGNPROG);
  GetPrgPath(FieldName, TRUE);
  strcat(FieldName, VG_BOND_TEMPLATE);
  if ((Ret = AssignFF(FieldName, InizAtm, TotAtm, ActiveOnly, TRUE))) {
    for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
      for(k = 0; k < Atm1 -> NSost; ++k) {
        Atm2 = Atm1 -> Conn[k];
        if (Atm2 -> Num > Atm1 -> Num) {
          for(j = 0; (j < Atm2 -> NSost) && (Atm2 -> Conn[j] != Atm1); ++j);
          Bond = VG_BOND_SINGLE;
          if (j < Atm2 -> NSost) {
            if ((*Atm1 -> Pot.C != '?') && (*Atm2 -> Pot.C != '?')) {
              if (Atm1 -> Pot.DL == Atm2 -> Pot.DL) {
                Bond = *Atm1 -> Pot.C - 0x30;

                /**** Check if the atoms aren't planar ****/

                if ((*Atm1 -> Pot.C == '4') && (*Atm2 -> Pot.C == '4') &&
                    (Atm1 -> NSost == 3) && (Atm2 -> NSost == 3)) {
                  for(i = 0; (i < Atm1 -> NSost) && (Atm1 -> Conn[i] == Atm2); ++i);
                  for(l = 0; (l < Atm2 -> NSost) && (Atm2 -> Conn[l] == Atm1); ++l);

                  T = (float)NormAngle(Torsion(Atm1 -> Conn[i], Atm1, Atm2, Atm2 -> Conn[l]));
                  if ((T > 20.0f) && (T < 160.0f)) Bond = VG_BOND_SINGLE;
                  else if ((T > 200.0f) && (T < 340.0f)) Bond = VG_BOND_SINGLE;
                }
              }
            }
            Atm2 -> Order[j] = Bond;
          }
          Atm1 -> Order[k] = Bond;
        }
      } /* End of for */
    } /* End of for */
  }

#ifdef __VG_OPENGL
  if (Store) {
    FFPop(InizAtm, Store);
    FREE(Store);
  }
#endif
  return Ret;
}


/**** Check if two atoms are already connected ****/

VG_BOOL BndCheck(ATOMO *Atm1, ATOMO *Atm2, VG_UBYTE *Type)
{
  VG_BYTE       k;

  if ((!Atm1 -> NSost) || (!Atm2 -> NSost)) return FALSE;

  for(k = 0; k < Atm2 -> NSost; ++k)
    if (Atm2 -> Conn[k] == Atm1) {
      if (Type) *Type = Atm2 -> Order[k];
      return TRUE;
    }

  return FALSE;
}


/**** Make bond ****/

VG_BOOL BndMake(ATOMO *Atm1, ATOMO *Atm2, VG_BYTE Type)
{
  if ((Atm1 -> NSost >= MAXBOND) || (Atm2 -> NSost >= MAXBOND))
    return FALSE;

  Atm1 -> Conn[Atm1 -> NSost]  = Atm2;
  Atm1 -> Order[Atm1 -> NSost] = Type;
  ++Atm1 -> NSost;
  Atm2 -> Conn[Atm2 -> NSost]  = Atm1;
  Atm2 -> Order[Atm2 -> NSost] = Type;
  ++Atm2 -> NSost;

  return TRUE;
}


#ifdef __VG_OPENGL

/**** Change the bond type ****/

void BndChange(ATOMO *Atm1, ATOMO *Atm2, VG_UBYTE Type)
{
  BndChangeSingle(Atm1, Atm2, Type);
  BndChangeSingle(Atm2, Atm1, Type);
}


/**** Change the bond type to all atoms ****/

void BndChangeAll(ATOMO *Atm, VG_UBYTE Type, VG_BOOL ActiveOnly)
{
  VG_BYTE       k;

  while(Atm) {
    if ((!ActiveOnly) || (Atm -> Active)) {
      for(k = 0; (k < Atm -> NSost); ++k) {
        if ((!ActiveOnly) || (Atm -> Conn[k] -> Active))
          Atm -> Order[k] = Type;
      } /* End of for */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Change the bond type to a single atom ****/

void BndChangeSingle(ATOMO *Atm1, ATOMO *Atm2, VG_UBYTE Type)
{
  VG_BYTE       k;

  for(k = 0; (k < Atm1 -> NSost); ++k)
    if (Atm1 -> Conn[k] == Atm2) {
      Atm1 -> Order[k] = Type;
      break;
    }
}


/**** Delete bond ****/

void BndDelete(ATOMO *Atm1, ATOMO *Atm2)
{
  BndUnlink(Atm1, Atm2);
  BndUnlink(Atm2, Atm1);
}


/**** Delete all bonds ****/

void BndDeleteAll(ATOMO *Atm, VG_BOOL ActiveOnly)
{
  register VG_BYTE      k;

  for(; Atm; Atm = Atm -> Ptr) {
    if ((ActiveOnly) && (!Atm -> Active)) continue;
    for(k = 0; k < Atm -> NSost; ++k) Atm -> Conn[k] = NULL;
    Atm -> NSost = 0;
  } /* End of while */
}


/**** Find recursively all atoms connected to one ****/

void BndFindConnAtms(VG_BYTE *ConnMtx, ATOMO *Atm, VG_ULONG *Count)
{
  VG_BYTE       k;

  if (ConnMtx[Atm -> Num - 1]) return;

  ConnMtx[Atm -> Num - 1] = TRUE;
  ++*Count;

  for(k = 0; k < Atm -> NSost; ++k)
    BndFindConnAtms(ConnMtx, Atm -> Conn[k], Count);
}


/**** Get the bond length ****/

float BndGetLength(VG_UWORD Elem1, VG_UWORD Elem2)
{
  if (Elem1 == Elem2)
    return (2.0 * GetAtmCovRad(Elem1));

  return (GetAtmCovRad(Elem1) + GetAtmCovRad(Elem2));
}


/**** Get the rotation matrix ****/

VG_BOOL BndGetRotMat(float mat[9], float angle, float x, float y, float z)
{
  float         xx, yy, zz, xy, yz, zx, xs, ys, zs, one_c;

  float         s   = (float)sin(angle * DEG_TO_RAD);
  float         c   = (float)cos(angle * DEG_TO_RAD);
  float         mag = (float)sqrt(x * x + y * y + z * z);


  /**** The matrix can be constructed ? ****/

  if (mag < VG_BOND_ROTPRECISION) return FALSE;

  mag    = 1.0f / mag;

  x     *= mag;
  y     *= mag;
  z     *= mag;

  one_c  = 1.0f - c;

  xx     = x * x * one_c;
  yy     = y * y * one_c;
  zz     = z * z * one_c;
  xy     = x * y * one_c;
  yz     = y * z * one_c;
  zx     = z * x * one_c;

  xs     = x * s;
  ys     = y * s;
  zs     = z * s;

  mat[0] = xx + c;
  mat[1] = xy - zs;
  mat[2] = zx + ys;

  mat[3] = xy + zs;
  mat[4] = yy + c;
  mat[5] = yz - xs;

  mat[6] = zx - ys;
  mat[7] = yz + xs;
  mat[8] = zz + c;

  return TRUE;
}


/**** Rotate a bond ****/

void BndRotate(VG_UBYTE *ConMtx, ATOMO *Atm1, ATOMO *Atm2, float dtor)
{
  ATOMO         *Atm;
  float         m[9], r[3];
  int           index;

  /**** Rotate around the vector r ****/

  r[0] = Atm2 -> x - Atm1 -> x;
  r[1] = Atm2 -> y - Atm1 -> y;
  r[2] = Atm2 -> z - Atm1 -> z;

  if (BndGetRotMat(m, dtor, r[0], r[1], r[2]) == FALSE) return;

  for(Atm = BegAtm; Atm; Atm = Atm -> Ptr) {
    if (ConMtx[Atm -> Num - 1]) {
      r[0] = Atm -> x - Atm1 -> x;
      r[1] = Atm -> y - Atm1 -> y;
      r[2] = Atm -> z - Atm1 -> z;
      Atm -> x = Atm1 -> x + m[0] * r[0] + m[1] * r[1] + m[2] * r[2];
      Atm -> y = Atm1 -> y + m[3] * r[0] + m[4] * r[1] + m[5] * r[2];
      Atm -> z = Atm1 -> z + m[6] * r[0] + m[7] * r[1] + m[8] * r[2];
    }
  } /* End of for */
}


/**** Unlink two atoms ****/

void BndUnlink(ATOMO *Atm1, ATOMO *Atm2)
{
  VG_BYTE       k;

  switch(Atm1 -> NSost) {
  case 0:
    return;

  case 1:
    Atm1 -> NSost = 0;
    *Atm1 -> Conn = NULL;
    return;

  default:

    /**** Find the connected atom ****/

    for(k = 0; (k < Atm1 -> NSost) && (Atm1 -> Conn[k] != Atm2); ++k);

    --Atm1 -> NSost;
    while(k < Atm1 -> NSost) {
      Atm1 -> Conn[k]  = Atm1 -> Conn[k + 1];
      Atm1 -> Order[k] = Atm1 -> Order[k + 1];
      ++k;
    } /* End of while */
    Atm1 -> Conn[Atm1 -> NSost] = NULL;
  } /* End of switch */
}


/**** Unlink all connected atoms to one ****/

void BndUnlinkAll(ATOMO *Atm1)
{
  VG_BYTE       k;

  for(k = 0; (k < Atm1 -> NSost); ++k)
    BndUnlink(Atm1 -> Conn[k], Atm1);
}
#endif


