
/*************************************************
****           VEGA - Database engine         ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <time.h>

#ifdef __WIN32__
#  include <windows.h>
#  include <gl/gl.h>
#endif

#ifdef __BORLANDC__ 
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "db_engine.h"
#include "formats.h"
#include "vglist.h"
#include "pk_def.h"

#ifdef __VG_OPENGL
#  include "gl_global.h"
#  include "gl_menu.h"
#  include "gl_update.h"
#  include "gl_wrkspace.h"
#endif

/**** Prototypes ****/

void             Bool2RetStr(VG_BOOL);
void             Float2RetStr(float);
void             Int2RetStr(VG_ULONG);
void             ZeroRetStr(void);

/**** Global variables ****/

const char      *DbDataStr = VG_DB_DATFILE_NAME;
VG_LIST         *DbList    = NULL;

/**** Prototypes ****/

void MC_New(VG_BOOL);


/**** Close a database ****/

VG_BOOL DbClose(VG_DBINFO *Dbh, VG_BOOL FreeAll, VG_BOOL Force)
{
  if ((!Force) && (Dbh -> Lock != VG_DB_LOCK_NONE))
    return FALSE;

  switch(Dbh -> Type) {
  case VG_DB_TYPE_FILE:         /* File database */
    DbFileClose(Dbh);
    break;

  case VG_DB_TYPE_SDF:          /* SDF database */
    DbSdfClose(Dbh);
    break;

  case VG_DB_TYPE_ZIP:          /* Zip database */
    DbZipClose(Dbh);
    break;
  } /* End of switch */
  DbMolListClear(Dbh);
  if (FreeAll) FREE(Dbh);

  return TRUE;
}


/**** Close all databases ****/

void DbCloseAll(VG_LIST **Lh, VG_BOOL Force)
{
  VG_BOOL       FreeAll;
  VG_DBINFO     *Dbh, *NextDbh;

  if (*Lh) {
    FreeAll = TRUE;
    Dbh     = (VG_DBINFO *)(*Lh) -> Beg;
    while(Dbh) {
      NextDbh = Dbh -> Next;
      if (!DbClose(Dbh, TRUE, Force)) FreeAll = FALSE;
      Dbh = NextDbh;
    } /* End of while */

    if (FreeAll) {
      FREE(*Lh);
      *Lh = NULL;
    }
  }
}


/**** Create a new database ****/

VG_DBINFO *DbCreate(char *FileName, VG_LONG Type, FILEFORMAT *MolFormat,
                    VG_UWORD MolPack, VG_LONG MolFlags)
{
  VG_DBINFO     *Dbh;
  VG_BOOL       Ret;

  if ((Dbh = (VG_DBINFO *)Alloca(sizeof(VG_DBINFO))) != NULL) {
    strcpy(Dbh -> FileName, FileName);
    Dbh -> Type      = Type;
    Dbh -> MolFlags  = MolFlags;
    Dbh -> MolFormat = MolFormat;
    Dbh -> MolPack   = MolPack;
    switch(Type) {
    case VG_DB_TYPE_FILE:         /* File database */
      Ret = DbFileCreate(Dbh);
      break;

    case VG_DB_TYPE_SDF:          /* SDF database */
      Ret = DbSdfCreate(Dbh);
      break;

    case VG_DB_TYPE_ZIP:          /* Zip database */
      Ret = DbZipCreate(Dbh);
      break;
    } /* End of switch */
    if (Ret) {
      if (MolFlags & VG_DB_MOLFLAG_VERBOSE)
        CatPrintf(stdout, MSG_DBENGINE_CREATE, Dbh -> FileName);
      return Dbh;
    }
    FREE(Dbh);
  }
  CatErr(MSG_ERR_DBENGINE_CREATE);

  return NULL;
}


/**** Create .dat file ****/

VG_BOOL DbDatCreate(char *FileName, VG_DBINFO *Dbh)
{
  FILE          *FH;

  VG_BOOL       Ret = FALSE;

  if ((FH = fopen(FileName, "w")) != NULL) {
    if (fprintf(FH, "#DBASEINFO " VG_DB_DATFILE_VER  "\n"
                    "\n"
                    "#MOLFORMAT %s %s %d %d\n",
                    Dbh -> MolFormat -> Com, PackerNames[Dbh -> MolPack],
                    (Dbh -> MolFlags & VG_DB_MOLFLAG_CONN) ? 1: 0,
                    (Dbh -> MolFlags & VG_DB_MOLFLAG_FIX) ? 1: 0) > 0) {
      Ret = TRUE;
    } else PrintDosErr();
    fclose(FH);
  }

  return Ret;
}


/**** Read .dat file ****/

VG_BOOL DbDatRead(char *FileName, VG_DBINFO *Dbh)
{
  char          Buf[256], *Ptr;
  FILE          *FH;
  FILEFORMAT    *Format;
  VG_LONG       Pack;

  VG_BOOL       Ret  = TRUE;
  VG_ULONG      Pos  = 0;

  if ((FH = fopen(FileName, "r")) != NULL) {
    while(fgets(Buf, 256, FH)) {
      if (((Ptr = strtok(Buf, FieldSep)) != NULL) && (*Ptr)) {
        if (Pos) {
          if ((!strcasecmp(Ptr, "#MOLFORMAT")) &&
              ((Ptr = strtok(NULL, FieldSep)) != NULL)) {
            for(Format = FileInfo; (Format -> Com) && (strcasecmp(Format -> Com, Ptr)); ++Format);
            if ((Format -> Com) && (!(Format -> Flags & FORM_FLAG_INONLY))) Dbh -> MolFormat = Format;
            if (((Ptr = strtok(NULL, FieldSep)) != NULL) &&
                ((Pack = ChkPackerName(Ptr, FALSE)) != -1)) {
              Dbh -> MolPack = Pack;
              if ((Ptr = strtok(NULL, FieldSep)) != NULL) {
                Dbh -> MolFlags |= VG_DB_MOLFLAG_CONN * atoi(Ptr);
                if ((Ptr = strtok(NULL, FieldSep)) != NULL) {
                  Dbh -> MolFlags |= VG_DB_MOLFLAG_FIX * atoi(Ptr);
                }
              }
            }
          }
        } else if (strcasecmp(Buf, "#DBASEINFO")) {
          Ret = FALSE;
          break;
        }
        ++Pos;
      }
    } /* End of while */
    fclose(FH);
  } else Ret = PrintDosErr();

  return Ret;
}


/**** Get a molucule ****/

VG_BOOL DbGet(VG_DBINFO *Dbh, char *MolName, VG_ULONG Idx, VG_LONG Mode, VG_BOOL Verbose)
{
  ATOMO         *TmpIniz;
  VG_BOOL       Ret;
  VG_DBMOLLIST  *File;
  VG_LONG       FormatID;
  VG_WRKSPACE   *Wks;

  if ((!MolName) && ((File = DbMolListGet(Dbh, NULL, Idx)) != NULL)) {
    MolName = File -> MolName;
  }

  if (MolName) {
    switch(Mode) {
    case VG_DB_GET_REPLACE:
      MC_New(FALSE);
      break;

#ifdef __VG_OPENGL
    case VG_DB_GET_WORKSPACE:
      if ((Wks = GL_WksAdd()) != NULL)
        GL_WksChange(Wks, 0, TRUE);
      else return FALSE;
#endif
      break;
    } /* End of switch */

    switch(Dbh -> Type) {
    case VG_DB_TYPE_FILE:         /* File database */
      TmpIniz  = DbFileGet(Dbh, MolName, &FormatID);
      break;

    case VG_DB_TYPE_SDF:          /* SDF database */
      TmpIniz  = DbSdfGet(Dbh, MolName);
      FormatID = FORM_MDL;
      break;

    case VG_DB_TYPE_ZIP:          /* Zip database */
      TmpIniz  = DbZipGet(Dbh, MolName, &FormatID);
      break;
    } /* End of switch */

    if (TmpIniz) {
#ifdef __VG_OPENGL
      ++ViewPrefs.TotMol;
      AssignVdwNew(TmpIniz, 0, 0);
      if (FormatID != FORM_IFF) {
        GL_ColorMol(TmpIniz, ViewPrefs.TotMol + 2);
        CalcGeoCent(BegAtm, &ViewPrefs.ViewCenter, TRUE);
        LastCent = ViewPrefs.ViewCenter;
      }
      FindRing(TmpIniz, TRUE);
      LastCent = ViewPrefs.ViewCenter;
      strcpy(LastFileName, MolName);
      ChangeExt(LastFileName, NULL, NULL);
      GL_WksSetName(LastFileName);
      GL_ChangeMenu(VG_CM_MOLE);
      GL_DrawForce();

      /**** Update the opened dialogs ****/

      GL_UpdateDialogs();
/*
      DlgAtmSelUpdate();
      DlgRemResUpdate();
      DlgSelMolUpdate();
*/
#endif
      strcpy(Dbh -> CurMolName, MolName);
      Ret = TRUE;
    } else Ret = FALSE;
  } else Ret = FALSE;

  if (!Ret) CatErr(MSG_ERR_DBENGINE_GET, MolName);
  else if (Verbose) CatPrintf(stdout, MSG_DBENGINE_GET, MolName);

  return Ret;
}


/**** Get information about the opened database ****/

void DbInfo(VG_DBINFO *Dbh, VG_ULONG Var)
{
  const char    *DbFormats[] = { "UNKNOWN",
                                 "FILE",
                                 "SDF",
                                 "ZIP"
                               };

  switch(Var) {
  case VG_DB_INFO_CURMOLNAME:
    strcpy(ReturnStr, Dbh -> CurMolName);
    break;

  case VG_DB_INFO_FILENAME:
    strcpy(ReturnStr, Dbh -> FileName);
    break;

  case VG_DB_INFO_FORMAT:
    strcpy(ReturnStr, DbFormats[Dbh -> Type]);
    break;

  case VG_DB_INFO_MOLECULES:
    Int2RetStr(Dbh -> TotMol);
    break;

  case VG_DB_INFO_MOLFLAGS:
    Int2RetStr(Dbh -> MolFlags);
    break;

  case VG_DB_INFO_MOLFORMAT:
    strcpy(ReturnStr, Dbh -> MolFormat -> Com);
    break;
  } /* End switch */
}


/**** Get the database handle from the id ****/

VG_DBINFO *DbInfoFromId(register VG_ULONG Id)
{
  register VG_DBINFO    *Dbh = NULL;

  if (DbList) {
    for(Dbh = (VG_DBINFO *)DbList -> Beg;
        (Dbh) && (Dbh -> Id != Id);
        Dbh = Dbh -> Next);
  }

  return Dbh;
}


/**** Add a molecule into the list ****/

VG_DBMOLLIST *DbMolListAdd(VG_DBINFO *Dbh, char *MolName)
{
  VG_DBMOLLIST          *Mol;

  if ((Mol = (VG_DBMOLLIST *)ListAppend(Dbh -> MolList)) != NULL) {
    strcpy(Mol -> MolName, MolName);
    ++Dbh -> TotMol;
  }

  return Mol;
}


/**** Clear the molecule list ****/

void DbMolListClear(VG_DBINFO *Dbh)
{
  ListClear(Dbh -> MolList);
  Dbh -> MolList = NULL;
  Dbh -> TotMol  = 0;
}


/**** Check the molecule name ****/

VG_BOOL DbMolListChk(VG_DBINFO *Dbh, char *MolName)
{
  VG_DBMOLLIST  *Ptr;
  VG_BOOL       Ret = FALSE;

  for(Ptr = (VG_DBMOLLIST *)Dbh -> MolList -> Beg; Ptr; Ptr = Ptr -> Next) {
    if (!strcasecmp(MolName, Ptr -> MolName)) {
      Ret = TRUE;
      break;
    }
  } /* End of for */
  return Ret;
}


/**** Get the molecule item ****/

VG_DBMOLLIST *DbMolListGet(VG_DBINFO *Dbh, char *MolName, VG_ULONG Idx)
{
  VG_DBMOLLIST          *Ptr;

  if (MolName) {
    for(Ptr = (VG_DBMOLLIST *)Dbh -> MolList;
        (Ptr) && (strcasecmp(Ptr -> MolName, MolName));
        Ptr = Ptr -> Next);
  } else Ptr = (VG_DBMOLLIST *)ListGet(Dbh -> MolList, Idx);

  return Ptr;
}


/**** Read the molecule list ****/

VG_BOOL DbMolListRead(VG_DBINFO *Dbh)
{
  VG_BOOL       Ret;

  if ((!Dbh -> MolList) &&
      ((Dbh -> MolList = ListNew(sizeof(VG_DBMOLLIST))) == NULL))
    return FALSE;

  switch(Dbh -> Type) {
  case VG_DB_TYPE_FILE:         /* File database */
    Ret = DbFileMolListRead(Dbh);
    break;

  case VG_DB_TYPE_SDF:          /* SDF database */
    Ret = DbSdfMolListRead(Dbh);
    break;

  case VG_DB_TYPE_ZIP:          /* Zip database */
    Ret = DbZipMolListRead(Dbh);
    break;
  } /* End of switch */

  if (!Ret) CatErr(MSG_ERR_DBENGINE_MOLLIST);

  return Ret;
}


/**** Remove an item in the molecule list ****/

void DbMolListRemove(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm)
{
  ListRemove((VG_LIST *)Dbh -> MolList, (VG_LISTITEM *)Itm);
  if (!Dbh -> MolList -> Beg) {
    FREE((VG_LIST *)Dbh -> MolList);
    Dbh -> MolList = NULL;
  }
  --Dbh -> TotMol;
}


/**** Create a new database ****/

VG_DBINFO *DbOpen(char *FileName, VG_BOOL Verbose)
{
  char          Buf[256];
  FILE          *FH;
  VG_ULONG      j, k;

  VG_BOOL       Ret  = FALSE;
  VG_DBINFO     *Dbh = NULL;
  VG_LONG       Type = VG_DB_TYPE_UNK;

  /**** Format recognition ****/

  j = strlen(DbDataStr);
  k = strlen(FileName);
  if ((k >= j) && (!strcasecmp(FileName + k - j, DbDataStr)))
    Type = VG_DB_TYPE_FILE;
  else if (((FH = fopen(FileName, "rb")) != NULL) &&
           (fread(&j, sizeof(VG_ULONG), 1, FH) == 1)) {
#ifdef LITTLE_ENDIAN
    if (j == 0x04034b50)
#else
    if (j == 0x504b0304)
#endif
      Type = VG_DB_TYPE_ZIP;
    fclose(FH);
  }

  if ((Type == VG_DB_TYPE_UNK) &&
      ((FH = fopen(FileName, "r")) != NULL)) {
    for (k = 0; (k < 5) && (fgets(Buf, 255, FH)); ++k) {
      if ((k == 4) &&
          (Buf[5]  == '.') && (Buf[15] == '.') &&
          (Buf[25] == '.') && (Buf[10] == ' ') &&
          (Buf[20] == ' ') && (Buf[30] == ' ')) {
        Type = VG_DB_TYPE_SDF;
      }
    } /* End of for */
    fclose(FH);
  }

  if (Type != VG_DB_TYPE_UNK) {
    if ((Dbh = (VG_DBINFO *)Alloca(sizeof(VG_DBINFO))) != NULL) {
#ifdef WIN32
      GetFullPathName(FileName, VG_MAX_PATH, Dbh -> FileName, NULL);
#else
      strcpy(Dbh -> FileName, FileName);
#endif
      Dbh -> Type = Type;
      switch(Type) {
      case VG_DB_TYPE_FILE:         /* File database */
        Ret = DbFileOpen(Dbh);
        break;

      case VG_DB_TYPE_SDF:          /* SDF database */
        Ret = DbSdfOpen(Dbh);
        break;

      case VG_DB_TYPE_ZIP:          /* Zip database */
        Ret = DbZipOpen(Dbh);
        break;
      } /* End of switch */

      /**** Read the file list ****/

      if (Ret) {
        Dbh -> Id = clock();
        Ret = DbMolListRead(Dbh);
      }

      if (!Ret) {
        CatErr(MSG_ERR_DBENGINE_UNKFOR);
        FREE(Dbh);
        Dbh = NULL;
      } else if (Verbose) CatPrintf(stdout, MSG_DBENGINE_OPEN, Dbh -> FileName);
    }
  } else CatErr(MSG_ERR_DBENGINE_UNKFOR);

  return Dbh;
}


/**** Put a molecule into the database ****/

VG_BOOL DbPut(VG_DBINFO *Dbh, char *MolName, VG_BOOL Update)
{
  VG_BOOL               Ret;
  VG_DBMOLLIST          *Itm;
  VG_UBYTE              ConSav, FixSav;
  VG_ULONG              Offset, Msg;

  if (!BegAtm) {
    if (Update) Msg = MSG_WARN_DBENGINE_PUT;
    else Msg = MSG_WARN_DBENGINE_UPDATE;
    PrintProg(Msg);
    return FALSE;
  }

  ConSav         = GLOBSW_CONNSAV;
  FixSav         = GLOBSW_FIXSAV;
  GLOBSW_CONNSAV = (Dbh -> MolFlags & VG_DB_MOLFLAG_CONN) ? 1: 0;
  GLOBSW_FIXSAV  = (Dbh -> MolFlags & VG_DB_MOLFLAG_FIX) ? 1: 0;

  switch(Dbh -> Type) {
  case VG_DB_TYPE_FILE:         /* File database */
    Ret = DbFilePut(Dbh, MolName);
    break;

  case VG_DB_TYPE_SDF:          /* SDF database */
    Ret = DbSdfPut(Dbh, MolName, &Offset);
    break;

  case VG_DB_TYPE_ZIP:          /* Zip database */
    Ret = DbZipPut(Dbh, MolName);
    break;
  } /* End of switch */

  GLOBSW_CONNSAV = ConSav;
  GLOBSW_FIXSAV  = FixSav;

  if ((Ret) && (!Update)){
    if ((Itm = DbMolListAdd(Dbh, MolName)) != NULL) {
      if (Dbh -> Type == VG_DB_TYPE_SDF)
        Itm -> Offset = Offset;
    } else Ret = FALSE;
  }

  if (!Ret) {
    if (Update) Msg = MSG_ERR_DBENGINE_UPDATE;
    else Msg = MSG_ERR_DBENGINE_PUT;
    CatErr(Msg, MolName);
  } else {
    strcpy(Dbh -> CurMolName, MolName);
    if (Update) Msg = MSG_DBENGINE_UPDATE;
    else Msg = MSG_DBENGINE_PUT;
    CatPrintf(stdout, Msg, MolName);
  }

  return Ret;
}


/**** Remove a molecule ****/

VG_BOOL DbRemove(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm)
{
  VG_BOOL       Ret = FALSE;

  switch(Dbh -> Type) {
  case VG_DB_TYPE_FILE:         /* File database */
    Ret = DbFileRemove(Dbh, Itm);
    break;

  case VG_DB_TYPE_SDF:          /* SDF database */
    Ret = DbSdfRemove(Dbh, Itm);
    break;

  case VG_DB_TYPE_ZIP:          /* Zip database */
    Ret = DbZipRemove(Dbh, Itm);
    break;
  } /* End of switch */

  if (Ret) {
    CatPrintf(stdout, MSG_DBENGINE_REMOVE, Itm -> MolName);
    DbMolListRemove(Dbh, Itm);
    strcpy(Dbh -> CurMolName, Itm -> MolName);
  } else CatErr(MSG_ERR_DBENGINE_REMOVE, Itm -> MolName);

  return Ret;
}


/**** Rename a molecule ****/

VG_BOOL DbRen(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm, char *NewName)
{
  VG_BOOL       Ret = FALSE;

  if (*NewName) {
    switch(Dbh -> Type) {
    case VG_DB_TYPE_FILE:         /* File database */
      Ret = DbFileRen(Dbh, Itm, NewName);
      break;

    case VG_DB_TYPE_SDF:          /* SDF database */
      Ret = DbSdfRen(Dbh, Itm, NewName);
      break;

    case VG_DB_TYPE_ZIP:          /* Zip database */
      Ret = FALSE;
      break;
    } /* End of switch */
  }

  if (Ret) {
    CatPrintf(stdout, MSG_DBENGINE_REN, Itm -> MolName, NewName);
    strcpy(Itm -> MolName   , NewName);
    strcpy(Dbh -> CurMolName, NewName);
  } else CatErr(MSG_ERR_DBENGINE_REN, Itm -> MolName);

  return Ret;
}





