
/*************************************************
****   VEGA - Interaction energy calculation  ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * This code calculates the energy interaction between two molecules
 * (e.g. a ligand and a macromolecule) using the CVFF force field
 * (Copyright Biosym/MSI).
 */


#include <math.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "comerr.h"
#include "count.h"
#include "ener.h"

#define  FastSgn(X)             (int)(*((int *)&X) & 0x80000000)


/**** Calculate non-bond interations between two molecules ****/

VG_BOOL CalcEnerCVFF(FILE *OUT, ATOMO *InizAtm, VG_ULONG TotAtm,
                  VG_LONG NomeRes, VG_LONG NRes, VG_INTENE **Table,
                  float *TotEner)
{
  ATOMO         *Atm1, *Atm2, *InizRes;
  char          File[256];
  FILE          *FH;
  float         Dist, TmpA, TmpB;
  float         *A, *Ai, *Aj, *B, *Bi, *Bj, *Ptr;
  float         Sgn;
  VG_OCHAR  Type;
  VG_INTENE     *Ene, *AttEne, TmpEne;
  VG_LONG       PrecResName, PrecResSeq;
  VG_ULONG      k, j, i;

  VG_BOOL       NbChk       = FALSE;
  VG_BOOL       Ret         = TRUE;
  float         CutOff      = Quad(Prefs.ENCUTOFF);
  float         Diel        = Prefs.ENDIEL;
  float         TotalQ      = 0.0f;
  float         TotalNB     = 0.0f;
  VG_ULONG      Count       = 0;
  VG_ULONG      NAtmRes     = 0;

  if (ChkForceField(InizAtm, TotAtm)) {
    if (ChkCharges(InizAtm, TotAtm, NULL)) {
      if ((A = (float *)Alloca(TotAtm * sizeof(float)))) {
        if ((B = (float *)Alloca(TotAtm * sizeof(float)))) {
          PrintProg(MSG_ENER_ASSPROG);
          GetPrgPath(File, TRUE);
          strcat(File, CVFF_PAR);

          /**** Template loader ****/

          if ((FH = fopen(File, "r"))) {
            if (fgets(File, 255, FH)) {
              if ((*File != ';') && (!strncasecmp(File, "#FF_Param CVFF", 14))) {
                while(fgets(File, 255, FH)) {
                  if (!strncasecmp(File, "#NonBond", 8)) {
                    NbChk = TRUE;
                    while((fgets(File, 255, FH)) && (Count != TotAtm)) {
                      if ((*File != ';') && (*File != '\n')) {
                        Type.DL = 0;
                        sscanf(File, "%s %f %f", Type.C, &TmpA, &TmpB);
                        Ai = A;
                        Bi = B;
                        for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
                          if (Atm1 -> Pot.DL == Type.DL) {
                            ++Count;
                            *Ai = TmpA;
                            *Bi = TmpB;
                          }
                          ++Ai;
                          ++Bi;
                        } /* End of for */
                      }
                    } /* End of while */
                    if (TotAtm == Count) {
                      if ((NAtmRes = FindResPtr(InizAtm, &InizRes, NomeRes, NRes))) {
                        Count = CountRes(InizAtm) - 1;
                        if (Count > 0) {
                           if ((Ene = (VG_INTENE *)Alloca(sizeof(VG_INTENE) * (Count + 1)))) {
                             PrintProg(MSG_ENER_CALCPROG);
                             Ai          = A;
                             AttEne      = Ene;
                             Bi          = B;
                             PrecResName = 0;
                             PrecResSeq  = 0;
                             for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
                               if (Atm1 == InizRes) {
                                 for(k = 0; k < NAtmRes; ++k) {
                                   Atm1 = Atm1 -> Ptr;
                                   ++Ai;
                                   ++Bi;
                                 } /* End of for */
                                 if (!Atm1) break;
                               }
                               if ((Atm1 -> ResName.L != PrecResName) ||
                                   (Atm1 -> ResSeq.L != PrecResSeq)) {
                                 if (PrecResName) ++AttEne;
                                 AttEne -> InizRes = Atm1;
                                PrecResName = Atm1 -> ResName.L;
                                PrecResSeq  = Atm1 -> ResSeq.L;
                              }

                              Atm2 = InizRes;
                              Aj   = &A[InizRes -> Num - 1];
                              Bj   = &B[InizRes -> Num - 1];
                              for(k = 0; k < NAtmRes; ++k) {

                                 /**** Square radius ****/

                                Dist = Quad(Atm1 -> x - Atm2 -> x) +
                                       Quad(Atm1 -> y - Atm2 -> y) +
                                       Quad(Atm1 -> z - Atm2 -> z);

                                if (Dist <= CutOff) {

                                  /**** Non-bond term ****/

                                  TmpA  = vgCube(Dist);
                                  TmpB  = (SQR((*Ai) * (*Aj)) / Quad(TmpA)) -
                                          (SQR((*Bi) * (*Bj)) / TmpA);
                                  AttEne -> NB += TmpB;
                                  TotalNB      += TmpB;

                                  /**** Coulombic term ****/

                                  TmpA         = CONV_COUL * (Atm1 -> Charge * Atm2 -> Charge) /
                                                 (Diel * SQR(Dist));
                                  AttEne -> C += TmpA;
                                  TotalQ      += TmpA;
                                }
                                Atm2 = Atm2 -> Ptr;
                                ++Aj;
                                ++Bj;
                              } /* End of for */
                              ++Ai;
                              ++Bi;
                            }

                            /**** Sorting energies ****/

                            PrintProg(MSG_ENER_SORTPROG);
                            for(k = 0; k < (Count - 1); ++k) {
                              i    = k;
                              TmpA = Ene[k].NB + Ene[k].C;
                              for(j = k; j < Count; ++j) {
                                TmpB = Ene[j].NB + Ene[j].C;
                                if (TmpB < TmpA) {
                                  TmpA = TmpB;
                                  i = j;
                                }
                              } /* End of for */
                              memcpy(&TmpEne, &Ene[k], sizeof(VG_INTENE));
                              memcpy(&Ene[k], &Ene[i], sizeof(VG_INTENE));
                              memcpy(&Ene[i], &TmpEne, sizeof(VG_INTENE));
                            } /* End of for */

                            /**** Print results ****/

                            if (CatPrintf(OUT, MSG_ENER_RESULTS,
                                InizRes -> ResName.C, InizRes -> ResSeq.C, NAtmRes,
                                Diel, TotalNB + TotalQ, TotalNB, TotalQ) > 0) {
                              TotalNB += TotalQ;
                              for(k = 0, AttEne = Ene; (Ret) && (k < Count); ++k, ++AttEne) {
                                Atm1 = AttEne -> InizRes;
                                TmpA = AttEne -> NB + AttEne -> C;
                                if (!k) Sgn = FastSgn(TmpA);
                                TmpB = (TmpA / TotalNB) * 100.0f;
                                if (fabs(TmpB) < Prefs.ENFILTER) continue;
                                if (FastSgn(TmpA) != Sgn) {
                                  if (LocPrintf(OUT, "\n") <= 0) Ret = PrintDosErr();
                                  Sgn = FastSgn(TmpA);
                                }
                                if ((Ret) && (LocPrintf(OUT, " %-4.4s  %4.4s   %8.4f   %8.4f   %8.4f\n",
                                              Atm1 -> ResName.C, Atm1 -> ResSeq.C, AttEne -> NB,
                                              AttEne -> C, TmpA) <= 0))
                                  Ret = PrintDosErr();
                              } /* End of for */
                              if (LocPrintf(OUT, "\n\n") <= 0) Ret = PrintDosErr();
                            } else Ret = PrintDosErr();
                            if ((Table) && (Ret)) {
                              *Table   = Ene;
                              *TotEner = TotalNB;
                            } else FREE(Ene);
                          } else Ret = FALSE;
                        } else Ret = CatErr(MSG_ERR_ENER_MORERES);
                      } else Ret = CatErr(MSG_ERR_ENER_RESNOTFOUND);
                    } else {
                      Ptr = A;
                      for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr) {
                        if (*Ptr == 0.0f) {
                          GlobErr = VG_PARET_CALENUNKTYP;
                          sprintf(File, GetStr(MSG_ERR_ENER_UNKATMTYPE), Atm1 -> Pot.C);
                          PrintAtmErr(File, Atm1, TRUE);
                          Ret = FALSE;
                        }
                        ++Ptr;
                      } /* Endo of for */
                    }
                    break;
                  }
                }
                if (!NbChk) Ret = CatErr(MSG_ERR_ENER_NONBOND);
              }
            }
            fclose(FH);
          } else Ret = CatErr(MSG_ERR_ENER_FFNOTFOUND, CVFF_PAR);
          FREE(B);
        } else Ret = FALSE;
        FREE(A);
      } else Ret = FALSE;
    } else Ret = CatErr(MSG_ERR_DCD_UNCHARGED);
  } else Ret = CatErr(MSG_ERR_ENER_FFNOTASS);

  return Ret;
}

