
/*************************************************
****     VEGA - Escher NG output analyzer     ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"
#include "traj.h"

/**** Constants ****/

#define  ES_LINELEN             255

/**** Local definitions ****/

typedef struct {
  float             Dx, Dy, Dz;         /* Transaltion                 */
  float             Ax, Ay, Az;         /* Rotation                    */
} VG_ESTRAJ;

typedef struct {
  VG_ESTRAJ     *TrjPtr;                /* Trajectory pointer          */
  VG_ESTRAJ     *CurFrm;                /* Current frame               */
  ATOMO         *ProbeBegAtm;           /* Probe atom pointer          */
  VG_ULONG      ProbeTotAtm;            /* Probe atoms                 */
  XYZ           *ProbeCoord;            /* Probe starting coordinates  */
  ATOMO         *TargetBegAtm;          /* Target atom pointer         */
  VG_ULONG      TargetTotAtm;           /* Target atoms                */
  XYZ           RotCent;                /* Rotation center             */
} VG_ESINFO;


/**** Free all resources ****/

void EscherClose(TRJINFO *Info)
{
  if (Info -> FH) {
    if (((VG_ESINFO *)Info -> FH) -> TrjPtr)
      FREE(((VG_ESINFO *)Info -> FH) -> TrjPtr);

    if (((VG_ESINFO *)Info -> FH) -> ProbeCoord)
      FREE(((VG_ESINFO *)Info -> FH) -> ProbeCoord);

    FREE(Info -> FH);
  }
}


/**** Open the trajectory file ****/

VG_BOOL EscherOpen(char *FileName, TRJINFO *Info)
{
  ATOMO                 *LastSeg;
  char                  Buf[ES_LINELEN + 1], MolName[256];
  char                  *Ptr;
  FILE                  *FH;
  float                 *Ener;
  VG_ESTRAJ             *Trj;
  VG_ULONG              i;
  XYZ                   *ProbePtr;

  VG_ULONG              Line = 0;
  VG_BOOL               Ret  = TRUE;

  BioDockChangeDir(FileName);

  if ((Info -> FH = (FILE *)Alloca(sizeof(VG_ESINFO))) != NULL) {
    if ((FH = PkOpen(FileName, "r", 0, TRUE)) != NULL) {
      while((Ret) && (fgets(Buf, ES_LINELEN, FH))) {
        if (++Line == 1) {
          if (strncmp(Buf, "#ESCHERNG_VER", 13))
            Ret = CatErr(MSG_ERR_ESCHER_UNKFORMAT);
        } else if (!strncmp(Buf, "#DOCKING_INFO", 13)) {

          /**** Target molecule ****/

          if (fgets(Buf, ES_LINELEN, FH)) {
            if (BioDockGetFileName(MolName, Buf)) {
              ((VG_ESINFO *)Info -> FH) -> TargetBegAtm = Loader(MolName,
                             &((VG_ESINFO *)Info -> FH) -> TargetTotAtm, NULL, NULL, 1,
                             &BegSrf, &TotalSrf);
              if ((((VG_ESINFO *)Info -> FH) -> TargetBegAtm) &&
                  (((VG_ESINFO *)Info -> FH) -> TargetBegAtm != (ATOMO *)-1) &&
                  (((VG_ESINFO *)Info -> FH) -> TargetBegAtm != (ATOMO *)-2)) {
                LastSeg = LastAtm;

                /**** Probe molecule ****/

                if (fgets(Buf, ES_LINELEN, FH)) {
                  if (BioDockGetFileName(MolName, Buf)) {
                    ((VG_ESINFO *)Info -> FH) -> ProbeBegAtm = Loader(MolName,
                                    &((VG_ESINFO *)Info -> FH) -> ProbeTotAtm, NULL, NULL, 2,
                                    &BegSrf, &TotalSrf);
                    if ((((VG_ESINFO *)Info -> FH) -> ProbeBegAtm) &&
                        (((VG_ESINFO *)Info -> FH) -> ProbeBegAtm != (ATOMO *)-1) &&
                        (((VG_ESINFO *)Info -> FH) -> ProbeBegAtm != (ATOMO *)-2)) {
                      TotalAtm = ((VG_ESINFO *)Info -> FH) -> TargetTotAtm +
                                 ((VG_ESINFO *)Info -> FH) -> ProbeTotAtm;
                      Info -> MolAtm = TotalAtm;
                      BegAtm = ((VG_ESINFO *)Info -> FH) -> TargetBegAtm;
                      LastSeg -> Ptr = ((VG_ESINFO *)Info -> FH) -> ProbeBegAtm;
                      ((VG_ESINFO *)Info -> FH) -> ProbeBegAtm -> Prev = LastSeg;

                      /**** Read the number of frames ****/

                      if (fgets(Buf, ES_LINELEN, FH)) {
                        if ((Ptr = strchr(Buf, ':')) != NULL) {
                          sscanf(Ptr + 1, "%d", &Info -> Frames);
                          if (Info -> Frames > 0) {
                            Info -> EneFrames = Info -> Frames;
                            Info -> EneStep   = 0;
                            Info -> StartTime = 0;
                            Info -> FrmStep   = 0;
                            Info -> Temp      = 0.0f;
                            if (((((VG_ESINFO *)Info -> FH) -> TrjPtr = (VG_ESTRAJ *)Alloca(Info -> Frames * sizeof(VG_ESTRAJ))) != NULL) &&
                                (Info -> Energy = (float *)Alloca(Info -> EneFrames * sizeof(float)))) {

                              /**** Rotation center ****/

                              if ((fgets(Buf, ES_LINELEN, FH)) && ((Ptr = strchr(Buf, ':')) != NULL)) {
                                sscanf(Ptr + 1, "%f %f %f",
                                                &((VG_ESINFO *)Info -> FH) -> RotCent.x,
                                                &((VG_ESINFO *)Info -> FH) -> RotCent.y,
                                                &((VG_ESINFO *)Info -> FH) -> RotCent.z);
                                if ((((VG_ESINFO *)Info -> FH) -> ProbeCoord = (XYZ *)Alloca(((VG_ESINFO *)Info -> FH) -> ProbeTotAtm * sizeof(XYZ))) != NULL) {
                                  ProbePtr = ((VG_ESINFO *)Info -> FH) -> ProbeCoord;
                                  for(LastSeg = ((VG_ESINFO *)Info -> FH) -> ProbeBegAtm;
                                      LastSeg; LastSeg = LastSeg -> Ptr) {
                                    ProbePtr -> x = LastSeg -> x - ((VG_ESINFO *)Info -> FH) -> RotCent.x;
                                    ProbePtr -> y = LastSeg -> y - ((VG_ESINFO *)Info -> FH) -> RotCent.y;
                                    ProbePtr -> z = LastSeg -> z - ((VG_ESINFO *)Info -> FH) -> RotCent.z;
                                    ++ProbePtr;
                                  } /* End of for */
                                } else Ret = FALSE;
                              } else Ret = CatErr(MSG_ERR_ESCHER_CORRFILE);
                            }
                          } else Ret = CatErr(MSG_ERR_ESCHER_NUMSOL);
                        } else Ret = CatErr(MSG_ERR_ESCHER_CORRFILE);
                      } else Ret = PrintDosErr();
                    } else Ret = FALSE;
                  } else Ret = FALSE;
                } else Ret = PrintDosErr();
              } else Ret = FALSE;
            } else Ret = FALSE;
          } else Ret = PrintDosErr();
        } else if (!strncmp(Buf, "#SOLUTIONS", 10)) {
          if ((Info -> Energy) && ((Ret = SkipLines(FH, Buf, 2)) == TRUE)) {
            i    = 0;
            Ener = Info -> Energy;
            Trj  = ((VG_ESINFO *)Info -> FH) -> TrjPtr;
            while((Ret) && (fgets(Buf, ES_LINELEN, FH)) && (*Buf != '#')) {
              if (i < Info -> Frames) {
                sscanf(Buf, "%*s %f %*s %*s %*s "
                            "%*s %*s %*s %*s "
                            "%f %f %f "
                            "%f %f %f\n",
                            Ener,
                            &Trj -> Ax , &Trj -> Ay , &Trj -> Az,
                            &Trj -> Dx, &Trj -> Dy, &Trj -> Dz);
                Trj -> Ax *= DEG_TO_RAD;
                Trj -> Ay *= DEG_TO_RAD;
                Trj -> Az *= DEG_TO_RAD;
                ++Ener;
                ++Trj;
                ++i;
              } else Ret = CatErr(MSG_ERR_ESCHER_CORRFILE);
            } /* End of while */
          } else Ret = CatErr(MSG_ERR_ESCHER_CORRFILE);
        }
      } /* End of while */
      PkClose(FH);
    } else Ret = FALSE;
  } else Ret = FALSE;

  if (!Ret) EscherClose(Info);

  return Ret;
}


/**** Read a frame ****/

VG_BOOL EscherReadFrm(TRJINFO *Info)
{
  float         Rad, Angle;

  ATOMO         *Atm      = ((VG_ESINFO *)Info -> FH) -> ProbeBegAtm;
  VG_ESTRAJ     *Trj      = ((VG_ESINFO *)Info -> FH) -> CurFrm;
  float         TransX    = ((VG_ESINFO *)Info -> FH) -> RotCent.x + Trj -> Dx;
  float         TransY    = ((VG_ESINFO *)Info -> FH) -> RotCent.y + Trj -> Dy;
  float         TransZ    = ((VG_ESINFO *)Info -> FH) -> RotCent.z + Trj -> Dz;
  XYZ           *ProbePtr = ((VG_ESINFO *)Info -> FH) -> ProbeCoord;

  while(Atm) {
    Atm -> x = ProbePtr -> x;
    Atm -> y = ProbePtr -> y;
    Atm -> z = ProbePtr -> z;

    /**** X rotation ****/

    Rad      = SQR(Quad(Atm -> z) + Quad(Atm -> y));
    Angle    = Trj -> Ax + atan2(Atm -> y, Atm -> z);
    Atm -> z = Rad * cos(Angle);
    Atm -> y = Rad * sin(Angle);

    /**** Y rotation ****/

    Rad      = SQR(Quad(Atm -> z) + Quad(Atm -> x));
    Angle    = Trj -> Ay + atan2(Atm -> x, Atm -> z);
    Atm -> z = Rad * cos(Angle);
    Atm -> x = Rad * sin(Angle);

    /**** Z rotation ****/

    Rad      = SQR(Quad(Atm -> x) + Quad(Atm -> y));
    Angle    = Trj -> Az + atan2(Atm -> y, Atm -> x);
    Atm -> x = Rad * cos(Angle);
    Atm -> y = Rad * sin(Angle);

    /**** Translation ****/

    Atm -> x += TransX;
    Atm -> y += TransY;
    Atm -> z += TransZ;

    Atm = Atm -> Ptr;
    ++ProbePtr;
  }

  return TRUE;
}


/**** Seek a trajectory frame ****/

VG_BOOL EscherSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  switch(Mode) {
  case SEEK_CUR:
    ((VG_ESINFO *)Info -> FH) -> CurFrm = ((VG_ESINFO *)Info -> FH) -> CurFrm + Frames;
    break;

  case SEEK_END:
    ((VG_ESINFO *)Info -> FH) -> CurFrm = ((VG_ESINFO *)Info -> FH) -> CurFrm +
                                          ((VG_ESINFO *)Info -> FH) -> ProbeTotAtm +
                                          Frames;
    break;

  case SEEK_SET:
    ((VG_ESINFO *)Info -> FH) -> CurFrm = ((VG_ESINFO *)Info -> FH) -> TrjPtr + Frames;
    break;
  } /* End of switch */

  return TRUE;
}
