
/*************************************************
****    VEGA - Math & geometry subroutines    ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <math.h>

#include "globdef.h"
#include "globvar.h"


/**** Mesures the bond angle ****/

double BondAngle(ATOMO *a, ATOMO *b, ATOMO *c)
{
  double  Angle;
  double  Dist;
  double  Cos_Theta;

  Dist = (double)Distance(a, b) * (double)Distance(b, c);
  if (Dist) {
    Cos_Theta = ((a -> x - b -> x) * (c -> x - b -> x) + (a -> y - b -> y) * (c -> y - b -> y) +
           (a -> z - b -> z) * (c -> z - b -> z)) / Dist;
    if ((Cos_Theta  + (double)1.0) < (double)0.0001)
      Angle = (double)180.0;
    else Angle = (acos(Cos_Theta)) * RAD_TO_DEG;
  } else Angle = 0.0;

  return Angle;
}

/**** Mesures the bond angle (single precision version) ****/

float BondAnglef(ATOMO *a, ATOMO *b, ATOMO *c)
{
  float  Angle, Cos_Theta, Dist;

  Dist = Distance(a, b) * Distance(b, c);
  if (Dist) {
    Cos_Theta = ((a -> x - b -> x) * (c -> x - b -> x) + (a -> y - b -> y) * (c -> y - b -> y) +
           (a -> z - b -> z) * (c -> z - b -> z)) / Dist;
    if ((Cos_Theta  + 1.0f) < 0.0001f) Angle = 180.0f;
    else Angle = acos(Cos_Theta) * RAD_TO_DEG;
  } else Angle = 0.0f;

  return Angle;
}


/**** Vectorial cross product ****/

void CrossProd(VECTYPE *vect1, VECTYPE *vect2, VECTYPE *normal)
{
  normal -> x = vect1 -> y * vect2 -> z - vect2 -> y * vect1 -> z;
  normal -> y = vect2 -> x * vect1 -> z - vect1 -> x * vect2 -> z;
  normal -> z = vect1 -> x * vect2 -> y - vect2 -> x * vect1 -> y;
}


double Magnitude(VECTYPE *vect1)
{
  double  Mag;

  Mag = SQR(Quad(vect1 -> x) + Quad(vect1 -> y) + Quad(vect1 -> z));

  return Mag;
}


/**** Determinat of 2x2 matrix ****/

float Det2x2(float A, float B, float C, float D)
{
  return (A * D - B * C);
}


/**** Dot product ****/

double Dot(VECTYPE *vect1, VECTYPE *vect2)
{
  double  Dot_prod;

  Dot_prod = vect1 -> x * vect2 -> x + vect1 -> y * vect2 -> y + vect1 -> z *
  vect2 -> z;
  return Dot_prod;
}


/**** Calculate the angle between two planes ****/
/**** starting from plane equations          ****/

float PlaneAngle(PLANE *P0, PLANE *P1)
{
   return acos((P0 -> a * P1 -> a + P0 -> b * P1 -> b + P0 -> c * P1 -> c) /
               (SQR(Quad(P0 -> a) + Quad(P0 -> b) + Quad(P0 -> c)) * \
                SQR(Quad(P1 -> a) + Quad(P1 -> b) + Quad(P1 -> c))));
}


/**** Calculate the angle between two planes ****/
/**** starting from atomic coordinates        ****/

float PlaneAngAtm(ATOMO **Atm)
{
  PLANE         P0, P1;

  PlaneCof(&P0, Atm[0], Atm[1], Atm[2]);
  PlaneCof(&P1, Atm[3], Atm[4], Atm[5]);

  return (float)PlaneAngle(&P0, &P1);
}


/**** Calculate the equation coefficients of plene through 3 atoms ****/

void PlaneCof(PLANE *Plane, ATOMO *A0, ATOMO *A1, ATOMO *A2)
{
  Plane -> a = Det2x2(A1 -> y - A0 -> y, A1 -> z - A0 -> z,
                      A2 -> y - A0 -> y, A2 -> z - A0 -> z);
  Plane -> b = Det2x2(A1 -> x - A0 -> x, A1 -> z - A0 -> z,
                      A2 -> x - A0 -> x, A2 -> z - A0 -> z);
  Plane -> c = Det2x2(A1 -> x - A0 -> x, A1 -> y - A0 -> y,
                      A2 -> x - A0 -> x, A2 -> y - A0 -> y);
  Plane -> d = A0 -> y * Plane -> b - A0 -> x * Plane -> a -
               A0 -> z * Plane -> c;
}


void PointToVect(ATOMO *pt, VECTYPE *v)
{
  v -> x = pt -> x;
  v -> y = pt -> y;
  v -> z = pt -> z;
}


/**** Mesure a torsion angle ****/

double Torsion(ATOMO *a, ATOMO *b, ATOMO *c, ATOMO *d)
{
  VECTYPE               b1, b2, b3;
  VECTYPE               c1, c2, c3;
  double                l1, l2;
  VECTYPE               v1, v2, v3, v4;
  double                angle;

  PointToVect(a, &v1);
  PointToVect(b, &v2);
  PointToVect(c, &v3);
  PointToVect(d, &v4);

  VectDiff(&v1, &v2, &b1);
  VectDiff(&v2, &v3, &b2);
  VectDiff(&v3, &v4, &b3);

  CrossProd(&b1, &b2, &c1);
  CrossProd(&b2, &b3, &c2);
  CrossProd(&c1, &c2, &c3);

  l1 = Magnitude(&c1);
  l2 = Magnitude(&c2);

  if ((l1 * l2) != 0) {
    angle = VectAng(&c1, &c2);
    if (Dot(&b2, &c3) > 0.0) angle = angle * -1.0;
  } else angle = 0;

  return angle;
}


double VectAng(VECTYPE *vect1, VECTYPE *vect2)
{
  double  angle;
  double  mag;
  double  dp;
  double  rad_ang;

  mag = Magnitude(vect1) * Magnitude(vect2);
  dp  = Dot(vect1, vect2) / mag;
  if (dp < -0.999999) dp = -0.9999999;
  if (dp > 0.9999999) dp = 0.9999999;
  rad_ang = acos(dp);
  angle   = RAD_TO_DEG * rad_ang;

  return angle;
}


void VectDiff(VECTYPE *vect1, VECTYPE *vect2, VECTYPE *vect_sm)
{
  vect_sm -> x = vect1 -> x - vect2 -> x;
  vect_sm -> y = vect1 -> y - vect2 -> y;
  vect_sm -> z = vect1 -> z - vect2 -> z;
}

