
/*************************************************
****    VEGA - Color managment for OpenGL     ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <gl\gl.h>
#include <stdio.h>

#ifdef __BORLANDC__
#pragma  hdrstop
#endif

#include "globdef.h"
#include "globvar.h"

#define  __COLORTABLE
#include "gl_global.h"
#include "gl_colors.h"

#define  VG_DEFAULTCOL          VGCOL_MAGENTA


/**** Color an atom ****/

void GL_ColorAtm(register ATOMO *Atm)
{
  register VG_ATMCOL       *AtmCol;

  for(AtmCol = AtmColTbl; (AtmCol -> Elem.S) &&
         (Atm -> Elem.S != AtmCol -> Elem.S); ++AtmCol);
  Atm -> ColorID = AtmCol -> ColorID;
}


/**** Color a molecule by atom ****/

VG_ULONG GL_ColorByAtm(register ATOMO *Atm)
{
  register VG_ATMCOL       *AtmCol;
  register VG_UBYTE        CurCol;

  register VG_ULONG        Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      for(AtmCol = AtmColTbl; (AtmCol -> Elem.S) &&
         (Atm -> Elem.S != AtmCol -> Elem.S); ++AtmCol);
      Atm -> ColorID = AtmCol -> ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Colored;
}


/**** Color by chain ****/

VG_ULONG GL_ColorByChain(register ATOMO *Atm)
{
  VG_BYTE      ChainBuf[256], *ChainPtr;
  VG_UBYTE     ColorID;

  VG_ULONG     Colored   = 0;

#ifdef WIN32
  ZeroMemory(ChainBuf, 256);
#else
  memset(ChainBuf, 0, 256);
#endif

  while(Atm) {
    ColorID = VGCOL_WHITE;
    for(ChainPtr = ChainBuf; (*ChainPtr) && (*ChainPtr != Atm -> ChainID); ++ChainPtr) {
      ++ColorID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_WHITE;
    } /* End of for */
    
    if (!*ChainPtr) *ChainPtr = Atm -> ChainID;

    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color by charge ****/

VG_ULONG GL_ColorByCharge(register ATOMO *Atm)
{
  struct _ColorRanges {
    float       Min, Max;
    VG_UBYTE    ColorID;
  };
  register struct _ColorRanges  *Ptr;
  register VG_ULONG             Colored       = 0;
  const struct _ColorRanges     ColorRanges[] = {{-0.50f, -0.20f, VGCOL_BLUE  },
                                                 {-0.20f, -0.10f, VGCOL_CYAN  },
                                                 {-0.10f,  0.10f, VGCOL_GREEN },
                                                 { 0.10f,  0.20f, VGCOL_YELLOW},
                                                 { 0.20f,  1.01f, VGCOL_RED   },
                                                 { 0.00f,  0.00f, 0           }
                                                };
  while(Atm) {
    if (Atm -> Active) {
      for(Ptr = (struct _ColorRanges *)ColorRanges; Ptr -> ColorID; ++Ptr) {
        if ((Atm -> Charge >= Ptr -> Min) && (Atm -> Charge < Ptr -> Max)) {
          Atm -> ColorID = Ptr -> ColorID;
          break;
        }
      } /* End of for */
      ++Colored;
    }
    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color by constraint ****/

VG_ULONG GL_ColorByConstr(register ATOMO *Atm)
{
  register VG_ULONG     Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      if (Atm -> Fix)
        Atm -> ColorID = VGCOL_BLUE;
      else
        Atm -> ColorID = VGCOL_GREEN;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  }

  return Colored;
}

/**** Color by molecule ****/

VG_ULONG GL_ColorByMol(register ATOMO *Atm)
{
  register VG_UBYTE     ColorID = VGCOL_WHITE;
  register VG_ULONG     Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }

    if (Atm -> Flags & VG_ATMF_MOLEND) {
      ++ColorID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_WHITE;
    }

    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color a molecule by residue ****/

VG_ULONG GL_ColorByRes(ATOMO *InizAtm)
{
  register ATOMO         *Atm;
  register VG_RESCOL     *Col    = ResColTab;
  register VG_ULONG      Colored = 0;
  register VG_ULONG      ToColor = 0;

  for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
    if (Atm -> Active) {
      Atm -> ColorID = VGCOL_WHITE;
      ++ToColor;
    }
  } /* End of for */

  while ((Col -> ColorID) && (Colored != ToColor)) {
    for(Atm = InizAtm; (Atm) && (Colored != ToColor); Atm = Atm -> Ptr) {
      if (Atm -> Active) {
        if (Col == ResColTab) ++ToColor;
        if (Atm -> ResName.L == Col -> ResName.L) {
          Atm -> ColorID = Col -> ColorID;
          ++Colored;
        }
      }
    } /* End of for */
    ++Col;
  } /* End of while */

  return Colored;
}


/**** Color by segment ****/

VG_ULONG GL_ColorBySeg(register ATOMO *Atm)
{
  register VG_UBYTE     ColorID = VGCOL_WHITE;
  register VG_ULONG     Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      if (Atm -> ResName.L == *(VG_LONG *)"HOH")
        Atm -> ColorID = VGCOL_BLUE;
      else
        Atm -> ColorID = ColorID;
      ++Colored;
    }

    if (Atm -> Flags & VG_ATMF_SEGEND) {
      if (++ColorID == VGCOL_BLUE) ++ColorID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_WHITE;
    }

    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color a molecule ****/

VG_ULONG GL_ColorMol(register ATOMO *Atm, register VG_UBYTE ColorID)
{
  register VG_ULONG      Colored = 0;

  if (ColorID > VGCOL_NUM) ColorID = VGCOL_NUM;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Colored;
}


/**** Color a residue ****/

VG_ULONG GL_ColorRes(ATOMO *Atm, VG_UBYTE ColorID)
{
  VG_BYTE       ChainID = Atm -> ChainID;
  VG_LONG       ResName = Atm -> ResName.L;
  VG_LONG       ResSeq  = Atm -> ResSeq.L;
  VG_ULONG      Colored = 0;

  if (ColorID > VGCOL_NUM) ColorID = VGCOL_NUM;

  while((Atm) &&
        (ChainID == Atm -> ChainID) &&
        (ResName == Atm -> ResName.L) &&
        (ResSeq  == Atm -> ResSeq.L)) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Colored;
}


/**** Color surface ****/

void GL_ColorSrf(register VG_SURFACE *Dot, VG_UBYTE ColorID)
{
  register VG_COLOR     *Col;

  if (ColorID > VGCOL_NUM) ColorID = VGCOL_NUM;
  Col = &ColorTable[ColorID];

  while(Dot) {
    Dot -> Color[0] = Col -> R;
    Dot -> Color[1] = Col -> G;
    Dot -> Color[2] = Col -> B;
    Dot = Dot -> Next;
  }
}

/**** Color surface by atom ****/

void GL_ColorSrfAtm(register VG_SURFACE *Dot, register ATOMO *Atm)
{
  register VG_ATMCOL    *AtmCol;
  VG_UBYTE              R, G, B;

  register VG_ULONG     PrecAtmNum = 0;

  while(Dot) {
    if (Dot -> AtmNum != PrecAtmNum) {

      /**** Search the atom ****/

      while((Dot -> AtmNum != Atm -> Num) && (Atm))
        Atm = Atm -> Ptr;

      /**** Search the color ****/

      for(AtmCol = AtmColTbl; (AtmCol -> Elem.S) &&
       (Atm -> Elem.S != AtmCol -> Elem.S); ++AtmCol);

      R = ColorTable[AtmCol -> ColorID].R;
      G = ColorTable[AtmCol -> ColorID].G;
      B = ColorTable[AtmCol -> ColorID].B;
    }
    Dot -> Color[0] = R;
    Dot -> Color[1] = G;
    Dot -> Color[2] = B;
    PrecAtmNum = Dot -> AtmNum;
    Dot = Dot -> Next;
  } /* End of dot loop */
}


/**** Color surface by color ****/

void GL_ColorSrfCol(VG_SURFACE *Dot, VG_ULONG Color)
{
  GL_ColorSrfColRGB(Dot, Color & 0xff, (Color & 0xff00) / 256,
                   (Color & 0xff0000) / 65536);
}


/**** Color surface by color (RGB version) ****/

void GL_ColorSrfColRGB(register VG_SURFACE *Dot, VG_UBYTE R, VG_UBYTE G, VG_UBYTE B)
{
  while(Dot) {
    Dot -> Color[0] = R;
    Dot -> Color[1] = G;
    Dot -> Color[2] = B;
    Dot = Dot -> Next;
  } /* End of while */
}


/**** Get the color ID by name ****/

VG_UBYTE GL_GetColorId(register char *ColStr)
{
  register VG_COLOR      *Color;

  for(Color = ColorTable + 1; Color -> Name; ++Color) {
    if (!strcasecmp(Color -> Name, ColStr)) {
      return Color -> ID;
    }
  } /* End of for */

  return VGCOL_NONE;
}


/**** Get the color ID by RGB value ****/

VG_UBYTE GL_GetColorIdRGB(register VG_ULONG Col)
{
  register VG_COLOR      *Color;

  for(Color = ColorTable + 1; Color -> Name; ++Color) {
    if ((VG_ULONG)(Color -> R + (Color -> G << 8) + (Color -> B << 16)) == Col) {
      return Color -> ID;
    }
  } /* End of for */

  return VGCOL_NONE;
}


