
/*************************************************
****    VEGA - Matrix functions for OpenGL    ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <string.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "gl_matrix.h"

/**** Macros ****/

#define MATSWAP(a,b) {temp=(a);(a)=(b);(b)=temp;}


/**** Copy the matrix to another ****/

void GL_Mtx4Copy(register float Dest[4][4], register float Src[4][4])
{
  register int          k;

  for(k = 0; k < 16; ++k)
    ((float *)Dest)[k] = ((float *)Src)[k];
}


/**** Calculate the inverse 4x4 matrix ****/

int GL_Mtx4Inverse(float matr[4][4])
{
  float         ident[4][4];
  int           i, j, k, l, ll;
  int           indxc[4], indxr[4], ipiv[4];
  float         big, dum, pivinv, temp;

  int           icol = 0;
  int           irow = 0;

  for(i = 0; i < 4; i++) {
    for(j = 0; j < 4; j++) {
      ident[i][j] = 0.0f;
    } /* End of for */
    ident[i][i]=1.0f;
  } /* End of for */

  for(j = 0; j <= 3; j++) ipiv[j] = 0;
  for(i = 0; i <= 3; i++) {
    big = 0.0;
    for(j = 0; j <= 3; j++) {
      if(ipiv[j] != 1) {
        for (k = 0; k <= 3; k++) {
          if(ipiv[k] == 0) {
            if(fabs(matr[j][k]) >= big) {
              big = (float)fabs(matr[j][k]);
              irow = j;
              icol = k;
            }
          } else if (ipiv[k] > 1) return 1;
        } /* End of for */
      }
    } /* End of for */
    ++(ipiv[icol]);
    if (irow != icol) {
      for (l = 0; l <= 3; l++) MATSWAP(matr[irow][l], matr[icol][l]);
      for (l = 0; l <= 3; l++) MATSWAP(ident[irow][l], ident[icol][l]);
    }
    indxr[i] = irow;
    indxc[i] = icol;
    if(matr[icol][icol] == 0.0f) return 1;
    pivinv = 1.0f / matr[icol][icol];
    matr[icol][icol] = 1.0f;
    for (l = 0; l <= 3; l++) matr[icol][l] *= pivinv;
    for (l = 0; l <= 3; l++) ident[icol][l] *= pivinv;
    for (ll = 0; ll <= 3; ll++) {
      if (ll != icol) {
        dum = matr[ll][icol];
        matr[ll][icol] = 0.0f;
        for(l = 0; l <= 3; l++) matr[ll][l]  -= matr[icol][l] * dum;
        for(l = 0; l <= 3; l++) ident[ll][l] -= ident[icol][l] * dum;
      }
    }
  } /* End of for */

  for (l = 3; l >= 0; l--) {
    if (indxr[l] != indxc[l]) {
      for (k = 0; k <= 3; k++) {
        MATSWAP(matr[k][indxr[l]],matr[k][indxc[l]]);
      }
    }
  } /* End of for */

  return 0;
}


/**** Multiply two 4x4 matrices ****/

void GL_Mat4Mult(register float r[4][4], register float m1[4][4],
                 register float m2[4][4])
{

  /**** Loop unrolling for maximum performances ****/

  r[0][0] = m1[0][0] * m2[0][0] + m1[0][1] * m2[1][0] +
            m1[0][2] * m2[2][0] + m1[0][3] * m2[3][0];
  r[0][1] = m1[0][0] * m2[0][1] + m1[0][1] * m2[1][1] +
            m1[0][2] * m2[2][1] + m1[0][3] * m2[3][1];
  r[0][2] = m1[0][0] * m2[0][2] + m1[0][1] * m2[1][2] +
            m1[0][2] * m2[2][2] + m1[0][3] * m2[3][2];
  r[0][3] = m1[0][0] * m2[0][3] + m1[0][1] * m2[1][3] +
            m1[0][2] * m2[2][3] + m1[0][3] * m2[3][3];

  r[1][0] = m1[1][0] * m2[0][0] + m1[1][1] * m2[1][0] +
            m1[1][2] * m2[2][0] + m1[1][3] * m2[3][0];
  r[1][1] = m1[1][0] * m2[0][1] + m1[1][1] * m2[1][1] +
            m1[1][2] * m2[2][1] + m1[1][3] * m2[3][1];
  r[1][2] = m1[1][0] * m2[0][2] + m1[1][1] * m2[1][2] +
            m1[1][2] * m2[2][2] + m1[1][3] * m2[3][2];
  r[1][3] = m1[1][0] * m2[0][3] + m1[1][1] * m2[1][3] +
            m1[1][2] * m2[2][3] + m1[1][3] * m2[3][3];

  r[2][0] = m1[2][0] * m2[0][0] + m1[2][1] * m2[1][0] +
            m1[2][2] * m2[2][0] + m1[2][3] * m2[3][0];
  r[2][1] = m1[2][0] * m2[0][1] + m1[2][1] * m2[1][1] +
            m1[2][2] * m2[2][1] + m1[2][3] * m2[3][1];
  r[2][2] = m1[2][0] * m2[0][2] + m1[2][1] * m2[1][2] +
            m1[2][2] * m2[2][2] + m1[2][3] * m2[3][2];
  r[2][3] = m1[2][0] * m2[0][3] + m1[2][1] * m2[1][3] +
            m1[2][2] * m2[2][3] + m1[2][3] * m2[3][3];

  r[3][0] = m1[3][0] * m2[0][0] + m1[3][1] * m2[1][0] +
            m1[3][2] * m2[2][0] + m1[3][3] * m2[3][0];
  r[3][1] = m1[3][0] * m2[0][1] + m1[3][1] * m2[1][1] +
            m1[3][2] * m2[2][1] + m1[3][3] * m2[3][1];
  r[3][2] = m1[3][0] * m2[0][2] + m1[3][1] * m2[1][2] +
            m1[3][2] * m2[2][2] + m1[3][3] * m2[3][2];
  r[3][3] = m1[3][0] * m2[0][3] + m1[3][1] * m2[1][3] +
            m1[3][2] * m2[2][3] + m1[3][3] * m2[3][3];
}


/**** Reset the matrix ****/

void GL_Mat4Reset(float Mat[4][4])
{
#ifdef WIN32
  ZeroMemory(Mat, 16 * sizeof(float));
#else
  memset(Mat, 0, 16 * sizeof(float));
#endif
  Mat[0][0] = 1.0;
  Mat[1][1] = 1.0;
  Mat[2][2] = 1.0;
  Mat[3][3] = 1.0;
}


/**** Rotate the transformation matrix ****/

void GL_Mat4RotXYZ(int a, int b, int c, double Theta, float A[4][4])
{
  register float      ct, st, T1, T2, T3;

  ct = cos(Theta);
  st = sin(Theta);

  T1 = A[0][0]*(a+ct*(1-a)) + A[0][1]*(  st*c)     + A[0][2]*( -st*b);
  T2 = A[0][0]*( -st*c)     + A[0][1]*(b+ct*(1-b)) + A[0][2]*(  st*a);
  T3 = A[0][0]*(  st*b)     + A[0][1]*( -st*a)     + A[0][2]*(c+ct*(1-c));
  A[0][0] = T1;
  A[0][1] = T2;
  A[0][2] = T3;

  T1 = A[1][0]*(a+ct*(1-a)) + A[1][1]*(  st*c)     + A[1][2]*( -st*b);
  T2 = A[1][0]*( -st*c)     + A[1][1]*(b+ct*(1-b)) + A[1][2]*(  st*a);
  T3 = A[1][0]*(  st*b)     + A[1][1]*( -st*a)     + A[1][2]*(c+ct*(1-c));
  A[1][0] = T1;
  A[1][1] = T2;
  A[1][2] = T3;

  T1 = A[2][0]*(a+ct*(1-a)) + A[2][1]*(  st*c)     + A[2][2]*( -st*b);
  T2 = A[2][0]*( -st*c)     + A[2][1]*(b+ct*(1-b)) + A[2][2]*(  st*a);
  T3 = A[2][0]*(  st*b)     + A[2][1]*( -st*a)     + A[2][2]*(c+ct*(1-c));
  A[2][0] = T1;
  A[2][1] = T2;
  A[2][2] = T3;
}


/**** Change the endian of the rotation matrix ****/

#ifdef LITTLE_ENDIAN
void GL_Mat4SwapEndian(register float Mat[4][4])
{
  register int  a;

  for(a = 0; a < 16; ++a) {
    Swap(&((float *)Mat)[a]);
  } /* End of for */
}
#endif