
/*************************************************
****          VEGA - OpenGL Utilities         ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>
#include <stdio.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "gl_global.h"
#include "gl_colors.h"
#include "gl_matrix.h"
#include "gl_measure.h"
#include "gl_menu.h"
#include "gl_objs.h"
#include "gl_prefs.h"
#include "bond.h"

/**** Constants ****/

#define  VG_ARM_LEN             0.5
#define  VG_BOND_WIDTH          0.15

/**** Local variables ****/

static GLUquadricObj   *SphereQuadric[3];
static GLuint          SphereList[3];

static GLenum          SphereStyles[] = { GLU_FILL, GLU_LINE, GLU_POINT };


/**** Local prototypes ****/

static VG_BOOL  IsAnti(ATOMO *, ATOMO *, ATOMO *, ATOMO *);


/**** Ultra fast anti configuration detection ****/

static VG_BOOL IsAnti(ATOMO *a, ATOMO *b, ATOMO *Atm3, ATOMO *d)
{
  float         cx, cy, cz;
  float         Cos_Theta, Dist;

  cx = d -> x - Atm3 -> x;
  cy = d -> y - Atm3 -> y;
  cz = d -> z - Atm3 -> z;

  Dist = Distance(a, b) * SQR(Quad(cx) + Quad(cy) + Quad(cz));

  if (Dist) {
    Cos_Theta = ((a -> x - b -> x) * cx + (a -> y - b -> y) * cy +
                 (a -> z - b -> z) * cz) / Dist;

    if (Cos_Theta <= -0.85) return TRUE;        /* About 148 degree */
  }

  return FALSE;
}


/**** Apply the transformation matrix ****/

void GL_ApplyTransMat(ATOMO *Atm, float A[4][4], VG_BOOL All)
{
  float         x, y, z;
  VG_SURFACE   *Dot;

  while(Atm) {
    x = Atm -> x;
    y = Atm -> y;
    z = Atm -> z;
    Atm -> x = x * A[0][0] + y * A[1][0] + z * A[2][0];
    Atm -> y = x * A[0][1] + y * A[1][1] + z * A[2][1];
    Atm -> z = x * A[0][2] + y * A[1][2] + z * A[2][2];
    if ((!All) && (Atm -> Flags & VG_ATMF_MOLEND)) break;
    Atm = Atm -> Ptr;
  } /* End of while */

  if ((All) && (BegSrf)) {
    for(Dot = BegSrf; Dot; Dot = Dot -> Next) {
      x = Dot -> x;
      y = Dot -> y;
      z = Dot -> z;
      Dot -> x = x * A[0][0] + y * A[1][0] + z * A[2][0];
      Dot -> y = x * A[0][1] + y * A[1][1] + z * A[2][1];
      Dot -> z = x * A[0][2] + y * A[1][2] + z * A[2][2];
    } /* End of for */
  }
}


/**** Calculate the range of residue number ****/

void GL_CalcResRange(ATOMO *Atm, VG_ULONG *Min, VG_ULONG *Max)
{
  VG_LONG       PrecResSeq = 0;
  VG_ULONG      Temp;

  *Max = 0;
  *Min = 0xefffffff;
  while(Atm) {
    if (Atm -> ResSeq.L  != PrecResSeq) {
      PrecResSeq  = Atm -> ResSeq.L;
      Temp        = Char2Int(Atm -> ResSeq.C, 4);
      if (Temp > *Max) *Max = Temp;
      if (Temp < *Min) *Min = Temp;
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Init spheres ****/

void GL_BuildSpheres(int Res)
{
  register int          k;

  for(k = 0; k < 3; ++k) {
    SphereList[k]    = glGenLists(1);
    SphereQuadric[k] = gluNewQuadric();

    gluQuadricDrawStyle(SphereQuadric[k], SphereStyles[k]);
    glNewList(SphereList[k], GL_COMPILE);
    gluSphere(SphereQuadric[k], 1.0, Res, Res);
    glEndList();
  } /* End of for */
}


/**** Draw a cylinder ****/

void GL_Cylinder(float *end, float *start, int rod_res, float rod_radius,
                 float rod_top_radius, int Mode)
{
  float                 R, RXY, phi, theta, lenaxis[3];
  GLUquadricObj         *objQuadric;

  if (Mode & VG_GLCYLTYPE_SOLID)
    objQuadric = SphereQuadric[0];
  else
    objQuadric = SphereQuadric[1];

  lenaxis[0] = end[0] - start[0];
  lenaxis[1] = end[1] - start[1];
  lenaxis[2] = end[2] - start[2];

  R = lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]+lenaxis[2]*lenaxis[2];

  if (R <= 0.0) return;

  R = sqrt(R);

  /**** Determine phi rotation angle, amount to rotate about Y ****/

  phi = acos(lenaxis[2] / R);

  /**** Determine theta rotation, amount to rotate about Z ****/

  RXY = sqrt(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]);
  if(RXY <= 0.0) {
    theta = 0.0;
  } else {
    theta = acos(lenaxis[0] / RXY);
    if (lenaxis[1] < 0.0) theta = (float) (2.0 * PI) - theta;
  }

  glPushMatrix();
  glTranslatef((GLfloat)(start[0]), (GLfloat)(start[1]), (GLfloat)(start[2]));

  if(theta != 0.0)
    glRotatef((GLfloat) ((theta / PI) * 180.0), 0.0, 0.0, 1.0);

  if(phi != 0.0)
    glRotatef((GLfloat) ((phi / PI) * 180.0), 0.0, 1.0, 0.0);

  gluCylinder(objQuadric, (GLdouble)rod_radius, (GLdouble)rod_top_radius,
	      (GLdouble)R, (GLint)rod_res, 1);

  /**** Draw a disc if requested ****/

  if (Mode & VG_GLCYLTYPE_DISC) {
    gluQuadricOrientation(objQuadric, (GLenum)GLU_INSIDE);
    gluDisk(objQuadric, (GLdouble)rod_top_radius, (GLdouble)rod_radius,
	  (GLint)rod_res, 1);
    gluQuadricOrientation(objQuadric, (GLenum)GLU_OUTSIDE);
  }

  glPopMatrix();
}


/**** Draw a disc ****/

void GL_Disc(float *end, float *start, int rod_res, float rod_radius, float rod_top_radius)
{
  float                 R, RXY, phi, theta, lenaxis[3];

  lenaxis[0] = end[0] - start[0];
  lenaxis[1] = end[1] - start[1];
  lenaxis[2] = end[2] - start[2];

  R = lenaxis[0] * lenaxis[0] + lenaxis[1] * lenaxis[1] + lenaxis[2] * lenaxis[2];

  if (R <= 0.0) return;

  R = sqrt(R);

  /**** Determine phi rotation angle, amount to rotate about Y ****/

  phi = acos(lenaxis[2] / R);

  /**** Determine theta rotation, amount to rotate about Z ****/

  RXY = sqrt(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]);
  if(RXY <= 0.0) {
    theta = 0.0;
  } else {
    theta = acos(lenaxis[0] / RXY);
    if (lenaxis[1] < 0.0) theta = (float) (2.0 * PI) - theta;
  }

  glPushMatrix();
    glTranslatef((end[0] + start[0]) / 2.0, (end[1] + start[1]) / 2.0, (end[2] + start[2]) / 2.0);

    if(theta != 0.0)
      glRotatef((GLfloat) ((theta / PI) * 180.0), 0.0, 0.0, 1.0);
    if(phi != 0.0)
      glRotatef((GLfloat) ((phi / PI) * 180.0), 0.0, 1.0, 0.0);

    gluDisk(SphereQuadric[0], (GLdouble)rod_radius, (GLdouble)rod_top_radius,
           (GLint)rod_res, 1);
  glPopMatrix();
}


/**** Draw a single atom ****/

void GL_DrawAtm(ATOMO *Atm)
{
  GL_SetColor(Atm -> ColorID);
  glBegin(GL_LINES);
    glVertex3f(Atm -> x - VG_ARM_LEN, Atm -> y             , Atm -> z             );
    glVertex3f(Atm -> x + VG_ARM_LEN, Atm -> y             , Atm -> z             );
    glVertex3f(Atm -> x             , Atm -> y - VG_ARM_LEN, Atm -> z             );
    glVertex3f(Atm -> x             , Atm -> y + VG_ARM_LEN, Atm -> z             );
    glVertex3f(Atm -> x             , Atm -> y             , Atm -> z - VG_ARM_LEN);
    glVertex3f(Atm -> x             , Atm -> y             , Atm -> z + VG_ARM_LEN);
  glEnd();
}


/**** Draw atom labels ****/

void GL_DrawAtmLbl(ATOMO *Atm)
{
  glPushMatrix();
  if ((Atm -> Label == VG_ATMLBL_TYPE) && (*Atm -> Pot.C == '?'))
    GL_SetColor(VGCOL_RED);
  else
    GL_SetColor(VGCOL_WHITE);

  glRasterPos3f(Atm -> x, Atm -> y, Atm -> z);

  switch(Atm -> Label) {
  case VG_ATMLBL_CHARGE:     /* Atom charge  */
    GL_Print("%.2f", Atm -> Charge);
    break;

  case VG_ATMLBL_ELEMENT:    /* Element name */
    GL_PrintDchar(Atm -> Elem);
    break;

  case VG_ATMLBL_FIX:       /* Atom fix constant */
    GL_Print("%.2f", Atm -> Fix);
    break;

  case VG_ATMLBL_NAME:       /* Atom name    */
    GL_PrintQchar(Atm -> Name);
    break;

  case VG_ATMLBL_NUMBER:     /* Atom number  */
    GL_PrintInt(Atm -> Num);
    break;

  case VG_ATMLBL_RESNAME:    /* Residue name */
    GL_PrintQchar(Atm -> ResName);
    break;

  case VG_ATMLBL_RESNAMESEQ: /* Residue name & sequence */
    if (Atm -> ChainID == ' ')
      GL_Print("%.4s:%.4s", Atm -> ResName.C, Atm -> ResSeq.C);
    else
      GL_Print("%.4s:%.4s:%c", Atm -> ResName.C, Atm -> ResSeq.C,
               Atm -> ChainID);
    break;

  case VG_ATMLBL_RESSEQ:    /* Residue name */
    GL_PrintQchar(Atm -> ResSeq);
    break;

  case VG_ATMLBL_TYPE:       /* Atom type    */
    GL_Print("%.8s", Atm -> Pot.C);
    break;
  } /* End of switch */
  glPopMatrix();
}


/**** Draw CPK & Liquorice ****/

void GL_DrawCpk(register ATOMO *Atm, VG_UWORD Mode)
{
  int                   CylDrawMode;
  int                   SphereType;
  float                 SphereRad, CylinderRad;
  register ATOMO        **ConnAtm;
  register VG_ULONG     k;
  VG_UWORD              CylRes;
  XYZ                   Mid;

  if (Mode == VG_CPKMOD_WIRE) {
    CylDrawMode = VG_GLCYLTYPE_WIRE;
    SphereType  = SphereList[1];
  } else {
    CylDrawMode = VG_GLCYLTYPE_SOLID;
    SphereType  = SphereList[0];
  }

  glLineWidth(1.0);

  if (Mode > VG_CPKMOD_WIRE) {
    SphereRad = CylinderRad = GlPrefs -> GlLiqCylRad;
    CylRes    = GlPrefs -> GlLiqCylRes;
  } else {
    CylinderRad = GlPrefs -> GlCpkCylRad;
    CylRes      = GlPrefs -> GlCpkCylRes;
  }

  while(Atm) {
    if ((Mode == VG_CPKMOD_SOLID) || (Mode == VG_CPKMOD_WIRE))
      SphereRad   = Atm -> Rad * GlPrefs -> GlCpkScale;

    if (!Atm -> Active) {
      Atm = Atm -> Ptr;
      continue;
    }

    ++ViewPrefs.RenderedAtoms;

    if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);

    glPushMatrix();
      GL_SetColor(Atm -> ColorID);
      glTranslatef(Atm -> x, Atm -> y, Atm -> z);
      glScalef(SphereRad, SphereRad, SphereRad);
      glCallList(SphereType);
    glPopMatrix();

    if (Atm -> NSost) {
      ConnAtm = Atm -> Conn;
      for(k = 0; k < (VG_ULONG)Atm -> NSost; ++k) {
        if (((*ConnAtm) -> Active) && ((*ConnAtm) -> Num > Atm -> Num)) {
          GL_SetColor(Atm -> ColorID);
          if (Atm -> ColorID != (*ConnAtm) -> ColorID) {
            Mid.x = ((*ConnAtm) -> x + Atm -> x) / 2.0;
            Mid.y = ((*ConnAtm) -> y + Atm -> y) / 2.0;
            Mid.z = ((*ConnAtm) -> z + Atm -> z) / 2.0;
            GL_Cylinder(&Mid.x, &Atm -> x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
            GL_SetColor((*ConnAtm) -> ColorID);
            GL_Cylinder(&(*ConnAtm) -> x, &Mid.x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
          } else {
            GL_Cylinder(&(*ConnAtm) -> x, &Atm -> x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
          }
        }
        ++ConnAtm;
      } /* End of connected atoms for */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Draw in wireframe mode ****/

void GL_DrawWireframe(ATOMO *Atm)
{
  ATOMO                 *Atm2, **Atm3, **ConnAtm;
  VG_ULONG              i, j, k;
  VG_BOOL               Conn;
  VG_UBYTE              ConnOrd, ConnOrd2;
  float                 M[3], V1[3], V2[3];
  float                 D1, D2, T;

  glLineWidth(GlPrefs -> GlWireTick);

  while(Atm) {
    if (Atm -> Active) {
      ++ViewPrefs.RenderedAtoms;
      if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);
      if (Atm -> NSost) {
        Conn    = FALSE;
        ConnAtm = Atm -> Conn;
        for(k = 0; k < (VG_ULONG)Atm -> NSost; ++k) {
          if ((*ConnAtm) -> Active) {
            Conn    = TRUE;
            if ((*ConnAtm) -> Num > Atm -> Num) {
              glBegin(GL_LINES);
                GL_SetColor(Atm -> ColorID);
                glVertex3f(Atm -> x, Atm -> y, Atm -> z);

                /**** Smooth wireframe ****/

                if (GlPrefs -> GlSmoothWire) {
                  GL_SetColor((*ConnAtm) -> ColorID);
                } else if (Atm -> ColorID != (*ConnAtm) -> ColorID) {
                  M[0] = ((*ConnAtm) -> x + Atm -> x) / 2.0;
                  M[1] = ((*ConnAtm) -> y + Atm -> y) / 2.0;
                  M[2] = ((*ConnAtm) -> z + Atm -> z) / 2.0;
                  glVertex3fv(M);
                  GL_SetColor((*ConnAtm) -> ColorID);
                  glVertex3fv(M);
                }
                glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z);
              glEnd();

              /**** Multivector mode ****/

              if ((!ViewPrefs.SmartMove) && (GlPrefs -> GlMultiVecWire)) {
                BndCheck(Atm, *ConnAtm, &ConnOrd);
                if ((ConnOrd) && (ConnOrd != VG_BOND_SINGLE)) {
                  if (ConnOrd != VG_BOND_TRIPLE) {
                    if ((*ConnAtm) -> Ring) {
                      for(i = (VG_ULONG)(*ConnAtm) -> NSost, Atm3 = (*ConnAtm) -> Conn;
                          (i > 0)  &&
                          ((!(*Atm3) -> Ring) || (*Atm3 == Atm));
                          --i, ++Atm3);
/*
                        if (((*Atm3) -> Ring) && (*Atm3 != Atm))
                          break;
*/
                    }
                    if (ConnOrd == VG_BOND_PARDOUBLE) glEnable(GL_LINE_STIPPLE);
                    for(j = 0; j < (VG_ULONG)Atm -> NSost; ++j) {
                      Atm2 = Atm -> Conn[j];
                      if (Atm -> Ring) {
                        if ((Atm2 != *ConnAtm) && (Atm2 -> Ring) && (i) &&
                            (!IsAnti(Atm2, Atm, *ConnAtm, *Atm3)))
                          break;
                      } else if (Atm2 != *ConnAtm) break;
                    } /* End of for */

                    V1[0] = ((Atm -> x + (*ConnAtm) -> x + Atm2 -> x) / 3.0 - Atm -> x) / 2.0;
                    V1[1] = ((Atm -> y + (*ConnAtm) -> y + Atm2 -> y) / 3.0 - Atm -> y) / 2.0;
                    V1[2] = ((Atm -> z + (*ConnAtm) -> z + Atm2 -> z) / 3.0 - Atm -> z) / 2.0;

                    /**** Calculate the scale factor ****/

                    D1 = SQR(Quad(Atm -> x - (*ConnAtm) -> x) +
                             Quad(Atm -> y - (*ConnAtm) -> y) +
                             Quad(Atm -> z - (*ConnAtm) -> z));
                    D2 = SQR(Quad(V1[0]) + Quad(V1[1]) + Quad(V1[2]));

                    if ((Atm -> NSost == 1) || ((*ConnAtm) -> NSost == 1))
                      D2 /= 3.0;

                    D1 = (D1 - D2) / D1;

                    V2[0] = Atm -> x + V1[0] + ((*ConnAtm) -> x - Atm -> x) * D1;
                    V2[1] = Atm -> y + V1[1] + ((*ConnAtm) -> y - Atm -> y) * D1;
                    V2[2] = Atm -> z + V1[2] + ((*ConnAtm) -> z - Atm -> z) * D1;

                    V1[0] = Atm -> x + V1[0];
                    V1[1] = Atm -> y + V1[1];
                    V1[2] = Atm -> z + V1[2];

                    glBegin(GL_LINES);
                      GL_SetColor(Atm -> ColorID);
                      glVertex3fv(V1);

                      /**** Smooth wireframe ****/

                      if (GlPrefs -> GlSmoothWire) {
                        GL_SetColor((*ConnAtm) -> ColorID);
                      } else if (Atm -> ColorID != (*ConnAtm) -> ColorID) {
                        M[0] = (V1[0] + V2[0]) / 2.0;
                        M[1] = (V1[1] + V2[1]) / 2.0;
                        M[2] = (V1[2] + V2[2]) / 2.0;
                        glVertex3fv(M);
                        GL_SetColor((*ConnAtm) -> ColorID);
                        glVertex3fv(M);
                      }
                      glVertex3fv(V2);
                    glEnd();
                    if (ConnOrd == VG_BOND_PARDOUBLE) glDisable(GL_LINE_STIPPLE);
                  } else {
                    glBegin(GL_LINES);
                      GL_SetColor(Atm -> ColorID);
                      glVertex3f(Atm -> x, Atm -> y, Atm -> z + VG_BOND_WIDTH);
                      if ((GlPrefs -> GlSmoothWire) || (Atm -> ColorID != (*ConnAtm) -> ColorID)) {
                        GL_SetColor((*ConnAtm) -> ColorID);
                        glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z + VG_BOND_WIDTH);
                        GL_SetColor(Atm -> ColorID);
                        glVertex3f(Atm -> x, Atm -> y, Atm -> z - VG_BOND_WIDTH);
                        GL_SetColor((*ConnAtm) -> ColorID);
                        glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z - VG_BOND_WIDTH);
                      } else {
                        M[0] = ((*ConnAtm) -> x + Atm -> x) / 2.0;
                        M[1] = ((*ConnAtm) -> y + Atm -> y) / 2.0;
                        M[2] = ((*ConnAtm) -> z + Atm -> z + VG_BOND_WIDTH) / 2.0;
                        glVertex3fv(M);
                        GL_SetColor((*ConnAtm) -> ColorID);
                        glVertex3fv(M);
                        glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z + VG_BOND_WIDTH);

                        GL_SetColor(Atm -> ColorID);
                        glVertex3f(Atm -> x, Atm -> y, Atm -> z - VG_BOND_WIDTH);
                        M[2] = ((*ConnAtm) -> z + Atm -> z - VG_BOND_WIDTH) / 2.0;
                        glVertex3fv(M);
                        GL_SetColor((*ConnAtm) -> ColorID);
                        glVertex3fv(M);
                        glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z - VG_BOND_WIDTH);
                      }
                    glEnd();
                  }
                }
              }
            }
          }
          ++ConnAtm;
        } /* End of for */
        if (!Conn) GL_DrawAtm(Atm);
      } else {
        GL_DrawAtm(Atm);
      }
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}

/**** Draw Van der Waals spheres ****/

void GL_DrawVdw(register ATOMO *Atm, VG_UWORD Mode)
{
  glLineWidth(1.0);
  glPointSize(GlPrefs -> GlVdwDotSize);

  while(Atm) {
    if (Atm -> Active) {
      ++ViewPrefs.RenderedAtoms;
      if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);
      glPushMatrix();
      GL_SetColor(Atm -> ColorID);
      glTranslatef(Atm -> x, Atm -> y, Atm -> z);
      glScalef(Atm -> Rad, Atm -> Rad, Atm -> Rad);
      glCallList(SphereList[Mode]);
      glPopMatrix();
    }
    Atm = Atm -> Ptr;
  } /* End of while */
  glFlush();
}


/**** Remove spheres ****/

void GL_KillSpheres(void)
{
  register int          k;

  for(k = 0; k < 3; ++k) {
   glDeleteLists(SphereList[k], 1);
   gluDeleteQuadric(SphereQuadric[k]);
  } /* End of for */
}


/**** Reset all memory data ****/

void GL_Reset(VG_GLPREFS *Prf)
{
  CloseAllMol();                          /* Free the memory          */
  Prf -> DrawMode     = VG_GLDRAW_WIREFRAME;
  Prf -> ViewCenter.x = 0.0;              /* Reset view center        */
  Prf -> ViewCenter.y = 0.0;
  Prf -> ViewCenter.z = 0.0;
  Prf -> Animation    = FALSE;            /* Stop animation           */
  Prf -> Light        = TRUE;             /* Activate the light       */
  Prf -> TotMol       = 0;
  Prf -> TxtStr       = NULL;             /* Disable text             */
  Prf -> SrfAct       = FALSE;            /* Disable surface          */
  Prf -> MoveMol      = NULL;             /* Move the world           */
  Prf -> MouseMode    = VG_MOUSE_ROT;     /* Mouse mode rotation      */
  Prf -> MeasureMode  = VG_PM_PICKNONE;   /* Measure mode             */
  GL_ObjRemoveAll();
  GL_ResetMtx(Prf);

  ChangeMousePopup();
  ChangeMovePopup();
  ChangePickPopup();
}


/**** Reset the transformation matrix ****/

void GL_ResetMtx(VG_GLPREFS *Prf)
{
  GL_Mat4Reset(RotMat);

  Prf -> RotCenter.x  = 0.0;              /* Reset rotation center    */
  Prf -> RotCenter.y  = 0.0;
  Prf -> RotCenter.z  = 0.0;

  Prf -> Scale       = 1.0;              /* Reset scale              */
  Prf -> StepRot.x   = 0.0;              /* Reset animation steps    */
  Prf -> StepRot.y   = VG_GLDEF_ROTSTEP;
  Prf -> StepRot.z   = 0.0;
  Prf -> StepTrans.x = 0.0;              /* Reset translations steps */
  Prf -> StepTrans.y = 0.0;
  Prf -> StepTrans.z = 0.0;
}


/**** Reset the view ****/

void GL_ResetView(VG_GLPREFS *Prf)
{
  Prf -> ViewCenter  = LastCent;         /* Reset rotation center    */
  Prf -> Animation   = FALSE;            /* Stop animation           */
  Prf -> Light       = TRUE;             /* Activate the light       */
  GL_ResetMtx(Prf);

  ChangeAnimMenu(FALSE);
  ChangeLightMenu(TRUE);
}


/**** Set the atom labels ****/

void GL_SetAtmLbl(register ATOMO *Atm, register VG_UBYTE Label)
{
  VG_BYTE       PrecChainID;
  VG_LONG       PrecResName, PrecResSeq;

  if (Label < VG_ATMLBL_RESNAMESEQ) {
    while(Atm) {
      if (Atm -> Active)
        Atm -> Label = Label;
      Atm = Atm -> Ptr;
    } /* End of while */
  } else {
    PrecChainID = 0;
    PrecResName = 0;
    PrecResSeq  = 0;
    while(Atm) {
      if (Atm -> Active) {
        if ((Atm -> ChainID   != PrecChainID) ||
            (Atm -> ResName.L != PrecResName) ||
            (Atm -> ResSeq.L  != PrecResSeq)) {
          Atm -> Label = Label;
          PrecChainID = Atm -> ChainID;
          PrecResName = Atm -> ResName.L;
          PrecResSeq  = Atm -> ResSeq.L;
        } else Atm -> Label = VG_ATMLBL_NONE;
      }
      Atm = Atm -> Ptr;
    } /* End of while */
  }
}


/**** Ribbon rendering ****/

void GL_DrawRibbon(register ATOMO *Atm, VG_UWORD Mode)
{
  GLfloat       t, x, y;
  GLfloat       *Cos_t, *Sin_t;
  VG_ULONG      i;

  VG_ULONG      SplineRes = 5;
  VG_ULONG      RibbonRes = 5;

  VG_ULONG      NumSeg    = 1 + 2 * RibbonRes;

  if (((Cos_t = (GLfloat *)Alloca(NumSeg)) != NULL) &&
      ((Sin_t = (GLfloat *)Alloca(NumSeg)) != NULL)) {
    Cos_t[0] = 1.0f;
    Sin_t[0] = 0.0f;
    x        = (3.141592653589793238f) / RibbonRes;

    for(i = 1;i < RibbonRes;i++) {
      t                    = x*i;
      Cos_t[i]             = (GLfloat)(cos(t));
      Sin_t[i]             = -(GLfloat)(sin(t));
      Cos_t[i + RibbonRes] = -Cos_t[i];
      Sin_t[i + RibbonRes] = -Sin_t[i];
    } /* End of for */

    Cos_t[RibbonRes]  = -1.0f;
    Sin_t[RibbonRes]  = 0.0f;
    Cos_t[NumSeg - 1] = 1.0f;
    Sin_t[NumSeg - 1] = 0.0f;


    FREE(Cos_t);
    FREE(Sin_t);
  }
}





