
/*************************************************
****           VEGA - Ion arrangment          ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/

/*
 * Original code by Alexander Balaeff
 * (C) Copyright 2000 The Theoretical Biophysics Group, Beckman Institute, and
 * The Board of Trustees of the University of Illinois
 */

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "progbar.h"

/**** Macros *****/

#define rint(x) (x<0?ceil((x)-0.5):floor((x)+0.5))

/**** Local proptotypes ****/

static VG_LONG excl_around(VG_LONG, VG_LONG, VG_LONG, float, VG_BYTE *,
		           VG_LONG, VG_LONG, VG_LONG);
static VG_LONG indices(VG_LONG, VG_LONG, VG_LONG, VG_LONG, VG_LONG *, VG_LONG *,
                       VG_LONG *);
static VG_LONG min_element(float *, VG_LONG, VG_BYTE *);

/***************************************************************************
  EXPLANATION OF THE CONFIGURATION PARAMETERS:

  Elem          -  ion element.
  Charge        -  ion charge.
  N_ions        -  the number of ions to place around the macromolecule.
  excl_atom_rad -  the closest distance (in Angstroms) allowed between any
                   of the placed ions and any atom of the macromolecule.
		   Due to an error in atomic positions, introduced by using
		   the grid, it is recommended to make this parameter slightly
		   larger than the actually allowed closest distance.
  excl_ion_rad  -  the closest distance between any two ions (in Angstroms).
                   This parameter prevents the placed ions from clustering
		   around a single negative charge, and would prevent
		   positively and a negatively charged ions from being placed
		   too close to each other.
  delta         -  the cell size of the cubic grid on which the ions are placed
                   (in Angstroms).  Grids with smaller cells allow more precise
		   ions placement (in part, due to the least disturbance of the
		   macromolecular atoms, relocated to the grid nodes). However,
		   a finer grid slows the computations down and requires
		   significantly more RAM.
  border_width  -  the thickness of the grid margin, surrounding the macromolecule
                   (in Angstroms).
***************************************************************************/



VG_BOOL AddIons(char *Elem, float Charge, VG_ULONG N_ions, float excl_atom_rad,
                float excl_ion_rad, float delta, float border_width)
{
  ATOMO         *BegIon, *Atm;
  float         *E_cur, *Ener, *SQRTs;
  float         Ener_cur, f_dum;
  float         x_max, y_max, z_max;
  float         x_min, y_min, z_min;
  VG_BYTE       *I_cur, *I_excl;
  VG_LONG       *i_tmp, *j_tmp, *i_tmp_at_0, *i_tmp_at_1, *SQs;
  VG_LONG       *ii_ions, *jj_ions, *kk_ions;
  VG_LONG       *ii_at, *jj_at, *kk_at;
  VG_LONG       i_tmp_ions, j_tmp_ions;
  VG_LONG       i_dum, ii_dum, jj_dum, kk_dum;
  VG_LONG       i_sod_point, m, k, N_SQs;
  VG_ULONG      i, ii, jj, kk, N_SQRTs;
  VG_ULONG      N_box_0, N_box_1, N_box_2, N_box_tot, N_Ener;

  VG_BOOL       Cont = TRUE;
  VG_BOOL       Ret  = FALSE;

  PrintProg(MSG_ION_START);

#ifdef WIN32
  if (GLOBSW_OPENGL) {
    ProgBarInit(GetStr(MSG_ION_PROGBAR_TITLE), 100);
    Cont = ProgBarUpd(GetStr(MSG_ION_PROGBAR_INIT), 0);
  }
#endif

  /**** Molecule dimensions ****/

  for(Atm = BegAtm; Atm; Atm = Atm -> Ptr) {
    if (Atm == BegAtm) {
      x_max = x_min = Atm -> x;
      y_max = y_min = Atm -> y;
      z_max = z_min = Atm -> z;
    } else {
      if (x_max < Atm -> x) x_max = Atm -> x;
      if (x_min > Atm -> x) x_min = Atm -> x;
      if (y_max < Atm -> y) y_max = Atm -> y;
      if (y_min > Atm -> y) y_min = Atm -> y;
      if (z_max < Atm -> z) z_max = Atm -> z;
      if (z_min > Atm -> z) z_min = Atm -> z;
    }
  } /* End of for */

  CatPrintf(stdout, MSG_ION_MOLDIM,
            x_min, x_max, y_min,y_max, z_min, z_max);

  /******************************/
  /*** The setup of the grid. ***/
  /******************************/

  /**** Finding the size of the grid box ****/

  N_box_0   = (VG_ULONG)((x_max - x_min + 2.*border_width) / delta ) + 1 ;
  N_box_1   = (VG_ULONG)((y_max - y_min + 2.*border_width) / delta ) + 1 ;
  N_box_2   = (VG_ULONG)((z_max - z_min + 2.*border_width) / delta ) + 1 ;
  N_box_tot = N_box_0 * N_box_1 * N_box_2;

  CatPrintf(stdout, MSG_ION_GRIDSIZE,
                    N_box_0, N_box_1, N_box_2, N_box_tot);


  /**** Tying up the atoms to the grid nodes ****/

  if ((ii_at = (VG_LONG *)Alloca(sizeof(VG_LONG) * TotalAtm)) != NULL) {
    if ((jj_at = (VG_LONG *)Alloca(sizeof(VG_LONG) * TotalAtm)) != NULL) {
      if ((kk_at = (VG_LONG *)Alloca(sizeof(VG_LONG) * TotalAtm)) != NULL) {
        for(Atm = BegAtm, i = 0; Atm; Atm = Atm -> Ptr, ++i) {
          ii_at[i] = rint((Atm -> x - x_min + border_width) / delta);
          jj_at[i] = rint((Atm -> y - y_min + border_width) / delta);
          kk_at[i] = rint((Atm -> z - z_min + border_width) / delta);
        } /* End of for */

        /**** Pre-computing the inverse distances between the grid nodes ****/

        N_SQRTs = N_box_0 * N_box_0 + N_box_1 * N_box_1 + N_box_2 * N_box_2 + 1;
        if ((SQRTs = (float *)Alloca(sizeof(float) * N_SQRTs)) != NULL ) {
          for(i = 1; i < N_SQRTs; i++)
            SQRTs[i] = 1.0 / SQR((float)i);

          /**** Pre-computing the squares of distances between the grid layers ****/

          N_SQs = N_box_0;
          if (N_box_1 > (VG_ULONG)N_SQs) N_SQs = N_box_1;
          if (N_box_2 > (VG_ULONG)N_SQs) N_SQs = N_box_2;
          if ((SQs = (VG_LONG *)Alloca(sizeof(VG_LONG) * (N_SQs * 2 + 1))) != NULL) {
            for(k = -N_SQs; k <= N_SQs; k++)
              SQs[k + N_SQs] = k * k;

            /**** Setting up the exclusion array ****/

            if ((I_excl = (VG_BYTE *)Alloca(N_box_tot)) != NULL ) {
              CatPrintf(stdout, MSG_ION_EXCLUSIONMAP);
              for(i = 0; i < N_box_tot; i++) I_excl[i] = 1;

              /**** Excluding the grid nodes within excl_atom_rad of protein/DNA ****/

              N_Ener         = N_box_tot;
              excl_atom_rad /= delta;

              for(i = 0; i < TotalAtm; i++) {
                N_Ener -= excl_around(ii_at[i], jj_at[i], kk_at[i], excl_atom_rad,
				      I_excl, N_box_0, N_box_1, N_box_2);
              } /* End of for */

              CatPrintf(stdout, MSG_ION_GRIDNODES, N_Ener);
#ifdef WIN32
              if (GLOBSW_OPENGL)
                Cont = ProgBarUpd(GetStr(MSG_ION_PROGBAR_INIT), 10);
#endif

              /***********************************/
              /*** Initial energy computation. ***/
              /***********************************/

              /**** Memory arrangement for the energy array ****/

              if ((Ener = (float *)Alloca(sizeof(float) * N_box_tot)) != NULL) {
                if ((i_tmp_at_0 = (VG_LONG *)Alloca(sizeof(VG_LONG) * TotalAtm)) != NULL) {
                  if ((i_tmp_at_1 = (VG_LONG *)Alloca(sizeof(VG_LONG) * TotalAtm)) != NULL) {
                    for(I_cur = I_excl, E_cur = Ener, ii_dum = N_SQs,
	                ii = 0; (Cont) && (ii < N_box_0); ii++, ii_dum--) {
#ifdef WIN32
                      if (GLOBSW_OPENGL)
                        Cont = ProgBarUpd(GetStr(MSG_ION_PROGBAR_ENER), 10 + (int)((float)(80 * ii) / (float)N_box_0));
#endif
                      for(i_tmp = i_tmp_at_0, i = 0; i < TotalAtm; i++, i_tmp++)  {
                        i_dum  = ii_dum + ii_at[i] ;
	                *i_tmp = SQs[i_dum];
                      } /* End of for */

                      for(jj_dum = N_SQs, jj=0; jj < N_box_1; jj++, jj_dum--) {
                        for(i_tmp = i_tmp_at_0, j_tmp = i_tmp_at_1,
                            i = 0; i < TotalAtm; i++, i_tmp++, j_tmp++ ) {
                          i_dum  = jj_dum + jj_at[i];
	                  *j_tmp = SQs[i_dum] + (*i_tmp);
                        } /* End of for */

#ifdef WIN32
                        if (GLOBSW_OPENGL) ProgBarMsgUpd();
#endif

                        for(kk_dum = N_SQs, kk = 0; kk < N_box_2; kk++,
                            kk_dum--, I_cur++, E_cur++ ) {
                          if(*I_cur ) {
                            /* The energy is computed only if the current node is not excluded */
/*
	                    for(Ener_cur = 0, Atm = BegAtm, i_tmp = i_tmp_at_1,
		                i = 0; i < TotalAtm; i++, Atm = Atm -> Ptr, i_tmp++) {
                              i_dum     = kk_dum + kk_at[i];
	                      m         = SQs[i_dum] + (*i_tmp);
	                      Ener_cur += Atm -> Charge * Charge * SQRTs[m];
*/
	                    for(Ener_cur = 0, Atm = BegAtm, i_tmp = i_tmp_at_1,
		                k = 0; (VG_ULONG)k < TotalAtm; k++, Atm = Atm -> Ptr, i_tmp++) {
                              i_dum     = kk_dum + kk_at[k];
	                      m         = SQs[i_dum] + (*i_tmp);
	                      Ener_cur += Atm -> Charge * Charge * SQRTs[m];

	                    } /* End of for */
	                    *E_cur = Ener_cur;
	                  } /* End of for */
                        } /* End of for */
                      } /* End of for */
                    } /* End of for */

                    /***************************************************/
                    /***  Place N_ions ions, one by one.             ***/
                    /***  Recompute the grid energy after each step. ***/
                    /***************************************************/

                    if (Cont) {
                      excl_ion_rad /= delta;
                      if ((ii_ions = (VG_LONG *)Alloca(sizeof(VG_LONG) * N_ions)) != NULL) {
                        if ((jj_ions = (VG_LONG *)Alloca(sizeof(VG_LONG) * N_ions)) != NULL) {
                          if ((kk_ions = (VG_LONG *)Alloca(sizeof(VG_LONG) * N_ions)) != NULL) {
                            CatPrintf(stdout, MSG_ION_PLACED);
                            for(k = 0; (Cont) && (k < (VG_LONG)N_ions); k++) {
#ifdef WIN32
                              if (GLOBSW_OPENGL)
                                Cont = ProgBarUpd(GetStr(MSG_ION_PROGBAR_PLACE),
                                                  90 + (int)((float)(10 * (k + 1)) / (float)N_ions));
#endif
                              i_sod_point = min_element(Ener, N_box_tot, I_excl);

                              indices(i_sod_point, N_box_0, N_box_1, N_box_2,
		                      ii_ions + k, jj_ions + k, kk_ions + k);

                              CatPrintf(stdout, MSG_ION_POSITION,
	                                k + 1, Ener[i_sod_point], i_sod_point, ii_ions[k],
                                        jj_ions[k], kk_ions[k]);

                              N_Ener -= excl_around(ii_ions[k], jj_ions[k], kk_ions[k], excl_ion_rad,
		                                    I_excl, N_box_0, N_box_1, N_box_2);
                              I_excl[i_sod_point] = 0;

                              for(I_cur = I_excl, E_cur = Ener, ii_dum = N_SQs,
                                  ii = 0; ii < N_box_0; ii++, ii_dum--) {
                                i_dum      = ii_dum + ii_ions[k];
                                i_tmp_ions = SQs[i_dum] ;

                                for(jj_dum = N_SQs, jj = 0; jj < N_box_1; jj++, jj_dum--) {
                                  i_dum      = jj_dum + jj_ions[k];
                                  j_tmp_ions = SQs[i_dum] + i_tmp_ions;

                                  for(kk_dum = N_SQs, kk = 0; kk < N_box_2; kk++,
                                      kk_dum--, I_cur++, E_cur++) {
                                    if(*I_cur) {
                                      i_dum   = kk_dum + kk_ions[k];
                                      m       = SQs[i_dum] + j_tmp_ions;
                                      *E_cur += SQRTs[m];
                                    }
                                  } /* End of for */
                                } /* End of for */
                              } /* End of for */
                            } /* End of for */

                            /**** Add the ion coordinates ****/

                            AtmLoaded = TotalAtm;
                            BegIon    = NULL;
                            i         = 0;
                            Ret       = TRUE;
                            for(k = 0; (Ret) && (k < (VG_LONG)N_ions); k++) {
                              if ((Atm = AllocAtm(&BegIon, &i))) {
                                Atm -> Name.C[0] = Atm -> Elem.C[0] = Elem[0];
                                Atm -> Name.C[1] = Atm -> Elem.C[1] = Elem[1];
                                Atm -> Charge    = Charge;
                                Atm -> ResName.L = *(VG_LONG *)"ION";
                                Atm -> x         = delta * (float)ii_ions[k] - border_width + x_min;
                                Atm -> y         = delta * (float)jj_ions[k] - border_width + y_min;
                                Atm -> z         = delta * (float)kk_ions[k] - border_width + z_min;
                              } else Ret = FALSE;
                            } /* End of for */
                            if (Atm) {
                              Atm -> Flags   |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
                              LastAtm -> Ptr   = BegIon;
                              BegIon  -> Prev  = LastAtm;
                              LastAtm          = Atm;
                              TotalAtm        += i;
                            }
                            FREE(kk_ions);
                          }
                          FREE(ii_ions);
                        }
                        FREE(jj_ions);
                      }
                    }
                    FREE(i_tmp_at_1);
                  }
                  FREE(i_tmp_at_0);
                }
                FREE(Ener);
              }
              FREE(I_excl);
            }
            FREE(SQs);
          }
          FREE(SQRTs);
        }
        FREE(kk_at);
      }
      FREE(jj_at);
    }
    FREE(ii_at);
  }
#ifdef WIN32
    if (GLOBSW_OPENGL) {
      ProgBarClose();
      LocPrintf(stdout, "\n");
    }
#endif

  return Ret;
}


/***********************************************************/
/*** This subroutine excludes all grid nodes closer than ***/
/*** the raidus R to the grid node (i_pt, j_pt, k_pt).   ***/
/*** Returned is the number of nodes, excluded this way. ***/ 
/***********************************************************/

static VG_LONG excl_around(VG_LONG i_pt, VG_LONG j_pt, VG_LONG k_pt, float excl_rad,
		           VG_BYTE *I_excl, VG_LONG N_box_0, VG_LONG N_box_1,
                           VG_LONG N_box_2)
{
  VG_LONG       ii, jj, kk;
  VG_LONG       i_max, j_max, k_max;
  VG_LONG       i_min, j_min, k_min;
  VG_LONG       ind, ind0, ind1, i_dum;
  VG_LONG       N_excl = 0;
  float         d0, d1, d2, excl_rad_sq;

  excl_rad_sq = excl_rad * excl_rad;

  i_max = i_pt + excl_rad;
  if (i_max >= N_box_0) i_max = N_box_0 - 1;

  i_min = i_pt - excl_rad;
  if (i_min < 0) i_min = 0;

  j_max = j_pt + excl_rad;
  if (j_max >= N_box_1) j_max = N_box_1 - 1;

  j_min = j_pt - excl_rad;
  if(j_min < 0) j_min = 0;

  k_max = k_pt + excl_rad;
  if (k_max >= N_box_2) k_max = N_box_2 - 1;

  k_min = k_pt - excl_rad;
  if(k_min < 0) k_min = 0;

  for(ind0 = i_min * N_box_1, ii = i_min; ii <= i_max;
      ind0 += N_box_1, ii++ ) {
    d0  = ii - i_pt;
    d0 *= d0;

    for(ind1 = (ind0 + j_min) * N_box_2, jj = j_min; jj <= j_max;
        ind1 += N_box_2, jj++ ) {
      d1 = jj - j_pt;
      d1 *= d1;

      for(ind = ind1 + k_min, kk = k_min; kk <= k_max;
          ind++, kk++ ) {
        if(I_excl[ind]) {
          d2 = kk - k_pt;
	  d2 *= d2;
	  if(d0 + d1 + d2 <= excl_rad_sq ) {
	    I_excl[ind] = 0;
	    N_excl++;
	  }
        }
      } /* End of for */
    } /* End of for */
  } /* End of for */

  return N_excl;
}


/********************************************************************************/
/*** This subroutine computes the 3-D indices of the ind-th node of the grid. ***/
/*** For the grid box with dimensions N_box_0, N_box_1, N_box_2, each of the  ***/
/*** N_box_0 layers has a size of N_box_1*N_box_2, and each point's index is: ***/
/***   index(i,j,k) = N_box_1*N_box_2*i + N_box_2*j + k .                     ***/
/********************************************************************************/

static VG_LONG indices(VG_LONG ind, VG_LONG N_box_0, VG_LONG N_box_1,
                       VG_LONG N_box_2, VG_LONG *i_pt, VG_LONG *j_pt,
                       VG_LONG *k_pt)
{
  VG_LONG       i_dum;

  i_dum  = ind;
  *i_pt  = i_dum / (N_box_1 * N_box_2);
  i_dum -= (*i_pt) * (N_box_1 * N_box_2);
  *j_pt  = i_dum / N_box_2;
  i_dum -= (*j_pt) * N_box_2;
  *k_pt  = i_dum;

  return ind;
}


/******************************************************************************/
/*** This subroutine returs the index of the smallest element of the array. ***/
/******************************************************************************/

static VG_LONG min_element(float *Ener, VG_LONG size, VG_BYTE *excl_map)
{
  VG_LONG       i;
  float         V;

  VG_LONG       i_min = -1;

  for(i = 0; i < size; ++i) {
    if(*excl_map) {
      if (i_min == -1) {
        V = *Ener;
        i_min = i;
      } else if(*Ener < V) {
        V     = *Ener;
        i_min = i;
      }
    }
    ++Ener;
    ++excl_map;
  } /* End of for */

  return i_min;
}

