
/*************************************************
****      VEGA - Utilities & Subroutines      ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"
#include "surface.h"

#ifdef AMIGA
#  include <dos/dos.h>
#  include <exec/types.h>
#  include <exec/memory.h>
#  include <exec/execbase.h>
#endif


/**** Add the packer file extension ****/

void AddPackExt(register char *FileName, VG_UWORD PackMode)
{
  register char          **Ptr;
  register VG_ULONG      k;

  if (PackMode) {
    for(Ptr = (char **)(PackExt + 1); *Ptr; ++Ptr) {
      k = strlen(FileName) - strlen(*Ptr);
      if ((FileName[k - 1] == '.') &&
          (!strcasecmp(FileName + k, *Ptr))) break;
    }
    if (!*Ptr) {
      strcat(FileName, ".");
      strcat(FileName, PackExt[PackMode]);
    }
  }
}


/**** Add a value to the radius matrix ****/

void AddToRad(register ATOMO *Atm, register float *Rad, register float Val)
{
  while(Atm) {
    *Rad += Val;
    Atm = Atm -> Ptr;
    ++Rad;
  }
}


/**** Convert a string to an integer specifing the size ****/

VG_LONG AnToI(char *Str, VG_LONG Size)
{
  char          Buf[32];
  VG_LONG       k;

  for(k = 0; k < Size; ++k) Buf[k] = Str[k];
  Buf[Size] = 0;
  if (!*Buf) return 0;

  return atoi(Buf);
}


/**** Convert the atom name into element ****/

void AtmName2Elem(register ATOMO *Atm)
{
  char                          N1;
  const char                    *Ptr;
  const VG_DCHAR                *Elem;

  static const char             *Tbl      = "HCONS";
  static const VG_DCHAR         ExtElem[] = {{"Du"}, {"Lp"}, {0, 0}};


  Atm -> Elem.C[0] = toupper(Atm -> Name.C[0]);
  N1               = Atm -> Name.C[1];

  if ((N1) && ((N1 < '0') || (N1 > '9'))) {
    if (islower(N1)) {
      Atm -> Elem.C[1] = N1;
    } else {
      for(Ptr = Tbl; *Ptr; ++Ptr)
        if (*Ptr == *Atm -> Name.C) return;

      N1 = tolower(N1);
      for(Elem = ExtElem; Elem -> S; ++Elem) {
        if ((Atm -> Elem.C[0] == Elem -> C[0]) &&
           (tolower(N1) == Elem -> C[1])) {
          Atm -> Elem.C[1] = Elem -> C[1];
          return;
        }
      } /* End of for */

      Atm -> Elem.C[1] = N1;
    }
  }
}


/**** Calc the geometry center of a molecule ****/

void CalcGeoCent(register ATOMO *Atm, register XYZ *Vect, VG_BOOL GlTrans)
{
  float                 Tmp;

  float                 MaxDist = 0.0;
  VG_ULONG              Tot     = 0;

  if (Atm) {
    Vect -> x = 0.0;
    Vect -> y = 0.0;
    Vect -> z = 0.0;

    while(Atm) {
      Vect -> x += Atm -> x;
      Vect -> y += Atm -> y;
      Vect -> z += Atm -> z;
      if (GlTrans) {
        Tmp = Atm -> x;
        FastAbs(Tmp);
        if (Tmp > MaxDist) MaxDist = Tmp;
        Tmp = Atm -> y;
        FastAbs(Tmp);
        if (Tmp > MaxDist) MaxDist = Tmp;
        Tmp = Atm -> z;
        FastAbs(Tmp);
        if (Tmp > MaxDist) MaxDist = Tmp;
      }
      Atm = Atm -> Ptr;
      ++Tot;
    } /* End of while */

    Vect -> x /= (float)Tot;
    Vect -> y /= (float)Tot;
    Vect -> z /= (float)Tot;

    if (GlTrans) {
      Vect -> x *= -1.0;
      Vect -> y *= -1.0;
      Vect -> z  = Vect -> z * -1.0 - MaxDist * 2.1;
    }
  }
}


/**** Change or add the file extension ****/

void ChangeExt(char *Str, char *NewExt, char *OldExt)
{
  register char *Ptr;

  for(Ptr = Str + strlen(Str) - 1; Ptr != Str; --Ptr) {
    if (*Ptr == '.') {
      if (NewExt) {
        ++Ptr;
        if (OldExt) strcpy(OldExt, Ptr);
        if (strcasecmp(Ptr, "tmp")) strcpy(Ptr, NewExt);
      } else *Ptr = 0;
      break;
    }
  } /* End of for */

  if (NewExt) {
    if ((Ptr == Str) && (*Ptr != '.')) {
      if (OldExt) *OldExt = 0;
      strcat(Ptr, ".");
      strcat(Ptr, NewExt);
    }
  } else if (OldExt) *OldExt = 0;
}


/**** Convert a string into an integer ****/

VG_ULONG Char2Int(char *Str, VG_ULONG k)
{
  VG_ULONG              Molt = 1;
  VG_ULONG              Num  = 0;

  --k;
  do {
    if ((Str[k] >= '0') && (Str[k] <= '9')) {
      Num  += ((Str[k] - '0') * Molt);
      Molt *= 10;
    }
  } while(k--);

  return Num;
}


/**** Check if the force field is assigned ****/

VG_BOOL ChkForceField(register ATOMO *Atm, VG_ULONG TotAtm)
{
  register VG_ULONG  Unassigned = 0;

  do {
    if (Atm -> Pot.DL == MakeInt64(0x3f00000000000000)) ++Unassigned;
    Atm = Atm -> Ptr;
  } while(Atm);
  if (Unassigned == TotAtm) return FALSE;
  else return TRUE;
}


/**** Close all molecules ****/

void CloseAllMol(void)
{
  SrfClose();
  SelRemAll();
  FreeAtm(BegAtm);
  BegAtm    = NULL;
  LastAtm   = NULL;
  TotalAtm  = 0;
  AtmLoaded = 0;
}


/**** Fast strlen ****/

#ifdef LITTLE_ENDIAN__
size_t fstrlen(register const char *s)
{
  register const char   *p;
  register VG_ULONG     d;

#  define M1    (0x7f7f7f7f)
#  define M2    (0x80808080)
#  define SW    (sizeof(int) / sizeof(char))

  p = s - 1;
  do {
    p++;
    if ((((int)p)&(SW - 1)) == 0) {
      do {
        d  = *((int *)p);
        p += SW;
        d  = (((d&M1)+M1)|d)&M2;
      } while(d == M2);
      p -= SW;
    }
  } while(*p != 0);

  return (p - s);
}
#endif

/**** Find a residue ****/

VG_ULONG FindResPtr(register ATOMO *Atm, ATOMO **ResPtr,
                    VG_LONG NomeRes, VG_LONG NRes)
{
  register VG_UWORD  Chk;
  register VG_ULONG  NAtm = 0;

  *ResPtr = 0;

  while(Atm) {
    Chk = 0;
    if (!NomeRes) Chk = 1;
    else if (Atm -> ResName.L == NomeRes) Chk = 1;
    if (!NRes) ++Chk;
    else if (Atm -> ResSeq.L == NRes) ++Chk;
    if (Chk == 2) {
      if (!*ResPtr) *ResPtr = Atm;
      ++NAtm;
    }
    Atm = Atm -> Ptr;
  }

  return NAtm;
}


/**** Reads a record in Fortran format ****/

VG_BOOL ForRead(FILE *FH, void *Ptr, VG_ULONG Len)
{
  VG_BOOL            Ret = TRUE;

  fseek(FH, sizeof(int), SEEK_CUR);
  if (fread(Ptr, Len, 1, FH) == 1)
    fseek(FH, sizeof(int), SEEK_CUR);
  else Ret = PrintDosErr();

  return Ret;
}


/**** Performs Seek from/to Fortran records ****/

VG_BOOL ForSeek(FILE *FH, VG_ULONG Rec, VG_BOOL SwapEndian)
{
  unsigned int  Len;
  VG_BOOL    Ret = TRUE;

  do {
    if (fread(&Len, sizeof(int), 1, FH)) {
      if (SwapEndian)
        Swap(&Len);
      fseek(FH, Len + sizeof(int), SEEK_CUR);
    } else Ret = PrintDosErr();
    --Rec;
  } while ((Rec) && (Ret));

  return Ret;
}


/**** Copy a string from Fortran to C ****/

void ForStrCpy(register char *Dest, register char *Src, VG_ULONG Len)
{
  register VG_ULONG  k;

  for(k = 0;(k < Len) && (Src[k] != ' ');++k)
    Dest[k] = Src[k];
}


/**** Writes a record in Fortran unformatted format ****/

VG_BOOL ForWrite(FILE *FH, void *Num, VG_ULONG Size)
{
  VG_BOOL      Ret = TRUE;

#ifdef LITTLE_ENDIAN
  VG_ULONG     Sz  = SwapI(Size);
#else
  VG_ULONG     Sz  = Size;
#endif

  if ((fwrite(&Sz, sizeof(VG_ULONG), 1, FH) != 1) ||
      (fwrite(Num, Size, 1, FH) != 1) ||
      (fwrite(&Sz, sizeof(VG_ULONG), 1, FH) != 1))
    Ret = PrintDosErr();

  return Ret;
}


/**** Change a string to lowercase ****/

void Lower(register char *Str)
{
  do *Str = (char)tolower(*Str); while(*++Str);
}


/**** Special floor function ****/

#ifdef __VG_OPENGL
int MyFloor(double Val)
{
  int           Res = (int)Val;

  if ((Val - (double)Res) > 0.5) ++Res;

  return Res;
}
#endif


/**** Normalize an angle into 0-360 range ****/

double NormAngle(double Val)
{
  double        Ang;

  Val /= 360.0;
  modf(Val, &Ang);
  Ang = (Val - Ang) * 360.0;

  if (Ang < (double)0.0)
    return (360.0 + Ang);

  return Ang;
}


/**** Normalize an angle into 0-360 range (single precision version) ****/

float NormAnglef(float Val)
{
  float        Ang;

  Val /= 360.0f;

  Ang = (Val - (float)((int)Val)) * 360.0f;

  if (Ang < 0.0f)
    return (360.0f + Ang);

  return Ang;
}


/**** Print an error with atom ****/

VG_BOOL PrintAtmErr(char *Err, ATOMO *Atm, VG_BOOL Force)
{
  sprintf(ErrStr, "%s: %-7d %-4.4s %-2.2s %.4s_%.4s", Err,
          Atm -> Num, Atm -> Name.C, Atm -> Elem.C, Atm -> ResName.C,
          Atm -> ResSeq.C);

  if ((ErrMode != VG_ERRMOD_QUIET) && ((!GLOBSW_STDOUT) || (Force))) {
    LocPrintf(stderr, "%s\n", ErrStr);
  }

  return FALSE;
}


/**** Print loading format ****/

void PrintLoad(VG_ULONG Msg, VG_ULONG NMol)
{
  if ((!GLOBSW_STDOUT) && (NMol))
    CatPrintf(stdout, MSG_UTILS_LOADPROG, NMol, GetStr(Msg));
}


/**** Print a progress message ****/

void PrintProg(VG_ULONG Msg)
{
  if (!GLOBSW_STDOUT)
    LocPrintf(stdout, "%s\n", GetStr(Msg));
}


/**** Convert a VG_QCHAR into a char ****/

char *Qchar2Str(VG_QCHAR *Qchar)
{
  static char   Res[6];

  strncpy(Res, (*Qchar).C, 4);
  Res[4] = 0;

  return Res;
}

/**** Remove hydrogens ****/

void RemoveH(ATOMO **InizAtm, VG_ULONG *TotAtm, VG_BOOL ActOnly)
{
  register ATOMO      *Con;
  register VG_UWORD   k, j;

  ATOMO               *PrecAtm = *InizAtm;
  ATOMO               *Atm     = *InizAtm;
  VG_ULONG            Count    = 1;

  while(Atm) {
    if (((!ActOnly) || (Atm -> Active)) &&
#ifdef LITTLE_ENDIAN
        (Atm -> Elem.S == 0x48)) {
#else
        (Atm -> Elem.S == 0x4800)) {
#endif

      if ((Con = Atm -> Conn[0]) != NULL) {
        for(k = 0;(k < MAXBOND) && (Con -> Conn[k]); ++k) {
          if (Con -> Conn[k] == Atm) {
            for(j = k; j < (MAXBOND - 1); ++j) {
              Con -> Conn[j]  = Con -> Conn[j + 1];
              Con -> Order[j] = Con -> Order[j + 1];
            } /* End of for */
            --Con -> NSost;
          }
        } /* End of for */
      }
      --*TotAtm;
      RemoveAtm(&Atm, &PrecAtm, InizAtm);
    } else {
      PrecAtm    = Atm;
      Atm -> Num = Count++;
      Atm        = Atm -> Ptr;
    }
  }
}


/**** Remove water molecules ****/

void RemoveH2O(ATOMO **InizAtm, VG_ULONG *TotAtm)
{
  ATOMO               *Atm     = *InizAtm;
  ATOMO               *PrecAtm = *InizAtm;
  VG_ULONG            Count    = 1;

  PrintProg(MSG_UTILS_RMWPROG);

  if (RenameWat(*InizAtm)) {
    while(Atm) {
      if ((Atm -> ResName.C[0] ==  'H') && (Atm -> ResName.C[1] ==  'O') &&
          (Atm -> ResName.C[2] ==  'H')) {
        --*TotAtm;
        RemoveAtm(&Atm, &PrecAtm, InizAtm);
      } else {
        PrecAtm    = Atm;
        Atm -> Num = Count++;
        Atm        = Atm -> Ptr;
      }
    }
  }
}


/**** Remove a residue ****/

void RemoveRes(register ATOMO **Atm, ATOMO **PrecAtm, ATOMO **InizAtm,
               register VG_ULONG *TotAtm)
{
  VG_LONG          ResSeq = (*Atm) -> ResSeq.L;

  do {
    RemoveAtm(Atm, PrecAtm, InizAtm);
    --*TotAtm;
  } while((*Atm) && ((*Atm) -> ResSeq.L == ResSeq));
}


/**** Rename a residue ****/

ATOMO *RenameRes(register ATOMO *Atm, VG_LONG Res)
{
  VG_QCHAR          ResName;
  register ATOMO    *PrecAtm = Atm;
  register VG_LONG  ResSeq   = Atm -> ResSeq.L;

  if ((Atm -> ResName.C[3] == '+') || (Atm -> ResName.C[3] == '-'))
    ((char *)&Res)[2] = Atm -> ResName.C[3];

  ResName.L = Atm -> ResName.L;
  while((Atm) && (ResName.L == Atm -> ResName.L) && (ResSeq == Atm -> ResSeq.L)) {
    Atm -> ResName.L = Res;
    PrecAtm = Atm;
    Atm     = Atm -> Ptr;
  }

  return PrecAtm;
}


/**** Rename waters ****/

VG_BOOL RenameWat(register ATOMO *Atm)
{
  register ATOMO        *H1, *H2;

  VG_BOOL   Ret  = FALSE;

  while(Atm) {
    H1 = Atm -> Conn[0];
    H2 = Atm -> Conn[1];

#ifdef LITTLE_ENDIAN
    if ((Atm -> Elem.S == 0x4f) && (Atm -> NSost == 2) &&
        (H1 -> Elem.S == 0x48) && (H2 -> Elem.S == 0x48)) {
      Atm -> ResName.L = 0x00484F48;
      H1  -> ResName.L = 0x00484F48;
      H2  -> ResName.L = 0x00484F48;
#else
    if ((Atm -> Elem.S == 0x4f00) && (Atm -> NSost == 2) &&
        (H1 -> Elem.S == 0x4800) && (H2 -> Elem.S == 0x4800)) {
      Atm -> ResName.L = 0x484F4800;
      H1  -> ResName.L = 0x484F4800;
      H2  -> ResName.L = 0x484F4800;
#endif
      Ret = TRUE;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Ret;
}


/**** Renumbers residues ****/

void RenRes(ATOMO *Atm, VG_ULONG ResSeq, VG_BOOL ActOnly)
{
  char          Tmp[6];

  VG_BOOL       Force       = FALSE;
  VG_BYTE       PrecChainID = Atm -> ChainID;
  VG_LONG       PrecResName = Atm -> ResName.L;
  VG_LONG       PrecResSeq  = Atm -> ResSeq.L;

  PrintProg(MSG_UTILS_RENRESPROG);

  while(Atm) {
    if ((!ActOnly) || (Atm -> Active)) {
      if ((Force) ||
          (Atm -> ChainID   != PrecChainID) ||
          (Atm -> ResName.L != PrecResName) ||
          (Atm -> ResSeq.L  != PrecResSeq)) {
        PrecChainID = Atm -> ChainID;
        PrecResName = Atm -> ResName.L;
        PrecResSeq  = Atm -> ResSeq.L;
        Force       = FALSE;
        ++ResSeq;
      }
      sprintf(Tmp, "%d", ResSeq);
      Str2Qchar(&Atm -> ResSeq, Tmp);
    }
    if ((Atm -> Flags & VG_ATMF_SEGEND) ||
        (Atm -> Flags & VG_ATMF_MOLEND))
      Force = TRUE;
    Atm = Atm -> Ptr;
  }
}


/**** Pop the residue names ****/

void ResNamePop(register ATOMO *Atm, VG_ULONG *ResNameBuf)
{
  register VG_ULONG     *Ptr = ResNameBuf;

  while(Atm) {
    Atm -> ResName.L = *Ptr;
    Atm  = Atm -> Ptr;
    ++Ptr;
  } /* End of while */
}


/**** Push the atom types ****/

VG_ULONG *ResNamePush(register ATOMO *Atm, VG_ULONG TotAtm)
{
  VG_ULONG              *ResNameBuf;
  register VG_ULONG     *Ptr;

  if ((ResNameBuf = (VG_ULONG *)Alloca(TotAtm * sizeof(VG_ULONG))) != NULL) {
    Ptr = ResNameBuf;
    while(Atm) {
      *Ptr = Atm -> ResName.L;
      Atm  = Atm -> Ptr;
      ++Ptr;
    } /* End of while */
  }

  return ResNameBuf;
}


/**** Check atom selection ****/

VG_BOOL SelAtmChk(ATMLST *Sel)
{
  VG_BOOL               Ret;

  int                   Count = 0;
  register ATOMO        *Atm  = BegAtm;

  while(Atm) {
    if ((Atm -> Name.L == Sel -> Name.L) &&
        ((!Sel -> ResName.L) || (Atm -> ResName.L == Sel -> ResName.L)) &&
        ((!Sel -> ResSeq.L) || (Atm -> ResSeq.L == Sel -> ResSeq.L))) {
      Sel -> Atm = Atm;
      ++Count;
    }
    Atm = Atm -> Ptr;
  }

  if (Count != 1) {
    Sel -> Atm = NULL;
    Ret        = FALSE;
  } else Ret = TRUE;

  return Ret;
}


/**** Remove all selected atoms ****/

void SelAtmRemAll(register ATMSEL *Sel)
{
  register ATMLST       *Rem;
  register ATMLST       *List = Sel -> BegList;

  while(List) {
    Rem  = List;
    List = List -> Ptr;
    FREE(Rem);
  } /* End of while */

  Sel -> BegList = NULL;
  Sel -> LastAtm = NULL;
}


/**** Return the pointer of a selection ****/

ATMSEL *SelRetPtr(VG_WORD Index, VG_WORD Type)
{
  register ATMSEL       *Sel = BegSel;

  while(Sel) {
    if ((Type == SL_NONE) || (Type == Sel -> Type))
      --Index;
    if (Index)
      Sel = Sel -> Ptr;
    else
      break;
  } /* End of while */

  return Sel;
}


/**** Remove all selections ****/

void SelRemAll(void)
{
  register ATMSEL       *NextSel;

  register ATMSEL       *Sel = BegSel;

  while(Sel) {
    SelAtmRemAll(Sel);
    NextSel = Sel -> Ptr;
    FREE(Sel);
    Sel = NextSel;
  } /* End of while */

  BegSel  = NULL;
  LastSel = NULL;
}


/**** Remove a selection by a single atom ****/

#ifdef __VG_OPENGL

VG_BOOL SelRemByAtm(ATOMO *Atm)
{
  ATMLST       *List;
  ATMSEL       *PrecSel, *Sel;

  if (!BegSel) return FALSE;

  PrecSel = NULL;
  for(Sel = BegSel; Sel; Sel = Sel -> Ptr) {
    for(List = Sel -> BegList; (List); List = List -> Ptr) {
      if (Atm == List -> Atm) {
        SelAtmRemAll(Sel);
        if (Sel == LastSel) LastSel = PrecSel;
        if (Sel == BegSel) {
          BegSel = Sel -> Ptr;
        } else PrecSel -> Ptr = Sel -> Ptr;
        FREE(Sel);
        return TRUE;
      }
    } /* End of for */
    PrecSel = Sel;
  } /* End of while */

  return FALSE;
}
#endif


/**** Skip file lines ****/

VG_BOOL SkipLines(FILE *FH, char *Buf, VG_ULONG Lines)
{
  while(Lines > 0) {
    if (!fgets(Buf, 255, FH)) {
      PrintDosErr();
      return FALSE;
    }
   --Lines;
  } /* End of while */

  return TRUE;
}


/**** Translates a string into a DCHAR ****/

void Str2Dchar(register VG_DCHAR *Res, register char *Str)
{
  (*Res).S    = 0;
  (*Res).C[0] = Str[0];
  if (Str[0])(*Res).C[1] = Str[1];
}


/**** Translates a string into a OCHAR ****/

void Str2Ochar(VG_OCHAR *Res, register char *Str)
{
  register char      *Dest = (*Res).C;
  register VG_UWORD  k;

  (*Res).DL = 0LL;
  for(k = 0;(*Str) && (k < 8); ++k) {
   *Dest = *Str;
   ++Str;
   ++Dest;
  }
}


/**** Translates a string into a QCHAR ****/

void Str2Qchar(VG_QCHAR *Res, register char *Str)
{
  register char      *Dest = (*Res).C;
  register VG_UWORD  k;

  (*Res).L = 0;
  for(k = 0;(*Str) && (k < 4); ++k) {
    *Dest = *Str;
    ++Str;
    ++Dest;
  } /* End of for */
}


/**** Check if a string is really empty ****/

char *StrExist(char *Str)
{
  while((*Str) &&
        ((*Str == ' ') ||
         (*Str == '\t') ||
         (*Str == 10) ||
         (*Str == 13))) ++Str;

  if (*Str) return Str;

  return NULL;
}


/**** Set the residue type ****/

void SetTypeRes(VG_QCHAR *Res, VG_UBYTE Car)
{
  if (Res -> L != *(VG_LONG *)"ASP-") {
    if ((Res -> C[3] == '+') || (Res -> C[3] == '-')) {
      Res -> C[2] = Res -> C[3];
    }
  } else Res -> L = *(VG_LONG *)"AP-";

  Res -> C[3] = Car;
}


/**** Free a memory list of surface dots ****/

void SrfFree(register VG_SURFACE *Dot)
{
  register VG_SURFACE *Succ;

  if (Dot) {
    do {
      Succ = Dot -> Next;
      SOFREE(Dot);
      Dot = Succ;
    } while(Dot);
  }
}


/**** Change the endian (if needed) and writes a long ****/

int SwapWrite(FILE *FH, void *Val)
{
  VG_LONG       Num = *(VG_LONG *)Val;
#ifdef LITTLE_ENDIAN
  Swap(&Num);
#endif

  return (int)fwrite(&Num, sizeof(VG_ULONG), 1, FH);
}


/**** Change a string to uppercase ****/

void Upper(register char *Str)
{
  do *Str = (char)toupper(*Str); while(*++Str);
}


/**** Input prompt for a string ****/

void VegaGetString(char *Prompt, char *Str, VG_UWORD Len)
{
  printf("%s: ", Prompt);
  fgets(Str, Len, stdin);
  Str[strlen(Str) - 1] = 0;
}


/**** Wildcard compare case insensible ****/

int Wildncmp(register char *Wild, register char *String, int Len)
{
  register char         *Cp, *End, *Mp;

  if (Len) End = String + Len;
  else End = String + strlen(String);

  while ((String != End) && (*String) && (*Wild != '*')) {
    if ((toupper(*Wild) != toupper(*String)) && (*Wild != '?')) {
      return 0;
    }
    Wild++;
    String++;
  } /* End of while */

  while ((String != End) && (*String)) {
    if (*Wild == '*') {
      if (!*++Wild) return 1;
      Mp = Wild;
      Cp = String + 1;
    } else if ((toupper(*Wild) == toupper(*String)) || (*Wild == '?')) {
      Wild++;
      String++;
    } else {
      Wild   = Mp;
      String = Cp++;
    }
  } /* End of while */

  while (*Wild == '*') Wild++;

  return !*Wild;
}

