
/*************************************************
****    VEGA - Atom list managment routines   ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "atom.h"
#include "bond.h"
#include "count.h"

#ifdef __VG_OPENGL
#ifdef WIN32
#  include <windows.h>
#endif
#  include "gl_global.h"
#  include "gl_objs.h"
#  include "gl_update.h"
#  include "selmol.h"
#endif

#ifdef AMIGA
#  include <exec/types.h>
#  include <exec/memory.h>
#  include <exec/execbase.h>
#endif


/**** Allocates the description structure for one atom  ****/
/**** using the S.O. specific memory allocation routines ****/

ATOMO *AllocAtm(ATOMO **InizAtm, VG_ULONG *TotAtomi)
{
  register ATOMO   *Atm;
  static ATOMO     *PrecAtm;

  if ((Atm = (ATOMO *)SOALLOC(sizeof(ATOMO)))) {
    if ((!*TotAtomi) || (!*InizAtm)) {
      *InizAtm = Atm;
    } else {
      PrecAtm -> Ptr  = Atm;
      Atm     -> Prev = PrecAtm;
    }
    ++*TotAtomi;
    PrecAtm           = Atm;                 /* Bind the list         */
    Atm -> ResName.L  = *(VG_LONG *)"UNK\0"; /* Unknown residue name  */
    *Atm  -> ResSeq.C = '1';                 /* Residue one           */
    Atm  -> ChainID   = ' ';                 /* Default chain         */
    Atm  -> Num       = ++AtmLoaded;         /* Squential atom number */
    *Atm -> Elem.C    = '?';                 /* Unknown element       */
    *Atm -> Pot.C     = '?';                 /* Unknown potential     */
    Atm -> Rad        = 1.8;                 /* Atomic radius         */
    Atm -> ColorID    = VGCOL_WHITE;         /* White color           */
    Atm -> Active     = 1;                   /* Active                */
  } else {
    if (*TotAtomi) {
      FreeAtm(*InizAtm);
      *InizAtm = NULL;
    }
    CatErr(MSG_ERR_UTILS_OUTOFMEM);
  }

  return Atm;
}


/**** Allocate the atom table for random access to the list ****/

ATOMO **AllocAtmTable(ATOMO *Atm, VG_ULONG TotAtm, VG_BOOL All)
{
  ATOMO         **AtmTable, **Ptr;

  if (!All) TotAtm = CountAtms(Atm, FALSE);

  if ((AtmTable = (ATOMO **)Alloca(sizeof(ATOMO *) * TotAtm)) != NULL) {
    Ptr = AtmTable;
    while(Atm) {
      if ((All) || (!(Atm -> Flags & VG_ATMF_CENTROID)))
        *Ptr++ = Atm;
     Atm    = Atm -> Ptr;
    } /* End of while */
  }

  return AtmTable;
}

/**** Convert an tom into a string ****/

char *Atm2Str(char *Dest, ATOMO *Atm)
{
  char          *Tmp = ":*:";

  if (Atm -> ChainID != ' ')
    Tmp[1] = Atm -> ChainID;

  strcpy(Dest, Qchar2Str(&Atm -> Name));
  strcat(Dest, ":");
  strcat(Dest, Qchar2Str(&Atm -> ResName));
  strcat(Dest, ":");
  strcat(Dest, Qchar2Str(&Atm -> ResSeq));
  strcat(Dest, Tmp);
  sprintf(Dest + strlen(Dest), "%d", MolListFindMol(Atm));

  return Dest;
}


/**** Attach a new atom ****/

ATOMO *AttachAtm(ATOMO *Atm)
{
  ATOMO         *Att;

  if ((Att = AllocAtm(&BegAtm, &TotalAtm)) != NULL) {
    if (Atm -> NSost < MAXBOND) {
      Att -> NSost     = 1;
      Att -> Conn[0]   = Atm;

      Att -> ChainID   = Atm -> ChainID;
      Att -> ColorID   = Atm -> ColorID;
      Att -> Flags     = (Atm -> Flags &~ VG_ATMF_HETATM);
      Att -> Pot.DL    = Atm -> Pot.DL;
      Att -> ResName.L = Atm -> ResName.L;
      Att -> ResSeq.L  = Atm -> ResSeq.L;
    } else {
      FreeAtm(Att);
      Att = NULL;
    }
  }

  return Att;
}


/**** Update the centroid coordinates ****/

void CentroidUpdate(ATOMO *Atm)
{
  VG_ULONG      k;

  Atm -> x = 0.0f;
  Atm -> y = 0.0f;
  Atm -> z = 0.0f;

  for(k = 0; (k < MAXBOND) && (Atm -> Conn[k]); ++k) {
    Atm -> x += Atm -> Conn[k] -> x;
    Atm -> y += Atm -> Conn[k] -> y;
    Atm -> z += Atm -> Conn[k] -> z;
  } /* End of for */

  Atm -> x /= (float)k;
  Atm -> y /= (float)k;
  Atm -> z /= (float)k;
}


/**** Update the coordinates of all centroides ****/

VG_BOOL CentroidUpdateAll(ATOMO *Atm)
{
  VG_BOOL       Ret = FALSE;

  while(Atm) {
    if ((Atm -> Flags & VG_ATMF_CENTROID) &&
        (!(Atm -> Flags & VG_ATMF_CENTROIDFIX))) {
      CentroidUpdate(Atm);
      Ret = TRUE;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Ret;
}


/**** Clone atoms ****/

ATOMO *CloneAtms(ATOMO *Src, VG_ULONG TotAtm)
{
  ATOMO         **Buf, **Conn, *Dest, *PrecAtm;
  VG_BYTE       NSost;

  ATOMO         *Start   = NULL;

  if ((Buf = (ATOMO **)Alloca(sizeof(ATOMO *) * TotAtm))) {
    Conn = Buf;
    while(Src) {
      if ((Dest = Alloca(sizeof(ATOMO)))) {
        *Conn = Dest;
#ifdef WIN32
        CopyMemory(Dest, Src, sizeof(ATOMO));
#else
        memcpy(Dest, Src, sizeof(ATOMO));
#endif
        if (Start) {
          PrecAtm -> Ptr  = Dest;
          Dest    -> Prev = PrecAtm;
        } else Start = Dest;
        PrecAtm = Dest;
      } else {
        FREE(Buf);
        FreeAtm(Start);
        return NULL;
      }
      Src = Src -> Ptr;
      ++Conn;
    } /* End of while */

    /**** Set the connectivity ****/

    for(Dest = Start; Dest; Dest = Dest -> Ptr) {
      Conn = Dest -> Conn;
      for(NSost = 0; NSost < Dest -> NSost; ++NSost) {
        *Conn = Buf[(*Conn) -> Num - 1];
        ++Conn;
      } /* End of for */
    } /* End of for */
    FREE(Buf);
  }

  return Start;
}


/**** Find residue by its atom ****/

ATOMO *FindResByAtm(ATOMO *ResAtm, ATOMO **LastAtm, VG_ULONG *ResLen)
{
  ATOMO         *Atm, *Beg, *Last;

  VG_LONG       Len = 0;

  /**** Search the beginning ****/

  for(Atm = ResAtm -> Prev;
      (Atm) &&
      (!(Atm -> Flags & VG_ATMF_SEGEND)) &&
      (!(Atm -> Flags & VG_ATMF_MOLEND)) &&
      (Atm -> ResName.L == ResAtm -> ResName.L) &&
      (Atm -> ResSeq.L == ResAtm -> ResSeq.L) &&
      (Atm -> ChainID == ResAtm -> ChainID);
      Atm = Atm -> Prev) {
    ++Len;
  } /* End of for */

  if (Beg) {
    if (!Atm) Beg = BegAtm;
    else Beg = Atm -> Ptr;
  }

  /**** Search the end ****/

  for(Atm = ResAtm;
      (Atm) &&
      (Atm -> ResName.L == ResAtm -> ResName.L) &&
      (Atm -> ResSeq.L == ResAtm -> ResSeq.L) &&
      (Atm -> ChainID == ResAtm -> ChainID);
      Atm = Atm -> Ptr) {
    Last = Atm;
    ++Len;
    if ((Atm -> Flags & VG_ATMF_SEGEND) ||
        (Atm -> Flags & VG_ATMF_MOLEND)) break;
  } /* End of for */

  if (ResLen)  *ResLen  = Len;
  if (LastAtm) *LastAtm = Last;

  return Beg;
}


/**** Free a memory list of atoms ****/

void FreeAtm(register ATOMO *Atm)
{
  register ATOMO	*Succ;

  if (Atm) {
    do {
      Succ = Atm -> Ptr;
      SOFREE(Atm);
      Atm = Succ;
    } while(Atm);
  }
}


/**** Insert an atom in the list ****/

ATOMO *InsertAtm(ATOMO *PrecAtm, VG_ULONG *TotAtomi)
{
  register ATOMO   *Atm;

  if ((Atm = (ATOMO *)SOALLOC(sizeof(ATOMO)))) {
    Atm -> Ptr        = PrecAtm -> Ptr;
    Atm -> Prev       = PrecAtm;

    if (PrecAtm -> Ptr) PrecAtm -> Ptr -> Prev = Atm;

    PrecAtm -> Ptr    = Atm;

    Atm -> ResName.L  = PrecAtm -> ResName.L;
    Atm  -> ResSeq.L  = PrecAtm -> ResSeq.L;
    Atm  -> ChainID   = PrecAtm -> ChainID;
    Atm  -> Flags     = PrecAtm -> Flags &~ VG_ATMF_HETATM;
    *Atm -> Elem.C    = '?';                 /* Unknown element       */
    *Atm -> Pot.C     = '?';                 /* Unknown potential     */
    Atm -> Rad        = 1.8;                 /* Atomic radius         */
    Atm -> ColorID    = VGCOL_WHITE;         /* White color           */
    Atm -> Active     = 1;                   /* Active                */

    PrecAtm -> Flags  = PrecAtm -> Flags &~ VG_ATMF_MOLEND;
    PrecAtm -> Flags  = PrecAtm -> Flags &~ VG_ATMF_SEGEND;
    if (PrecAtm == LastAtm) LastAtm = Atm;
    ++*TotAtomi;
  } else CatErr(MSG_ERR_UTILS_OUTOFMEM);

  return Atm;
}


/**** Remove a single atom from list ****/

void RemoveAtm(ATOMO **Atm, ATOMO **PrecAtm, ATOMO **InizAtm)
{
#ifdef __VG_OPENGL
  if ((ViewPrefs.MoveMol) && (*Atm == ViewPrefs.MoveMol)) {
    if ((*Atm) -> Flags & VG_ATMF_MOLEND) {
      CatPrintf(stdout, MSG_DLGMVMOL_CONNWORLD);
      ViewPrefs.MoveMol = NULL;
      ChangeMovePopup();
    } else ViewPrefs.MoveMol = (*Atm) -> Ptr;
  }

  /**** Update graphic and windows ****/

  if (GLOBSW_OPENGL) {
    GL_ObjRemoveAtm(*Atm);
    if (SelRemByAtm(*Atm)) {
      DlgSelectUpdate();
      DlgTorsionUpdate();
    }
  }
#endif

  if (*Atm == *InizAtm) {
    *InizAtm = (*Atm) -> Ptr;
    *PrecAtm = (*Atm) -> Ptr;
    if (*InizAtm) (*InizAtm) -> Prev = NULL;
    FREE(*Atm);
    *Atm = *PrecAtm;
  } else {
    (*PrecAtm) -> Ptr    = (*Atm) -> Ptr;
    (*PrecAtm) -> Flags |= ((*Atm) -> Flags &~ (VG_ATMF_HETATM|VG_ATMF_CENTROID));
    if ((*Atm) -> Ptr) (*Atm) -> Ptr -> Prev  = *PrecAtm;

    if (*Atm == LastAtm) LastAtm = *PrecAtm;

    FREE(*Atm);
    *Atm = (*PrecAtm) -> Ptr;
  }
}


/**** Renumber atoms ****/

void RenAtm(register ATOMO *Atm, register VG_ULONG StartNum)
{
  while(Atm) {
    Atm -> Num = StartNum++;
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Set an univocal atom name ****/

void SetAtmName(ATOMO *Atm2)
{
  ATOMO         *Atm1, *BegRes;
  char          *Ptr;
  VG_QCHAR      NewName;
  VG_ULONG      j, k, ResAtm;

  NewName.L    = 0;
  NewName.C[0] = Atm2 -> Elem.C[0];
  NewName.C[1] = Atm2 -> Elem.C[1];
  Ptr = NewName.C + 1;
  if (NewName.C[1]) {
    NewName.C[1] = toupper(NewName.C[1]);
    ++Ptr;
  }

  BegRes = FindResByAtm(Atm2, NULL, &ResAtm);

  for(k = 1; k < 100; ++k) {
    if (k >= 10) {
      j = k / 10;
      Ptr[0] = j + '0';
      Ptr[1] = k + '0' - j * 10;
    } else *Ptr = k + '0';


    for(j = 0, Atm1 = BegRes;
        (j < ResAtm) && (NewName.L != Atm1 -> Name.L);
        ++j, Atm1 = Atm1 -> Ptr);

    if (j == ResAtm) break;
  } /* End of for */

  Atm2 -> Name.L = NewName.L;
}


#ifdef __VG_OPENGL

/**** Add a centriod ****/

ATOMO *CentroidAdd(ATOMO *ParentAtm, VG_BOOL Fix)
{
  ATOMO         *Att;

  if ((Att = InsertAtm(ParentAtm, &TotalAtm)) != NULL) {
    Att -> Flags     |= VG_ATMF_CENTROID | (VG_ATMF_CENTROIDFIX * Fix);
    Att -> x          = ParentAtm -> x;
    Att -> y          = ParentAtm -> y;
    Att -> z          = ParentAtm -> z;
    Att -> Rad        = 0.0f;
    Att -> ColorID    = VGCOL_SAND;
    Att -> Elem.S     = VG_CENTROID_ELEM;
    Att -> Conn[0]    = ParentAtm;
    SetAtmName(Att);
    RenAtm(Att, ParentAtm -> Num + 1);
  }

  return Att;
}


/**** Add an atom to a centroid ****/

VG_LONG CentroidAddAtm(ATOMO *Ctr, ATOMO *Atm)
{
  VG_LONG              k;

  for(k = 0; (k < MAXBOND) && (Ctr -> Conn[k]); ++k) {
    if (Ctr -> Conn[k] == Atm) return 0;
  } /* End of for */

  if (k == MAXBOND) {
    for(k = 1; k < MAXBOND; ++k)
      Ctr -> Conn[k - 1] = Ctr -> Conn[k];
    Ctr -> Conn[MAXBOND - 1] = Atm;
    k = 2;
  } else {
    Ctr -> Conn[k] = Atm;
    k = 1;
  }

  CentroidUpdate(Ctr);

  return k;
}


/**** Centroid fix ****/

void CentroidFix(ATOMO *Atm)
{
  while(Atm) {
    if ((Atm -> Elem.C[0] == 'X') && (Atm -> Elem.C[1] == 'c'))
      Atm -> Flags |= VG_ATMF_CENTROID;
    else if (Atm -> Flags & VG_ATMF_CENTROID)
      Atm -> Elem.S = VG_CENTROID_ELEM;
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Remove all centroides ****/

void CentroidRemAll(ATOMO **InizAtm, VG_ULONG *TotAtm)
{
  ATOMO               *Atm     = *InizAtm;
  ATOMO               *PrecAtm = *InizAtm;
  VG_ULONG            Count    = (*InizAtm) -> Num;


  while(Atm) {
    if (Atm -> Flags & VG_ATMF_CENTROID) {
      --*TotAtm;
      RemoveAtm(&Atm, &PrecAtm, InizAtm);
    } else {
      PrecAtm    = Atm;
      Atm -> Num = Count++;
      Atm        = Atm -> Ptr;
    }
  } /* End of while */
}


/**** Duplicate atoms ****/

ATOMO *DuplicateAtms(ATOMO *Src, VG_ULONG TotAtm)
{
  ATOMO         **Buf, **Conn, *Dest;
  VG_BYTE       NSost;
  VG_ULONG      k, NumEnd;

  ATOMO         *Start   = NULL;
  VG_ULONG      NumStart = Src -> Num;

  if ((Buf = (ATOMO **)Alloca(sizeof(ATOMO *) * TotAtm))) {
    for(k = 0; k < TotAtm; ++k) {
      if ((Dest = AllocAtm(&Start, &TotalAtm))) {
        Buf[k] = Dest;
#ifdef WIN32
        CopyMemory(Dest, Src, sizeof(ATOMO));
#else
        memcpy(Dest, Src, sizeof(ATOMO));
#endif
        Dest -> Num = TotalAtm;
        Dest -> Ptr = NULL;
        NumEnd      = Src -> Num;
        Src         = Src -> Ptr;
      } else {
        FREE(Buf);
        FreeAtm(Start);
        return NULL;
      }
    } /* End of for */

    if (Dest) {
      LastAtm -> Ptr = Start;
      LastAtm        = Dest;
    }

    /**** Set the connectivity ****/

    Dest = Start;
    for(k = 0; k < TotAtm; ++k) {
      Conn = Dest -> Conn;
      for(NSost = 0; NSost < Dest -> NSost; ++NSost) {
        if (((*Conn) -> Num >= NumStart) && ((*Conn) -> Num <= NumEnd)) {
          *Conn = Buf[(*Conn) -> Num - NumStart];
        }
        ++Conn;
      }
      Dest = Dest -> Ptr;
    }
    FREE(Buf);
  }

  return Start;
}


/**** Get the complete atom name ****/

char *GetAtmFullName(char *Dest, ATOMO *Atm)
{
  char          *Ptr;
  static char   Buf[32];
  VG_BYTE       ChainID;

  if (Dest) Ptr = Dest;
  else Ptr = Buf;

  if (Atm -> ChainID == ' ') ChainID = '*';
  else ChainID = Atm -> ChainID;
  sprintf(Ptr, "%.4s:%.4s:%.4s:%c:%d",
                Atm -> Name.C, Atm -> ResName.C, Atm -> ResSeq.C, ChainID,
                MolListFindMol(Atm));

  return Ptr;
}


/**** Remove atoms using the pattern matching ****/

VG_ULONG RemoveAtmMatch(ATOMO *InizAtm, char *Sel)
{
  ATOMO         *Atm, *PrecAtm;
  char          *MyArg[5], NMolStr[12];
  VG_ULONG      AtmNum, NumArgs;
  VG_UBYTE      i;

  VG_ULONG      Count    = InizAtm -> Num;
  VG_ULONG      NMol     = 1;
  VG_ULONG      Removed  = 0;

  NumArgs = GL_MatchFormatPattern(&AtmNum, MyArg, Sel);

  PrecAtm = InizAtm -> Prev;
  if ((!NumArgs) && (AtmNum)) {

    /**** Atom number ****/

    for(Atm = InizAtm; Atm; ) {
      if (Atm -> Num == AtmNum) {
        for(i = 0; i < Atm -> NSost; ++i)
          BndUnlink(Atm -> Conn[i], Atm);
        --TotalAtm;
        RemoveAtm(&Atm, &PrecAtm, &BegAtm);
        ++Removed;
      } else {
        PrecAtm    = Atm;
        Atm -> Num = Count++;
        Atm        = Atm -> Ptr;
      }
    } /* End of for */
  } else {

    /**** Pattern matching ****/

    NMolStr[0] = '1';
    NMolStr[1] = 0;
    for(Atm = InizAtm; Atm; ) {
      if (GL_MatchAtm(Atm, MyArg, NumArgs, NMolStr)) {
        for(i = 0; i < Atm -> NSost; ++i)
          BndUnlink(Atm -> Conn[i], Atm);
        --TotalAtm;
        RemoveAtm(&Atm, &PrecAtm, &BegAtm);
        ++Removed;
      } else {
        PrecAtm    = Atm;
        Atm -> Num = Count++;
        if (Atm -> Flags & VG_ATMF_MOLEND)
          sprintf(NMolStr, "%d", ++NMol);
        Atm        = Atm -> Ptr;
      }
    } /* End of for */
  }

  return Removed;
}

#endif
