
/*************************************************
****           VEGA - Count routines          ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "count.h"


/**** Count the active atoms ****/

VG_ULONG CountActiveAtms(register ATOMO *Atm)
{
  register VG_ULONG     Count = 0;

  while(Atm) {
    if (Atm -> Active) ++Count;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Count;
}


/**** Count the atoms excluding centroids ****/

VG_ULONG CountAtms(register ATOMO *Atm, register VG_BOOL ActiveOnly)
{
  register VG_ULONG     Count = 0;

  while(Atm) {
    if (((!ActiveOnly) || (Atm -> Active)) &&
        (!(Atm -> Flags & VG_ATMF_CENTROID)))
      ++Count;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Count;
}


/**** Count the number of angles ****/

VG_ULONG CountAngle(ATOMO **Bond1)
{
  ATOMO                 **Bond2;

  VG_ULONG              Angles = 0;

  while(*Bond1) {
    for(Bond2 = Bond1 + 2; *Bond2; Bond2 += 2) {
      if ((Bond1[1] == Bond2[0]) ||
          (Bond1[1] == Bond2[1]) ||
          (Bond1[0] == Bond2[0]) ||
          (Bond1[0] == Bond2[1])) {
        ++Angles;
      }
    } /* End of for */
    Bond1 += 2;
  } /* End of while */

  return Angles;
}


/**** Count the heavy atoms bonded to one ****/

VG_ULONG CountBndHeavyAtms(register ATOMO *Atm1)
{
  register ATOMO        *Atm2;
  register VG_BYTE      k;

  register VG_ULONG     Count = 0;

  for(k = 0; k < Atm1 -> NSost; ++k) {
    Atm2 = Atm1 -> Conn[k];
    if ((Atm2 -> Elem.S != VG_ELEM_H) &&
        (Atm2 -> Elem.S != VG_ELEM_D) &&
        (Atm2 -> Elem.S != VG_ELEM_T))
      ++Count;
  } /* End of for */

  return Count;
}

/**** Count the heavy atoms bonded to one ****/

VG_ULONG CountBndHyd(register ATOMO *Atm1)
{
  register VG_BYTE      k;

  register VG_ULONG     Count = 0;

  for(k = 0; k < Atm1 -> NSost; ++k) {
    if (Atm1 -> Conn[k] -> Elem.S == VG_ELEM_H)
      ++Count;
  } /* End of for */

  return Count;
}


/**** Count the connected free oxygens ****/

VG_ULONG CountBndFreeOx(register ATOMO *Atm1)
{
  register ATOMO        *Atm2;
  register VG_BYTE      k;

  register VG_ULONG     Count = 0;

  for(k = 0; k < Atm1 -> NSost; ++k) {
    Atm2 = Atm1 -> Conn[k];
    if ((Atm2 -> Elem.S == VG_ELEM_O) &&
        (CountBndHeavyAtms(Atm2) == 1))
      ++Count;
  } /* End of for */

  return Count;
}


/**** Count the number of bonds ****/

VG_ULONG CountBond(register ATOMO *Atm)
{
  register VG_UWORD  k;
  register VG_ULONG  Count = 0;

  while(Atm) {
    for(k = 0; k < (VG_UWORD)Atm -> NSost; ++k) {
      if (Atm -> Conn[k] -> Num > Atm -> Num)
        ++Count;
    } /* End of for */
    Atm = Atm -> Ptr;
  } /* End of for */

  return Count;
}


/**** Count the chains ****/

VG_ULONG CountChain(ATOMO *Atm)
{
  VG_BYTE      ChainBuf[256], *ChainPtr;

  VG_ULONG     Count   = 0;

#ifdef WIN32
  ZeroMemory(ChainBuf, 256);
#else
  memset(ChainBuf, 0, 256);
#endif

  while(Atm) {
    for(ChainPtr = ChainBuf; (*ChainPtr) && (*ChainPtr != Atm -> ChainID); ++ChainPtr)
      ++Count;

    if (!*ChainPtr) *ChainPtr = Atm -> ChainID;
    Atm = Atm -> Ptr;
  }

  return Count;
}


/**** Count the number of H-bond acceptors ****/

VG_ULONG CountHbAcceptor(register ATOMO *Atm)
{
  register VG_ULONG     Acceptors = 0;

  while(Atm) {
    if ((*Atm -> Pot.C != 'h') &&
        (*Atm -> Pot.C != 'x')) ++Acceptors;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Acceptors;
}


/**** Count the number of H-bond donors ****/

VG_ULONG CountHbDonor(register ATOMO *Atm)
{
  register VG_ULONG     Donors = 0;

  while(Atm) {
    if (*Atm -> Pot.C == 'h') ++Donors;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Donors;
}


/**** Count the number of impropers ****/

VG_ULONG CountImproper(ATOMO **Angle1, ATOMO **ImpDest)
{
  ATOMO                 **Angle2, *Imp[4];

  ATOMO                 *PrecImp1  = NULL;
  ATOMO                 *PrecImp2  = NULL;
  VG_ULONG              Impropers  = 0;

  while(*Angle1) {
    for(Angle2 = Angle1 + 3; *Angle2; Angle2 += 3) {
      Imp[1] = NULL;
      if ((Angle1[1] == Angle2[1]) && (Angle1[1] -> NSost == 3)) {
        Imp[0] = Angle1[1];
        if (Angle1[0] == Angle2[0]) {
          Imp[1] = Angle1[0];
          Imp[2] = Angle1[2];
          Imp[3] = Angle2[2];
        } else if (Angle1[0] == Angle2[2]) {
          Imp[1] = Angle1[0];
          Imp[2] = Angle1[2];
          Imp[3] = Angle2[0];
        } else if (Angle1[2] == Angle2[0]) {
          Imp[1] = Angle1[2];
          Imp[2] = Angle1[0];
          Imp[3] = Angle2[2];
        }
      }

      if ((Imp[1]) && (Imp[1] != PrecImp2) && (Imp[2] != PrecImp1)) {
        if (ImpDest) {
          *ImpDest++ = Imp[0];
          *ImpDest++ = Imp[1];
          *ImpDest++ = Imp[2];
          *ImpDest++ = Imp[3];
        }
/*
        DosPrintf(stdout, "%d %d %d %d\n",
                 Imp[0] -> Num, Imp[1] -> Num, Imp[2] -> Num, Imp[3] -> Num);
*/
        PrecImp1 = Imp[1];
        PrecImp2 = Imp[2];
        ++Impropers;
      }
    } /* End of for */
    Angle1 += 3;
  } /* End of while */

  return Impropers;
}


/**** Count the number of residues ****/

VG_ULONG CountRes(register ATOMO *Atm)
{
  register VG_ULONG  PrecResName = 0;
  register VG_ULONG  PrecResSeq  = 0;
  register VG_ULONG  TotRes      = 0;

  while(Atm) {
    if ((Atm -> ResName.L != (VG_LONG)PrecResName) ||
        (Atm -> ResSeq.L  != (VG_LONG)PrecResSeq)) {
      PrecResName = Atm -> ResName.L;
      PrecResSeq  = Atm -> ResSeq.L;
      ++TotRes;
    }
    Atm = Atm -> Ptr;
  }

  return TotRes;
}


/**** Count the number of torsions ****/

VG_ULONG CountTorsion(ATOMO **Angle1, ATOMO **TorDest)
{
  ATOMO                 **Angle2, *Tor[4];
  VG_BOOL               Keep;
  VG_ULONG              Count, j, k, l;

  VG_ULONG              Torsions = 0;

  while(*Angle1) {
    for(Angle2 = Angle1 + 3; *Angle2; Angle2 += 3) {
      Tor[0] = NULL;
      if (((Angle1[0] == Angle2[0]) && (Angle1[1] == Angle2[1])) ||
          ((Angle1[0] == Angle2[1]) && (Angle1[1] == Angle2[0]))) {
        Tor[0] = Angle1[2];
        Tor[1] = Angle1[1];
        Tor[2] = Angle1[0];
        Tor[3] = Angle2[2];
      } else if (((Angle1[1] == Angle2[0]) && (Angle1[2] == Angle2[1])) ||
                 ((Angle1[1] == Angle2[1]) && (Angle1[2] == Angle2[0]))) {
        Tor[0] = Angle1[0];
        Tor[1] = Angle1[1];
        Tor[2] = Angle1[2];
        Tor[3] = Angle2[2];
      } else if (((Angle1[0] == Angle2[1]) && (Angle1[1] == Angle2[2])) ||
                 ((Angle1[0] == Angle2[2]) && (Angle1[1] == Angle2[1]))) {
        Tor[0] = Angle1[2];
        Tor[1] = Angle1[1];
        Tor[2] = Angle1[0];
        Tor[3] = Angle2[0];
      } else if (((Angle1[1] == Angle2[1]) && (Angle1[2] == Angle2[2])) ||
                 ((Angle1[1] == Angle2[2]) && (Angle1[2] == Angle2[1]))) {
        Tor[0] = Angle1[0];
        Tor[1] = Angle1[1];
        Tor[2] = Angle1[2];
        Tor[3] = Angle2[0];
      }

      if (Tor[0]) {
        Keep = TRUE;
        for(j = 0; (Keep) && (j < 4); ++j) {
          Count = 0;
          for(k = 0; (Keep) && (k < (VG_ULONG)Tor[j] -> NSost); ++k) {
            for(l = 0; (Keep) && (l < 4); ++l) {
              if ((l != j) && (Tor[j] -> Conn[k] == Tor[l])) {
                if (++Count == 3) Keep = FALSE;
              }
            } /* End of for */
          } /* End of for */
       } /* End of for */

        if (Keep) {
          if (TorDest) {
            *TorDest++ = Tor[0];
            *TorDest++ = Tor[1];
            *TorDest++ = Tor[2];
            *TorDest++ = Tor[3];
          }
/*
          DosPrintf(stdout, "%d %d %d %d\n",
                   Tor[0] -> Num, Tor[1] -> Num, Tor[2] -> Num, Tor[3] -> Num);
*/
          ++Torsions;
        }
      }
    } /* End of for */
    Angle1 += 3;
  } /* End of while */

  return Torsions;
}


/**** Generate the bond matrix ****/

VG_ULONG GenAngleMtx(ATOMO **Bond1, ATOMO ***AngleMtx)
{
  ATOMO         **Bond2, **Ptr;
  VG_ULONG      j, k;

  VG_ULONG      AngleNum = CountAngle(Bond1);

  if (*AngleMtx = (ATOMO **)Alloca((AngleNum + 1) * sizeof(ATOMO *) * 3)) {
    Ptr = *AngleMtx;
    while(*Bond1) {
      for(Bond2 = Bond1 + 2; *Bond2; Bond2 += 2) {
        for(j = 0; j < 2; ++j) {
          for(k = 0; k < 2; ++k) {
            if (Bond1[j] == Bond2[k]) {
              *Ptr++ = Bond1[!j];
              *Ptr++ = Bond1[j];
              *Ptr++ = Bond2[!k];
            }
          } /* End of for */
        } /* End of for */
      } /* End of for */
      Bond1 += 2;
    } /* End of while */
  } else AngleNum = 0;

  return AngleNum;
}


/**** Generate the bond matrix ****/

VG_ULONG GenBondMtx(ATOMO *Atm1, ATOMO ***BondMtx)
{
  ATOMO         **Ptr, *Atm2;
  VG_ULONG      k;

  VG_ULONG      BondNum = CountBond(Atm1);

  if (*BondMtx = (ATOMO **)Alloca((BondNum + 1) * sizeof(ATOMO *) * 2)) {
    Ptr = *BondMtx;
    while(Atm1) {
      for(k = 0; k < (VG_ULONG)Atm1 -> NSost; ++k) {
        Atm2 = Atm1 -> Conn[k];
        if (Atm2 -> Num > Atm1 -> Num) {
          *Ptr++ = Atm1;
          *Ptr++ = Atm2;
        }
      } /* End of for */
      Atm1 = Atm1 -> Ptr;
    } /* End of while */
  } else BondNum = 0;

  return BondNum;
}


/**** Generate the improper matrix ****/

VG_ULONG GenImproperMtx(ATOMO **Angles, ATOMO ***ImproperMtx)
{
  VG_ULONG      ImproperNum = CountImproper(Angles, NULL);

  if (ImproperNum) {
    if (*ImproperMtx = (ATOMO **)Alloca((ImproperNum + 1) * sizeof(ATOMO *) * 4)) {
      CountImproper(Angles, *ImproperMtx);
    } else ImproperNum = 0;
  }

  return ImproperNum;
}


/**** Generate the torsion matrix ****/

VG_ULONG GenTorsionMtx(ATOMO **Angles, ATOMO ***TorsionMtx)
{
  VG_ULONG      TorsionNum = CountTorsion(Angles, NULL);

  if (TorsionNum) {
    if (*TorsionMtx = (ATOMO **)Alloca((TorsionNum + 1) * sizeof(ATOMO *) * 4)) {
      CountTorsion(Angles, *TorsionMtx);
    } else TorsionNum = 0;
  }

  return TorsionNum;
}


