
/*************************************************
****            VEGA - SDF database           ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <malloc.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "formats.h"
#include "db_engine.h"

#ifdef WIN32
#  include <windows.h>
#endif

/**** Constants ****/

#define  VG_DBSDF_BUFSIZE       65536

/**** Local prototypes ****/

static VG_BOOL DbSdfCopy(FILE *, FILE *, VG_ULONG, VG_ULONG);
static void    DbSdfReOffset(VG_DBMOLLIST *, VG_ULONG, VG_LONG);


/**** Close the database ****/

void DbSdfClose(VG_DBINFO *Dbh)
{
  fclose((FILE *)Dbh -> Handle);
}


/**** Create a new database ****/

VG_BOOL DbSdfCreate(VG_DBINFO *Dbh)
{
  ChangeExt(Dbh -> FileName, "sdf", NULL);
  if ((Dbh -> Handle = (FILE *)fopen(Dbh -> FileName, "w+")) != NULL) {
    return TRUE;
  }

  return FALSE;
}


/**** Copy a file section ****/

static VG_BOOL DbSdfCopy(FILE *DH, FILE *SH, VG_ULONG SOffset, VG_ULONG SSize)
{
  char          *Buf;
  VG_ULONG      Size;

  VG_BOOL       Ret = TRUE;

  if (!SSize) {
    fseek(SH, 0, SEEK_END);
    SSize = ftell(SH) - SOffset;
  }
  fseek(SH, SOffset, SEEK_SET);
  if ((Buf = (char *)Alloca(VG_DBSDF_BUFSIZE)) != NULL) {
    for(;(Ret) && (SSize);) {
      if (SSize < VG_DBSDF_BUFSIZE) Size = SSize;
      else Size = VG_DBSDF_BUFSIZE;
      if ((fread(Buf, 1, Size, SH) != Size) ||
          (fwrite(Buf, 1, Size, DH) != Size))
        Ret = FALSE;
      SSize -= Size;
    } /* End of for */
    FREE(Buf);
  } else Ret = FALSE;

  return Ret;
}


/**** Get a molucule ****/

ATOMO *DbSdfGet(VG_DBINFO *Dbh, char *MolName)
{
  ATOMO                 *LastSeg;
  RECORD                Lin;
  VG_UBYTE              Conn;
  VG_DBMOLLIST          *Ptr;
  VG_ULONG              k, TmpTotAtm;

  ATOMO                 *TmpIniz = NULL;

  if ((Ptr = DbMolListGet(Dbh, MolName, 0)) != NULL) {
    fseek((FILE *)Dbh -> Handle, Ptr -> Offset, SEEK_SET);
    for(k = 0; k < 4; ++k) fgets(Lin.Line, VG_LINELEN, (FILE *)Dbh -> Handle);

    LastSeg         = LastAtm;
    Conn            = GLOBSW_CONNCALC;
    GLOBSW_CONNCALC = TRUE;
    TmpIniz         = MdlMolLoad((FILE *)Dbh -> Handle, &Lin, &TmpTotAtm);

    if (TmpIniz) {
      TotalAtm += TmpTotAtm;
      if (BegAtm) {
        LastSeg -> Ptr  = TmpIniz;
        TmpIniz -> Prev = LastSeg;
      } else BegAtm = TmpIniz;
      if (GLOBSW_CONNCALC) Connect(TmpIniz, TmpTotAtm, Prefs.CONNTOL, FALSE);
    }
    GLOBSW_CONNCALC = Conn;
  }

  return TmpIniz;
}


/**** Read the molecule list ****/

VG_BOOL DbSdfMolListRead(VG_DBINFO *Dbh)
{
  char          Buf[128];
  VG_DBMOLLIST  *Entry;
  VG_ULONG      Len;

  VG_BOOL       Ret    = TRUE;
  VG_ULONG      Line   = 0;
  VG_ULONG      Offset = 0;

  fseek((FILE *)Dbh -> Handle, 0, SEEK_SET);
  while((Ret) && (fgets(Buf, 127, (FILE *)Dbh -> Handle))) {
    if (!Line) {
      Len = strlen(Buf);
      Buf[Len - 1] = 0;
      if (Buf[Len - 2] == 13) {
        Dbh -> Flags |= VG_DB_FLAG_CR;
        Buf[Len - 2] = 0;
      }
      if (!StrExist(Buf)) {
        sprintf(Buf, "%06d", Dbh -> TotMol + 1);
      }
      if ((Entry = DbMolListAdd(Dbh, Buf)) != NULL) {
        Entry -> Offset = Offset;
      } else Ret = FALSE;
    }

    if (!strncmp(Buf, "$$$$", 4)) {
      Offset = ftell((FILE *)Dbh -> Handle);
      Line = 0;
    } else ++Line;
  } /* End of while */

  return Ret;
}


/**** Open the database ****/

VG_BOOL DbSdfOpen(VG_DBINFO *Dbh)
{
  if ((Dbh -> Handle = (FILE *)fopen(Dbh -> FileName, "rb+")) != NULL) {
    Dbh -> MolFormat = &FileInfo[FORM_MDL & 0xff];
    Dbh -> MolFlags  = VG_DB_MOLFLAG_CONN;
    return TRUE;
  }

  return FALSE;
}


/**** Remove a molecule ****/

VG_BOOL DbSdfRemove(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm)
{
  char          Buf[VG_MAX_PATH];
  FILE          *DH;
  VG_DBMOLLIST  *Mol;
  VG_LONG       Delta;

  VG_BOOL       Ret = TRUE;

  if (Dbh -> TotMol == 1) {
    if (Dbh -> MolList -> Beg == (VG_LISTITEM *)Itm) {
      fclose(Dbh -> Handle);
      DbSdfCreate(Dbh);
    } else Ret = FALSE;
  } else {
    strcpy(Buf, Dbh -> FileName);
    strcat(Buf, "~");

    if ((DH = fopen(Buf, "wb")) != NULL) {

      /**** Last molecule ****/

      if (!Itm -> Next) {
        Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, 0, Itm -> Offset);
        Delta = 0;

      /**** First molecule ****/

      } else if (!Itm -> Prev) {
        Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, Itm -> Next -> Offset, 0);
        Delta = - (VG_LONG)Itm -> Next -> Offset;

      /**** Inside molecule ****/

      } else {
        if ((Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, 0, Itm -> Offset))) {
          Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, Itm -> Next -> Offset, 0);
          Delta = Itm -> Offset - Itm -> Next -> Offset;
        }
      }
      fclose(DH);
      if (Ret) {
        DbSdfClose(Dbh);
#ifdef WIN32
        DeleteFile(Dbh -> FileName);
        MoveFile(Buf, Dbh -> FileName);
#else
        remove(Dbh -> FileName);
        rename(Buf, Dbh -> FileName);
#endif
        if (Delta) DbSdfReOffset((VG_DBMOLLIST *)Dbh -> MolList -> Beg, Itm -> Offset, Delta);
        DbSdfOpen(Dbh);
      } else {
#ifdef WIN32
        DeleteFile(Buf);
#else
        remove(Buf);
#endif
      }
    } else Ret = PrintDosErr();
  }

  return Ret;
}


/**** Rename a molecule ****/

VG_BOOL DbSdfRen(VG_DBINFO *Dbh, VG_DBMOLLIST *Itm, char *NewName)
{
  char          Buf[VG_MAX_PATH];
  FILE          *DH;
  VG_ULONG      NewLen, OldLen;

  VG_BOOL       Ret = TRUE;

  NewLen = strlen(NewName);
  OldLen = strlen(Itm -> MolName);
  if (OldLen != NewLen) {
    strcpy(Buf, Dbh -> FileName);
    strcat(Buf, "~");
    if ((DH = fopen(Buf, "wb")) != NULL) {
      if (Itm -> Offset) {
        Ret = DbSdfCopy(DH, Dbh -> Handle, 0, Itm -> Offset);
      }
      if ((Ret) && (fwrite(NewName, 1, NewLen, DH) == NewLen)) {
        if (Dbh -> Flags & VG_DB_FLAG_CR) {
          if (fputc(13, DH) != 13) Ret = FALSE;
          NewLen += 2;
          OldLen += 2;
        } else {
          ++NewLen;
          ++OldLen;
        }
        if ((Ret) && (fputc(10, DH) != 10))
          Ret = FALSE;
      } else Ret = FALSE;
      if (Ret)
        Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, Itm -> Offset + OldLen, 0);
      fclose(DH);

      if (Ret) {
        DbSdfClose(Dbh);
#ifdef WIN32
        DeleteFile(Dbh -> FileName);
        MoveFile(Buf, Dbh -> FileName);
#else
        remove(Dbh -> FileName);
        rename(Buf, Dbh -> FileName);
#endif
        DbSdfReOffset((VG_DBMOLLIST *)Dbh -> MolList -> Beg, Itm -> Offset, NewLen - OldLen);
        DbSdfOpen(Dbh);
      } else {
#ifdef WIN32
        DeleteFile(Buf);
#else
        remove(Buf);
#endif
      }
    } else Ret = PrintDosErr();
  } else {
    fseek((FILE *)Dbh -> Handle, Itm -> Offset, SEEK_SET);
    if (fwrite(NewName, 1, NewLen, (FILE *)Dbh -> Handle) != NewLen)
      Ret = FALSE;
  }

  return Ret;
}


/**** Recalculate the offsets ****/

static void DbSdfReOffset(VG_DBMOLLIST *Mol, VG_ULONG Offset, VG_LONG Delta)
{
  while(Mol) {
    if (Mol -> Offset > Offset)
      Mol -> Offset += Delta;
    Mol = Mol -> Next;
  } /* End of while */
}


/**** Put a molecule ****/

VG_BOOL DbSdfPut(VG_DBINFO *Dbh, char *MolName, VG_ULONG *Offset)
{
  char                  Buf[VG_MAX_PATH], *Ldem;
  FILE                  *DH;
  VG_DBMOLLIST          *Itm;

  VG_BOOL               Ret = TRUE;

  if (Dbh -> Flags & VG_DB_FLAG_CR) Ldem = "\015\012";
  else Ldem = "\012";

  if ((Itm = DbMolListGet(Dbh, MolName, 0)) == NULL) {

    /**** New molecule ****/

    DH = (FILE *)Dbh -> Handle;
    fseek(DH, 0, SEEK_END);
    *Offset = ftell(DH);
    Ret     = MdlMolSave(DH, BegAtm, TotalAtm, MolName, Ldem, TRUE);
  } else {

    /**** Update molecule ****/

    strcpy(Buf, Dbh -> FileName);
    strcat(Buf, "~");
    if ((DH = fopen(Buf, "wb")) != NULL) {
      if (Itm -> Offset) {
        Ret = DbSdfCopy(DH, Dbh -> Handle, 0, Itm -> Offset);
      }
      if ((Ret) && (Ret = MdlMolSave(DH, BegAtm, TotalAtm, Itm -> MolName, Ldem, TRUE))) {
        *Offset = ftell(DH);
        if (Itm -> Next) {
          Ret = DbSdfCopy(DH, (FILE *)Dbh -> Handle, Itm -> Next -> Offset, 0);
        }
      }
      fclose(DH);
      if (Ret) {
        DbSdfClose(Dbh);
#ifdef WIN32
        DeleteFile(Dbh -> FileName);
        MoveFile(Buf, Dbh -> FileName);
#else
        remove(Dbh -> FileName);
        rename(Buf, Dbh -> FileName);
#endif
        if (Itm -> Next)
          DbSdfReOffset((VG_DBMOLLIST *)Dbh -> MolList -> Beg, Itm -> Offset,
                        *Offset - Itm -> Next -> Offset);
        DbSdfOpen(Dbh);
      } else {
#ifdef WIN32
        DeleteFile(Buf);
#else
        remove(Buf);
#endif
      }
    } else Ret = PrintDosErr();
  }

  return Ret;
}


