
/*************************************************
****       VEGA - Interactive measures        ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <gl\gl.h>
#include <gl\glu.h>
#include <stdio.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "bond.h"
#include "compar.h"
#include "selmol.h"
#include "gl_global.h"
#include "gl_prefs.h"
#include "gl_measure.h"
#include "gl_menu.h"
#include "gl_objs.h"

/**** Constants ****/

#define  VG_SELBUFSIZE          512
#define  VG_SELSIZE             15.0

/**** Measure flags ****/

#define  VG_MEASFLAG_NONE       0
#define  VG_MEASFLAG_3DOBJ      1
#define  VG_MEASFLAG_3DTXT      2
#define  VG_MEASFLAG_CONTXT     4

#define  VG_MEASFLAG_ALL        (VG_MEASFLAG_3DOBJ | VG_MEASFLAG_3DTXT | VG_MEASFLAG_CONTXT)


/**** Find atom by number ****/

ATOMO *GL_FindAtmByNum(VG_ULONG Num)
{
  register ATOMO        *Atm;

  if (Num > TotalAtm) return NULL;
  for(Atm = BegAtm; (Atm) && (Atm -> Num != Num); Atm = Atm -> Ptr);

  return Atm;
}

/**** Interactive measure ****/

void GL_Measure(VG_GLPREFS *Prf, ATOMO *PickedAtm)
{
  char                ChainID;
  PLANE               P0, P1;
  static ATOMO        *AtmList[6];
  VG_UWORD            k;
  ATOMO               *Atm;

  if (Prf -> MeasureMode == VG_PM_PICKCENTER) {
    CatPrintf(stdout, MSG_GLMEAS_CENTER, PickedAtm -> x, PickedAtm -> y, PickedAtm -> y);
    Prf -> RotCenter.x = PickedAtm -> x;
    Prf -> RotCenter.y = PickedAtm -> y;
    Prf -> RotCenter.z = PickedAtm -> z;
    if (!ViewPrefs.MoveMol) CatPrintf(stdout, MSG_GLMEAS_CENTERWARN);
  } else {
    if (Prf -> MeasureMode > VG_PM_PICKATOM) {
      for(k = 0; k < Prf -> PickedAtoms; ++k) {
        if (AtmList[k] == PickedAtm) {
          CatPrintf(stdout, MSG_GLMEAS_ALREADYPICKED);
          return;
        }
      } /* End of for */
    }

    if (PickedAtm -> ChainID == ' ') ChainID = '*';
    else ChainID = PickedAtm -> ChainID;

    CatPrintf(stdout, MSG_GLMEAS_PICKATM, PickedAtm -> Num, PickedAtm -> Name.C,
              PickedAtm -> ResName.C, PickedAtm -> ResSeq.C, ChainID,
              MolListFindMol(PickedAtm));

    if (Prf -> MeasureMode > VG_PM_PICKATOM) {
      AtmList[Prf -> PickedAtoms++] = PickedAtm;

      switch(Prf -> MeasureMode) {
      case VG_PM_PICKDIST:       /* Distance */
        if (Prf -> PickedAtoms == 2) {
          GL_MeasureObj(VG_PM_PICKDIST, AtmList, VG_MEASFLAG_ALL, NULL, NULL);
          Prf -> PickedAtoms = 0;
        }
        break;

      case VG_PM_PICKANG:      /* Angle */
        if (Prf -> PickedAtoms == 3) {
          GL_MeasureObj(VG_PM_PICKANG, AtmList, VG_MEASFLAG_ALL, NULL, NULL);
          Prf -> PickedAtoms = 0;
        }
        break;

      case VG_PM_PICKTOR:     /* Torsion */
        if (Prf -> PickedAtoms == 4) {
          GL_MeasureObj(VG_PM_PICKTOR, AtmList, VG_MEASFLAG_ALL, NULL, NULL);
          Prf -> PickedAtoms = 0;
        }
        break;

      case VG_PM_PICKPLANG:     /* Angle between two planes */
        if (Prf -> PickedAtoms == 6) {
          GL_MeasureObj(VG_PM_PICKPLANG, AtmList, VG_MEASFLAG_ALL, NULL, NULL);
          Prf -> PickedAtoms = 0;
        }
        break;
      } /* End of switch */
    }
  }
}


/**** Measure command ****/

float GL_MeasureCom(VG_COMM *Com)
{
  ATOMO         *Atm, *AtmList[6];
  char          *MyArg[5], NMolStr[12];
  VG_PARAM      *Arg;
  VG_UBYTE      j, k;
  VG_ULONG      AtmNum, MeasureMode, NumArgs, NMol;

  float         Res    = FALSE;
  VG_BOOL       Ret    = TRUE;
  VG_UBYTE      NumPar = Com -> NumPar - 1;

  /**** Build the atom list ****/

  Arg = Com -> ParList;
  for(k = 0; k < NumPar; ++k) {
    AtmList[k] = NULL;
    NMol       = 1;
    NumArgs    = GL_MatchFormatPattern(&AtmNum, MyArg, Arg -> Data.C);
    if ((!NumArgs) && (AtmNum)) {
      AtmList[k] = GL_FindAtmByNum(AtmNum);
    } else {
      NMolStr[0] = '1';
      NMolStr[1] = 0;
      for(Atm = BegAtm; Atm; Atm = Atm -> Ptr) {
        if (GL_MatchAtm(Atm, MyArg, NumArgs, NMolStr)) {
          AtmList[k] = Atm;
          break;
        }
        if (Atm -> Flags & VG_ATMF_MOLEND)
          sprintf(NMolStr, "%d", ++NMol);
      } /* End of for */
    }
    Arg = Arg -> Ptr;
  } /* End of for */

  for(k = 0; (AtmList[k]) && (k < NumPar); ++k);

  if (k == NumPar) {
    for(k = 0; (Ret) && (k < NumPar); ++k) {
      for(j = k + 1; j < NumPar; ++j) {
        if (AtmList[k] == AtmList[j]) {
          Ret = CatErr(MSG_GLMEAS_ERR_ATMMULTI);
          break;
        }
      } /* End of for */
    } /* End of for */
  } else Ret = CatErr(MSG_GLMEAS_ERR_ATMNOTFND, (int)k + 1);

  if (Ret) {
    switch(Com -> Id) {
    case VG_PARCOM_ANGLE:
      MeasureMode = VG_PM_PICKANG;
      break;

    case VG_PARCOM_DISTANCE:
      MeasureMode = VG_PM_PICKDIST;
      break;

    case VG_PARCOM_TORSION:
      MeasureMode = VG_PM_PICKTOR;
      break;

    case VG_PARCOM_PLANG:
      MeasureMode = VG_PM_PICKPLANG;
      break;
    } /* End of switch */
    Res = GL_MeasureObj(MeasureMode, AtmList, Arg -> Data.L, NULL, NULL);
  }

  if ((Arg -> Data.L & VG_MEASFLAG_3DTXT) || (Arg -> Data.L & VG_MEASFLAG_3DOBJ))
    GL_DrawScene(&ViewPrefs, TRUE);

  return Res;
}


/**** Measure adding a 3D object ****/

float GL_MeasureObj(VG_ULONG MeasureMode, ATOMO **AtmList, VG_ULONG Flags,
                   void *UserData, char *Label)
{
  float                 T;
  VG_GLOBJ              *Obj;
  VG_LONG               GrpFlag;
  VG_ULONG              Msg;
  VG_ULONG              k;

  char                  *FormatStr = "%.2f";

  if (Flags & VG_MEASFLAG_SELECT) GrpFlag = VG_OBJGRP_SELECT;
  else GrpFlag = 0;
  
  switch(MeasureMode) {
  case VG_PM_PICKDIST:       /* Distance */
    Msg   = MSG_GLMEAS_DIST;
    T     = Distance(AtmList[0], AtmList[1]);

    if (Flags & VG_MEASFLAG_3DOBJ) {
      if (!BndCheck(AtmList[1], AtmList[0], NULL)) {
        Obj = GL_ObjLineA(AtmList[0], AtmList[1], VG_MEASCOL_DISTANCE,
                          VG_OBJFLAG_STIPPLE, VG_OBJGRP_DISTANCE|GrpFlag);
        Obj -> UserData = UserData;
      }
    }

    if (Flags & VG_MEASFLAG_3DTXT) {
      if (Label) FormatStr = Label;
      else FormatStr = "%.2f";
      Obj = GL_ObjDistA(AtmList[0], AtmList[1], VG_MEASCOL_TEXT,
                        VG_OBJFLAG_NONE, VG_OBJGRP_DISTANCE|GrpFlag, FormatStr);
      Obj -> UserData = UserData;
    }
    break;

  case VG_PM_PICKANG:      /* Angle */
    Msg   = MSG_GLMEAS_ANGLE;
    T     = NormAnglef(BondAnglef(AtmList[0], AtmList[1], AtmList[2]));

    if (Flags & VG_MEASFLAG_3DOBJ) {
      Obj = GL_ObjTriangleA(AtmList, VG_MEASCOL_ANGLE,
                            VG_OBJFLAG_BLEND|VG_OBJFLAG_MONITOR, VG_OBJGRP_ANGLE|GrpFlag);
      Obj -> UserData = UserData;
      if (!BndCheck(AtmList[1], AtmList[0], NULL)) {
        Obj = GL_ObjLineA(AtmList[1], AtmList[0], VG_MEASCOL_SUPLINE,
                          VG_OBJFLAG_STIPPLE, VG_OBJGRP_ANGLE|GrpFlag);
        Obj -> UserData = UserData;
      }
      if (!BndCheck(AtmList[1], AtmList[2], NULL)) {
        Obj = GL_ObjLineA(AtmList[1], AtmList[2], VG_MEASCOL_SUPLINE,
                          VG_OBJFLAG_STIPPLE, VG_OBJGRP_ANGLE|GrpFlag);
        Obj -> UserData = UserData;
      }
    }

    if (Flags & VG_MEASFLAG_3DTXT) {
      if (Label) FormatStr = Label;
      Obj = GL_ObjAngA(AtmList, VG_MEASCOL_TEXT,
                       VG_OBJFLAG_NONE, VG_OBJGRP_ANGLE|GrpFlag, FormatStr);
      Obj -> UserData = UserData;
    }
    break;

  case VG_PM_PICKTOR:     /* Torsion */
    Msg   = MSG_GLMEAS_TORSION;
    T     = NormAnglef((float)Torsion(AtmList[0], AtmList[1], AtmList[2], AtmList[3]));

    if (Flags & VG_MEASFLAG_3DOBJ) {
      Obj = GL_ObjDiscA(AtmList, 20, 0.0, 0.5, VG_MEASCOL_TORSION,
                        VG_OBJFLAG_BLEND, VG_OBJGRP_TORSION|GrpFlag);
      Obj -> UserData = UserData;
      for(k = 0; k < 3; ++k) {
        if (!BndCheck(AtmList[k], AtmList[k + 1], NULL)) {
          Obj = GL_ObjLineA(AtmList[k], AtmList[k + 1], VG_MEASCOL_SUPLINE,
                           VG_OBJFLAG_STIPPLE, VG_OBJGRP_TORSION|GrpFlag);
          Obj -> UserData = UserData;
        }
      } /* End of for */
    }

    if (Flags & VG_MEASFLAG_3DTXT) {
      if (Label) FormatStr = Label;
      Obj = GL_ObjTorA(AtmList, VG_MEASCOL_TEXT,
                       VG_OBJFLAG_NONE, VG_OBJGRP_TORSION|GrpFlag, FormatStr);
      Obj -> UserData = UserData;
    }
    break;

  case VG_PM_PICKPLANG:     /* Angle between two planes */
    Msg   = MSG_GLMEAS_PLANG;
    T     = GL_PlaneAngle(AtmList);

    if (Flags & VG_MEASFLAG_3DOBJ) {
      if (Flags & VG_MEASFLAG_3DTXT) {
        if (Label) FormatStr = Label;
      } else FormatStr = NULL;
      Obj = GL_ObjPlangA(AtmList, VG_MEASCOL_PLANGLINE, VG_OBJFLAG_NONE, VG_OBJGRP_PLANG|GrpFlag, FormatStr);
      Obj -> UserData = UserData;
    }
    break;
  } /* End of switch */

  if (Flags & VG_MEASFLAG_CONTXT)
    CatPrintf(stdout, Msg, T);

  return T;
}


/**** Find the picked atom ****/

ATOMO *GL_PickAtom(VG_GLPREFS *Prf, int x, int y)
{
  ATOMO                 *PickedAtm;
  GLuint                SelectBuf[VG_SELBUFSIZE];
  GLint                 ViewPort[4];
  register ATOMO        *Atm;

  glGetIntegerv(GL_VIEWPORT, ViewPort);
  glSelectBuffer(VG_SELBUFSIZE, SelectBuf);
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(0);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix() ;
    glLoadIdentity();
    gluPickMatrix((GLdouble)x, (GLdouble)(ViewPort[3] - y),
                  VG_SELSIZE, VG_SELSIZE, ViewPort);
    gluPerspective(45.0f, (GLfloat)ViewPort[2] / (GLfloat)ViewPort[3], 0.1f,
                   GlPrefs -> GlZClip);
    glPushMatrix();

      /**** Apply the world transformation ****/

/*
      glTranslatef(Prf -> ViewCenter.x + Prf -> RotCenter.x,
                   Prf -> ViewCenter.y + Prf -> RotCenter.y,
                   Prf -> ViewCenter.z + Prf -> RotCenter.z);
*/

    glTranslatef(Prf -> ViewCenter.x,
                 Prf -> ViewCenter.y,
                 Prf -> ViewCenter.z);

      glMultMatrixf((const float *)RotMat);
//      glTranslatef(-Prf -> RotCenter.x, -Prf -> RotCenter.y, -Prf -> RotCenter.z);

      glScalef(Prf -> Scale, Prf -> Scale, Prf -> Scale);

      for(Atm = BegAtm; Atm; Atm = Atm -> Ptr) {
        if (Atm -> Active) {
          glPushName((GLuint)Atm);
          glBegin(GL_POINTS);
            glVertex3f(Atm -> x, Atm -> y, Atm -> z);
          glEnd();
          glPopName();
        }
      } /* End of for */
    glPopMatrix();
  glPopMatrix();
  if (!glRenderMode(GL_RENDER))
    PickedAtm = NULL;
  else
    PickedAtm = (ATOMO *)SelectBuf[4];

  glMatrixMode(GL_MODELVIEW);

  if (PickedAtm) {
    if (PickedAtm -> Label == VG_ATMLBL_NONE)
      PickedAtm -> Label = VG_ATMLBL_NAME;
    else
      PickedAtm -> Label = VG_ATMLBL_NONE;
  }

  return PickedAtm;
}


/**** Measure the angle between two planes ****/

float GL_PlaneAngle(ATOMO **AtmList)
{
  PLANE         P0, P1;

  PlaneCof(&P0, AtmList[0], AtmList[1], AtmList[2]);
  PlaneCof(&P1, AtmList[3], AtmList[4], AtmList[5]);
  return NormAnglef((float)PlaneAngle(&P0, &P1) * RAD_TO_DEG);
}



