
/*************************************************
****          VEGA - OpenGL objects           ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <math.h>
#pragma hdrstop

#include "globdef.h"
#include "globvar.h"
#include "gl_global.h"
#include "gl_measure.h"
#include "gl_objs.h"
#include "gl_wrkspace.h"

#define  VG_MEAS_GROUPS         5

/**** Local variables ****/

VG_GLOBJ         *BegObj  = NULL;
VG_GLOBJ         *LastObj = NULL;

static const VG_UWORD   MonGrps[] = {VG_OBJGRP_ANGLE,
                                     VG_OBJGRP_BOND,
                                     VG_OBJGRP_DISTANCE,
                                     VG_OBJGRP_PLANG,
                                     VG_OBJGRP_TORSION
                                    };

static const VG_UBYTE   MonAtms[] = {3,         /* Angle                    */
                                     2,         /* Bond                     */
                                     2,         /* Distance                 */
                                     6,         /* Angle betweem two planes */
                                     4          /* Torsion                  */
                                    };


/**** Add an object to the video chain ****/

VG_GLOBJ *GL_ObjAdd(VG_ULONG Type, VG_ULONG Len, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = (VG_GLOBJ *)Alloca(sizeof(VG_GLOBJ))) != NULL) {
    if ((Obj -> Data = Alloca(Len)) != NULL) {
      Obj -> Flags = Flags;
      Obj -> Group = Group;
      Obj -> Type  = Type;
      if (BegObj) {
        Obj     -> Prev = LastObj;
        LastObj -> Next = Obj;
        LastObj         = Obj;
      } else BegObj = LastObj = Obj;
    } else {
      FREE(Obj);
      Obj = NULL;
    }
  }

  return Obj;
}


/**** Add an angle monitor between three atoms ****/

VG_GLOBJ *GL_ObjAngA(ATOMO **Atm, VG_LONG Color, VG_LONG Flags,
                      VG_UWORD Group, char *LabelStr)
{
  VG_GLOBJ      *Obj;
  VG_ULONG      k;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_ANGA, sizeof(VG_GLANGA) + strlen(LabelStr), Flags, Group)) != NULL) {
    for(k = 0; k < 3; ++k)
      ((VG_GLANGA *)Obj -> Data) -> Atm[k] = Atm[k];
    ((VG_GLANGA *)Obj -> Data) -> Color    = Color;
    strcpy(((VG_GLANGA *)Obj -> Data) -> LabelStr, LabelStr);
  }

  return Obj;
}


/**** Change text ****/

void GL_ObjChgText(VG_GLOBJ *Obj, char *NewLabel)
{
  char          *Str;
  void          *Data;

  struct _VgObjTab {
    VG_UWORD    Type;
    VG_UWORD    Size;
  };

  const struct _VgObjTab *TabPtr;
  const struct _VgObjTab VgObjTab[] = {{VG_GLOBJ_ANGA  , sizeof(VG_GLANGA  )},
                                       {VG_GLOBJ_DISTA , sizeof(VG_GLDISTA )},
                                       {VG_GLOBJ_PLANGA, sizeof(VG_GLPLANGA)},
                                       {VG_GLOBJ_TEXT3 , sizeof(VG_GLTEXT  )},
                                       {VG_GLOBJ_TORA  , sizeof(VG_GLTORA  )},
                                       {VG_GLOBJ_NONE  , 0                 }};

  for(TabPtr = VgObjTab; TabPtr -> Type != VG_GLOBJ_NONE; ++TabPtr) {
    if (TabPtr -> Type == Obj -> Type) {
      if ((Data = (void *)Alloca(TabPtr -> Size + strlen(NewLabel))) != NULL) {
        memcpy(Data, Obj -> Data, TabPtr -> Size - 4);
        strcpy(((char *)Data) + TabPtr -> Size - 4, NewLabel);
        FREE(Obj -> Data);
        Obj -> Data = (void *)Data;
      }
      break;
    }
  } /* End of for */
}


/**** Change the vertex ****/

void GL_ObjChgVertex(VG_GLVERTEX *Vert, float *V, VG_LONG Color)
{
  Vert -> Vect[0] = V[0];
  Vert -> Vect[1] = V[1];
  Vert -> Vect[2] = V[2];
  Vert -> Color   = Color;
}


/**** Change the vertex ****/

void GL_ObjChgVertexA(VG_GLVERTEXA *Vert, ATOMO *Atm, VG_LONG Color)
{
  Vert -> Atm   = Atm;
  Vert -> Color = Color;
}


/**** Add a centered cube ****/

VG_GLOBJ *GL_ObjCCube(float *V0, float S, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;
  float         V1[8][3];

  S /= 2.0;

  V1[0][0] = V1[4][0] = V0[0] - S;    V1[0][1] = V1[4][1] = V0[1] + S;
  V1[1][0] = V1[5][0] = V0[0] + S;    V1[1][1] = V1[5][1] = V0[1] + S;
  V1[2][0] = V1[6][0] = V0[0] + S;    V1[2][1] = V1[6][1] = V0[1] - S;
  V1[3][0] = V1[7][0] = V0[0] - S;    V1[3][1] = V1[7][1] = V0[1] - S;

  V1[0][2] = V1[1][2] = V1[2][2] = V1[3][2] = V0[2] + S;
  V1[4][2] = V1[5][2] = V1[6][2] = V1[7][2] = V0[2] - S;

  return GL_ObjCube(V1, Color, Flags, Group);
}


/**** Change rendering color ****/

void GL_ObjColor(VG_LONG Color)
{
#ifdef LITTLE_ENDIAN
  glColor4ub(((char *)&Color)[3], ((char *)&Color)[2],
             ((char *)&Color)[1], ((char *)&Color)[0]);
#else
  glColor4ubv((char *)&Color);
#endif
}


/**** Add a cube ****/

VG_GLOBJ *GL_ObjCube(float V0[8][3], VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;
  VG_ULONG      k;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_CUBE, sizeof(VG_GLCUBE), Flags, Group)) != NULL) {
    for(k = 0; k < 8; ++k) {
      GL_ObjChgVertex(&((VG_GLLINE *)Obj -> Data) -> Vertex[k], V0[k], Color);
    } /* End of for */
  }

  return Obj;
}


/**** Add a cross ****/

VG_GLOBJ *GL_ObjCross(float *V0, float Size, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_CROSS, sizeof(VG_GLCROSS), Flags, Group)) != NULL) {
    ((VG_GLCROSS *)Obj -> Data) -> Size = Size / 2.0;
    GL_ObjChgVertex(&((VG_GLCROSS *)Obj -> Data) -> Vertex, V0, Color);
  }

  return Obj;
}


/**** Add cylinder ****/

VG_GLOBJ *GL_ObjCylinder(float *V0, float *V1, VG_LONG Res, float Rad,
                         VG_LONG Color, VG_BOOL Disc, VG_LONG Flags,
                         VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_CYLINDER, sizeof(VG_GLCYLINDER), Flags, Group)) != NULL) {
    ((VG_GLCYLINDER *)Obj -> Data) -> Res  = Res;
    ((VG_GLCYLINDER *)Obj -> Data) -> Rad  = Rad;
    ((VG_GLCYLINDER *)Obj -> Data) -> Disc = Disc;
    GL_ObjChgVertex(&((VG_GLCYLINDER *)Obj -> Data) -> Vertex[0], V0, Color);
    GL_ObjChgVertex(&((VG_GLCYLINDER *)Obj -> Data) -> Vertex[1], V1, Color);
  }

  return Obj;
}


/**** Add a cylinder around a bond ****/

VG_GLOBJ *GL_ObjBondA(ATOMO *Atm0, ATOMO *Atm1, VG_LONG Res, float Rad,
                      VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_BONDA, sizeof(VG_GLBONDA), Flags, Group)) != NULL) {
    ((VG_GLBONDA *)Obj -> Data) -> Atm[0] = Atm0;
    ((VG_GLBONDA *)Obj -> Data) -> Atm[1] = Atm1;
    ((VG_GLBONDA *)Obj -> Data) -> Color  = Color;
    ((VG_GLBONDA *)Obj -> Data) -> Rad    = Rad;
    ((VG_GLBONDA *)Obj -> Data) -> Res    = Res;
  }

  return Obj;
}



/**** Add disc ****/

VG_GLOBJ *GL_ObjDisc(float *V0, float *V1, VG_LONG RodRes, float RodRad,
                     float RodTopRad, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_DISC, sizeof(VG_GLDISC), Flags, Group)) != NULL) {
    ((VG_GLDISC *)Obj -> Data) -> RodRes    = RodRes;
    ((VG_GLDISC *)Obj -> Data) -> RodRad    = RodRad;
    ((VG_GLDISC *)Obj -> Data) -> RodTopRad = RodTopRad;
    GL_ObjChgVertex(&((VG_GLDISC *)Obj -> Data) -> Vertex[0], V0, Color);
    GL_ObjChgVertex(&((VG_GLDISC *)Obj -> Data) -> Vertex[1], V1, Color);
  }

  return Obj;
}


/**** Add disc (atom coordinates) ****/

VG_GLOBJ *GL_ObjDiscA(ATOMO **Atm, VG_LONG RodRes, float RodRad, float RodTopRad,
                      VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_DISCA, sizeof(VG_GLDISCA), Flags, Group)) != NULL) {
    ((VG_GLDISCA *)Obj -> Data) -> RodRes    = RodRes;
    ((VG_GLDISCA *)Obj -> Data) -> RodRad    = RodRad;
    ((VG_GLDISCA *)Obj -> Data) -> RodTopRad = RodTopRad;
    GL_ObjChgVertexA(&((VG_GLDISCA *)Obj -> Data) -> Vertex[0], Atm[0], Color);
    GL_ObjChgVertexA(&((VG_GLDISCA *)Obj -> Data) -> Vertex[1], Atm[1], Color);
    GL_ObjChgVertexA(&((VG_GLDISCA *)Obj -> Data) -> Vertex[2], Atm[2], Color);
    GL_ObjChgVertexA(&((VG_GLDISCA *)Obj -> Data) -> Vertex[3], Atm[3], Color);
  }

  return Obj;
}


/**** Add a distance monitor between two atoms ****/

VG_GLOBJ *GL_ObjDistA(ATOMO *Atm0, ATOMO *Atm1, VG_LONG Color, VG_LONG Flags,
                      VG_UWORD Group, char *LabelStr)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_DISTA, sizeof(VG_GLDISTA) + strlen(LabelStr), Flags, Group)) != NULL) {
    ((VG_GLDISTA *)Obj -> Data) -> Atm[0] = Atm0;
    ((VG_GLDISTA *)Obj -> Data) -> Atm[1] = Atm1;
    ((VG_GLDISTA *)Obj -> Data) -> Color  = Color;
    strcpy(((VG_GLDISTA *)Obj -> Data) -> LabelStr, LabelStr);
  }

  return Obj;
}


/**** Find an object ****/

VG_GLOBJ *GL_ObjFind(VG_GLOBJ *Start, VG_UWORD Type, VG_UWORD Group, void *UserData)
{
  if (!Start) Start = BegObj;

  while((Start) &&
        ((Start -> Type != Type) ||
         (Start -> Group != Group) ||
         (Start -> UserData != UserData)))
    Start = Start -> Next;

  return Start;
}


/**** Hide all objects ****/

void GL_ObjHideAll(VG_BOOL Flag)
{
  register VG_GLOBJ      *Obj;

  for(Obj = BegObj; Obj; Obj = Obj -> Next) {
    if (Flag) Obj -> Flags |= VG_OBJFLAG_HIDE;
    else Obj -> Flags = Obj -> Flags &~ VG_OBJFLAG_HIDE;
  } /* End of for */
}


/**** Add a line ****/

VG_GLOBJ *GL_ObjLine(float *V0, float *V1, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_LINE, sizeof(VG_GLLINE), Flags, Group)) != NULL) {
    GL_ObjChgVertex(&((VG_GLLINE *)Obj -> Data) -> Vertex[0], V0, Color);
    GL_ObjChgVertex(&((VG_GLLINE *)Obj -> Data) -> Vertex[1], V1, Color);
  }

  return Obj;
}


/**** Add a line between two atoms ****/

VG_GLOBJ *GL_ObjLineA(ATOMO *Atm0, ATOMO *Atm1, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_LINEA, sizeof(VG_GLLINEA), Flags, Group)) != NULL) {
    GL_ObjChgVertexA(&((VG_GLLINEA *)Obj -> Data) -> Vertex[0], Atm0, Color);
    GL_ObjChgVertexA(&((VG_GLLINEA *)Obj -> Data) -> Vertex[1], Atm1, Color);
  }

  return Obj;
}

/**** Add a plane angle monitor between six atoms ****/

VG_GLOBJ *GL_ObjPlangA(ATOMO **Atm, VG_LONG Color, VG_LONG Flags,
                      VG_UWORD Group, char *LabelStr)
{
  VG_GLOBJ      *Obj;
  VG_ULONG      k = sizeof(VG_GLPLANGA);

  if (LabelStr) k += strlen(LabelStr);

  if ((Obj = GL_ObjAdd(VG_GLOBJ_PLANGA, k, Flags, Group)) != NULL) {
    for(k = 0; k < 6; ++k)
      ((VG_GLPLANGA *)Obj -> Data) -> Atm[k] = Atm[k];
    ((VG_GLPLANGA *)Obj -> Data) -> Color    = Color;
    strcpy(((VG_GLPLANGA *)Obj -> Data) -> LabelStr, LabelStr);
  }

  return Obj;
}

/**** Remove one or more groups of objects ****/

VG_BOOL GL_ObjRemove(VG_UWORD *Group, VG_UWORD Num)
{
  VG_GLOBJ      *Next, *Obj;
  VG_UWORD      k;

  VG_BOOL       Ret = FALSE;

  for(Obj = BegObj; Obj;) {
    Next = Obj -> Next;
    for(k = 0; k < Num; ++k) {
      if (Group[k] == Obj -> Group) {
        GL_ObjRemoveSingle(&BegObj, &LastObj, Obj);
        Ret = TRUE;
        break;
      }
    } /* End of for */
    Obj = Next;
  } /* End of for */

  return Ret;
}


/**** Remove all objects ****/

void GL_ObjRemoveAll(void)
{
  register VG_GLOBJ      *Next, *Obj;

  for(Obj = BegObj; Obj;) {
    FREE(Obj -> Data);
    Next = Obj -> Next;
    FREE(Obj);
    Obj = Next;
  } /* End of for */
  BegObj = LastObj = NULL;
}


/**** Remove monitors conatining an antom ****/

void GL_ObjRemoveAtm(ATOMO *Atm)
{
  VG_GLOBJ      *Next, *Obj;
  VG_UBYTE      j;
  VG_ULONG      k;

  for(Obj = BegObj; Obj;) {
    Next = Obj -> Next;
    for(k = 0; k < VG_MEAS_GROUPS; ++k) {
      if (MonGrps[k] == (Obj -> Group &~ VG_OBJGRP_SELECT)) {
        if ((Obj -> Type == VG_GLOBJ_DISCA) ||
            (Obj -> Type == VG_GLOBJ_LINEA) ||
            (Obj -> Type == VG_GLOBJ_TRIANGLEA)) {
          for(j = 0; j < MonAtms[k]; ++j) {
            if (((VG_GLLINEA *)(Obj -> Data)) -> Vertex[j].Atm  == Atm) {
              GL_ObjRemoveSingle(&BegObj, &LastObj, Obj);
              break;
            }
          } /* End of for */
        } else {
          for(j = 0; j < MonAtms[k]; ++j) {
            if (((VG_GLPLANGA *)(Obj -> Data)) -> Atm[j] == Atm) {
              GL_ObjRemoveSingle(&BegObj, &LastObj, Obj);
              break;
            }
          } /* End of for */
        }
        break;
      }
    } /* End of for */
    Obj = Next;
  } /* End of for */
}


/**** Remove a single groups of objects ****/

VG_BOOL GL_ObjRemoveGrp(VG_UWORD Group)
{
  VG_GLOBJ      *Next, *Obj;

  VG_BOOL       Ret = FALSE;

  for(Obj = BegObj; Obj;) {
    Next = Obj -> Next;
    if (Group & Obj -> Group) {
      GL_ObjRemoveSingle(&BegObj, &LastObj, Obj);
      Ret = TRUE;
    }
    Obj = Next;
  } /* End of for */

  return Ret;
}


/**** Remove monitors ****/

void GL_ObjRemoveMon(void)
{
  GL_ObjRemove((VG_UWORD *)MonGrps, VG_MEAS_GROUPS);
}


/**** Remove a single object ****/

void GL_ObjRemoveSingle(VG_GLOBJ **Beg, VG_GLOBJ **Last, VG_GLOBJ *Obj)
{
  if (Obj) {
    FREE(Obj -> Data);
    if (Obj != *Beg) {
      if (Obj -> Next) {
        Obj -> Prev -> Next = Obj -> Next;
        Obj -> Next -> Prev = Obj -> Prev;
      } else {
        *Last           = Obj -> Prev;
        (*Last) -> Next = NULL;
      }
    } else *Beg = Obj -> Next;
    FREE(Obj);
  }
}


/**** Remove a single object safe mode ****/

void GL_ObjRemoveSingleSafe(VG_GLOBJ **Beg, VG_GLOBJ **Last, VG_GLOBJ **RmObj)
{
  VG_GLOBJ      *Obj;

  for(Obj = *Beg; Obj; Obj = Obj -> Next) {
    if (Obj == *RmObj) {
      GL_ObjRemoveSingle(Beg, Last, *RmObj);
      *RmObj = NULL;
      break;
    }
  } /* End of for */
}


/**** Remove one or more objects by group and user data ****/

void GL_ObjRemoveUser(VG_UWORD Group, void *UserData, VG_BOOL All)
{
  VG_GLOBJ      *Next, *Obj;

  for(Obj = BegObj; Obj;) {
    Next = Obj -> Next;
    if ((Group == Obj -> Group) && (UserData == Obj -> UserData)) {
      GL_ObjRemoveSingle(&BegObj, &LastObj, Obj);
      if (!All) break;
    }
    Obj = Next;
  } /* End of for */
}


/**** Remove a single object from all workspaces ****/

void GL_ObjRemoveSingleWks(VG_GLOBJ **RmObj)
{
  VG_WRKSPACE           *Wks;

  /**** Try to remove it from the current workspace ****/

  GL_ObjRemoveSingleSafe(&BegObj, &LastObj, RmObj);

  /**** TRy to find it in the other workspaces ****/

  if (*RmObj) {
    for(Wks = (VG_WRKSPACE *)WksList -> Beg; (Wks) && (*RmObj); Wks = Wks -> Next)
      if (Wks != WksCur) {
        GL_ObjRemoveSingleSafe(&Wks -> BegObj, &Wks -> LastObj, RmObj);
      }
  }
}


/**** Render the OpenGL objects ****/

void GL_ObjRender(void)
{
  ATOMO         **Atm;
  float         T, V[3][3], V0[3], V1[3];
  VG_QCHAR      *Color;
  VG_GLOBJ      *Obj;
  VG_GLVERTEX   *Vertex;
  VG_GLVERTEXA  *VertexA;

  for(Obj = BegObj; Obj; Obj = Obj -> Next) {
    if (Obj -> Flags & VG_OBJFLAG_HIDE)    continue;
    if (Obj -> Flags & VG_OBJFLAG_BLEND)   glEnable(GL_BLEND);
    if (Obj -> Flags & VG_OBJFLAG_STIPPLE) glEnable(GL_LINE_STIPPLE);

    switch(Obj -> Type) {
    case VG_GLOBJ_ANGA:
      Atm = ((VG_GLANGA *)Obj -> Data) -> Atm;
      if ((Atm[0] -> Active) && (Atm[1] -> Active) && (Atm[2] -> Active)) {
        V0[0] = (Atm[0] -> x + Atm[2] -> x) / 2.0f;
        V0[1] = (Atm[0] -> y + Atm[2] -> y) / 2.0f;
        V0[2] = (Atm[0] -> z + Atm[2] -> z) / 2.0f;
        GL_ObjColor(((VG_GLANGA *)Obj -> Data) -> Color);
        glRasterPos3fv(V0);
        GL_Print(((VG_GLANGA *)Obj -> Data) -> LabelStr,
                 NormAnglef(BondAnglef(Atm[0], Atm[1], Atm[2])));
      }
      break;

    case VG_GLOBJ_BONDA:
      Atm = ((VG_GLBONDA *)Obj -> Data) -> Atm;
      if ((Atm[0] -> Active) && (Atm[1] -> Active)) {
        T     = (Atm[0] -> x - Atm[1] -> x) / 4.0f;
        V0[0] = Atm[0] -> x - T;
        V1[0] = Atm[1] -> x + T;

        T     = (Atm[0] -> y - Atm[1] -> y) / 4.0f;
        V0[1] = Atm[0] -> y - T;
        V1[1] = Atm[1] -> y + T;

        T     = (Atm[0] -> z - Atm[1] -> z) / 4.0f;
        V0[2] = Atm[0] -> z - T;
        V1[2] = Atm[1] -> z + T;
        GL_ObjColor(((VG_GLBONDA *)Obj -> Data) -> Color);
        GL_Cylinder(V0, V1,
                    ((VG_GLBONDA *)Obj -> Data) -> Res,
                    ((VG_GLBONDA *)Obj -> Data) -> Rad,
                    ((VG_GLBONDA *)Obj -> Data) -> Rad,
                    VG_GLCYLTYPE_SOLID);
      }
    break;

    case VG_GLOBJ_CROSS:
      GL_ObjRenderCross(&((VG_GLCROSS *)Obj -> Data) -> Vertex,
                        ((VG_GLCROSS *)Obj -> Data) -> Size);
      break;

    case VG_GLOBJ_CUBE:
      Vertex = ((VG_GLCUBE *)Obj -> Data) -> Vertex;
      GL_ObjColor(Vertex -> Color);
      GL_ObjRenderCube(Vertex);
      break;

    case VG_GLOBJ_CYLINDER:
      Vertex = ((VG_GLDISC *)Obj -> Data) -> Vertex;
      GL_ObjColor(Vertex -> Color);
      GL_Cylinder((float *)Vertex,
                  (float *)&Vertex[1],
                  ((VG_GLCYLINDER *)Obj -> Data) -> Res,
                  ((VG_GLCYLINDER *)Obj -> Data) -> Rad,
                  ((VG_GLCYLINDER *)Obj -> Data) -> Rad,
                  (((VG_GLCYLINDER *)Obj -> Data) -> Disc * VG_GLCYLTYPE_DISC) |
                  VG_GLCYLTYPE_SOLID);
      break;

    case VG_GLOBJ_DISC:
      Vertex = ((VG_GLDISC *)Obj -> Data) -> Vertex;
      GL_ObjColor(Vertex -> Color);
      GL_Disc((float *)Vertex,
              (float *)&Vertex[1],
              ((VG_GLDISC *)Obj -> Data) -> RodRes,
              ((VG_GLDISC *)Obj -> Data) -> RodRad,
              ((VG_GLDISC *)Obj -> Data) -> RodTopRad);
      break;

    case VG_GLOBJ_DISCA:
      VertexA = ((VG_GLDISCA *)Obj -> Data) -> Vertex;
      if ((VertexA[1].Atm -> Active) && (VertexA[2].Atm -> Active)) {
        GL_ObjColor(VertexA -> Color);
        GL_Disc((float *)&VertexA[1].Atm -> x,
                (float *)&VertexA[2].Atm -> x,
                ((VG_GLDISCA *)Obj -> Data) -> RodRes,
                ((VG_GLDISCA *)Obj -> Data) -> RodRad,
                ((VG_GLDISCA *)Obj -> Data) -> RodTopRad);
      }
      break;

    case VG_GLOBJ_DISTA:
      Atm = ((VG_GLDISTA *)Obj -> Data) -> Atm;
      if ((Atm[0] -> Active) && (Atm[1] -> Active)) {
        V0[0] = (Atm[0] -> x + Atm[1] -> x) / 2.0f;
        V0[1] = (Atm[0] -> y + Atm[1] -> y) / 2.0f;
        V0[2] = (Atm[0] -> z + Atm[1] -> z) / 2.0f;
        GL_ObjColor(((VG_GLDISTA *)Obj -> Data) -> Color);
        glRasterPos3fv(V0);
        V0[0] = (Atm[0] -> x - Atm[1] -> x);
        V0[1] = (Atm[0] -> y - Atm[1] -> y);
        V0[2] = (Atm[0] -> z - Atm[1] -> z);
        GL_Print(((VG_GLDISTA *)Obj -> Data) -> LabelStr,
                 SQR(Quad(V0[0]) + Quad(V0[1]) + Quad(V0[2])));
      }
      break;

    case VG_GLOBJ_LINE:
      glBegin(GL_LINES);
        GL_ObjVertex(((VG_GLLINE *)Obj -> Data) -> Vertex, 2);
      glEnd();
      break;

    case VG_GLOBJ_LINEA:
      if ((((VG_GLLINEA *)Obj -> Data) -> Vertex[0].Atm -> Active) &&
          (((VG_GLLINEA *)Obj -> Data) -> Vertex[01].Atm -> Active)) {
        glBegin(GL_LINES);
          GL_ObjVertexA(((VG_GLLINEA *)Obj -> Data) -> Vertex, 2);
        glEnd();
      }
      break;

    case VG_GLOBJ_PLANGA:
      Atm = ((VG_GLPLANGA *)Obj -> Data) -> Atm;
      if ((Atm[0] -> Active) && (Atm[1] -> Active) && (Atm[1] -> Active) &&
          (Atm[3] -> Active) && (Atm[4] -> Active) && (Atm[5] -> Active)) {
        V[0][0] = (Atm[0] -> x + Atm[1] -> x + Atm[2] -> x) / 3.0f;
        V[0][1] = (Atm[0] -> y + Atm[1] -> y + Atm[2] -> y) / 3.0f;
        V[0][2] = (Atm[0] -> z + Atm[1] -> z + Atm[2] -> z) / 3.0f;
        V[1][0] = (Atm[3] -> x + Atm[4] -> x + Atm[5] -> x) / 3.0f;
        V[1][1] = (Atm[3] -> y + Atm[4] -> y + Atm[5] -> y) / 3.0f;
        V[1][2] = (Atm[3] -> z + Atm[4] -> z + Atm[5] -> z) / 3.0f;
        GL_ObjColor(VG_MEASCOL_PLANGLINE);
        glEnable(GL_LINE_STIPPLE);
        glBegin(GL_LINES);
          glVertex3fv(V[0]);
          glVertex3fv(V[1]);
        glEnd();
        glDisable(GL_LINE_STIPPLE);
        GL_ObjColor(VG_MEASCOL_PLANGCUBE);
        GL_ObjRenderCCube(V[0], 0.25);
        GL_ObjRenderCCube(V[1], 0.25);
        if (((VG_GLPLANGA *)Obj -> Data) -> LabelStr) {
          V0[0] = (V[0][0] + V[1][0]) / 2.0f;
          V0[1] = (V[0][1] + V[1][1]) / 2.0f;
          V0[2] = (V[0][2] + V[1][2]) / 2.0f;
          GL_ObjColor(VG_MEASCOL_TEXT);
          glRasterPos3fv(V0);
          GL_Print(((VG_GLPLANGA *)Obj -> Data) -> LabelStr, GL_PlaneAngle(Atm));
        }
      }
      break;

    case VG_GLOBJ_TEXT3:
      Vertex =  &((VG_GLTEXT *)Obj -> Data) -> Vertex;
      GL_ObjColor(Vertex -> Color);
      glRasterPos3fv(Vertex -> Vect);
      GL_Print(((VG_GLTEXT *)Obj -> Data) -> Str);
      break;

    case VG_GLOBJ_TORA:
      Atm   = ((VG_GLTORA *)Obj -> Data) -> Atm;
      if ((Atm[0] -> Active) && (Atm[1] -> Active) &&
          (Atm[2] -> Active) && (Atm[3] -> Active)) {
        V0[0] = (Atm[1] -> x + Atm[2] -> x) / 2.0f;
        V0[1] = (Atm[1] -> y + Atm[2] -> y) / 2.0f + 0.55f;
        V0[2] = (Atm[1] -> z + Atm[2] -> z) / 2.0f;
        GL_ObjColor(((VG_GLTORA *)Obj -> Data) -> Color);
        glRasterPos3fv(V0);
        GL_Print(((VG_GLTORA *)Obj -> Data) -> LabelStr,
                 NormAngle(Torsion(Atm[0], Atm[1], Atm[2], Atm[3])));
      }
      break;

    case VG_GLOBJ_TRIANGLE:
      glBegin(GL_TRIANGLES);
        GL_ObjVertex(((VG_GLTRIANGLE *)Obj -> Data) -> Vertex, 3);
      glEnd();
      break;

    case VG_GLOBJ_TRIANGLEA:
      VertexA = ((VG_GLTRIANGLEA *)Obj -> Data) -> Vertex;
      if ((VertexA[0].Atm -> Active) && (VertexA[1].Atm -> Active) &&
          (VertexA[2].Atm -> Active)) {
        glBegin(GL_TRIANGLES);
          if (Obj -> Flags && VG_OBJFLAG_MONITOR) {
            GL_ObjColor(VertexA[0].Color);
            glVertex3f((VertexA[0].Atm -> x + VertexA[1].Atm -> x) / 2.0f,
                       (VertexA[0].Atm -> y + VertexA[1].Atm -> y) / 2.0f,
                       (VertexA[0].Atm -> z + VertexA[1].Atm -> z) / 2.0f);
            GL_ObjColor(VertexA[1].Color);
            glVertex3fv(&VertexA[1].Atm -> x);
            GL_ObjColor(VertexA[2].Color);
            glVertex3f((VertexA[1].Atm -> x + VertexA[2].Atm -> x) / 2.0f,
                       (VertexA[1].Atm -> y + VertexA[2].Atm -> y) / 2.0f,
                       (VertexA[1].Atm -> z + VertexA[2].Atm -> z) / 2.0f);
          } else {
            GL_ObjVertexA(((VG_GLTRIANGLEA *)Obj -> Data) -> Vertex, 2);
          }
        glEnd();
      }
      break;
    } /* End of switch */

    if (Obj -> Flags & VG_OBJFLAG_BLEND)   glDisable(GL_BLEND);
    if (Obj -> Flags & VG_OBJFLAG_STIPPLE) glDisable(GL_LINE_STIPPLE);
  } /* End of for */
}


/**** Render a cross ****/

void GL_ObjRenderCross(VG_GLVERTEX *V, float S)
{
  GL_ObjColor(V -> Color);

  glBegin(GL_LINES);
    glVertex3f(V -> Vect[0] - S, V -> Vect[1]    , V -> Vect[2]    );
    glVertex3f(V -> Vect[0] + S, V -> Vect[1]    , V -> Vect[2]    );
    glVertex3f(V -> Vect[0]    , V -> Vect[1] - S, V -> Vect[2]    );
    glVertex3f(V -> Vect[0]    , V -> Vect[1] + S, V -> Vect[2]    );
    glVertex3f(V -> Vect[0]    , V -> Vect[1]    , V -> Vect[2] - S);
    glVertex3f(V -> Vect[0]    , V -> Vect[1]    , V -> Vect[2] + S);
  glEnd();
}


/**** Render a cube ****/

void GL_ObjRenderCube(VG_GLVERTEX *V)
{
  glBegin(GL_QUADS);

    /**** Front Face ****/

    glVertex3f(V[0].Vect[0], V[0].Vect[1], V[0].Vect[2]);
    glVertex3f(V[1].Vect[0], V[1].Vect[1], V[1].Vect[2]);
    glVertex3f(V[2].Vect[0], V[2].Vect[1], V[2].Vect[2]);
    glVertex3f(V[3].Vect[0], V[3].Vect[1], V[3].Vect[2]);

    /**** Back face ****/

    glVertex3f(V[4].Vect[0], V[4].Vect[1], V[4].Vect[2]);
    glVertex3f(V[5].Vect[0], V[5].Vect[1], V[5].Vect[2]);
    glVertex3f(V[6].Vect[0], V[6].Vect[1], V[6].Vect[2]);
    glVertex3f(V[7].Vect[0], V[7].Vect[1], V[7].Vect[2]);

    /**** Top face ****/

    glVertex3f(V[0].Vect[0], V[0].Vect[1], V[0].Vect[2]);
    glVertex3f(V[1].Vect[0], V[1].Vect[1], V[1].Vect[2]);
    glVertex3f(V[5].Vect[0], V[5].Vect[1], V[5].Vect[2]);
    glVertex3f(V[4].Vect[0], V[4].Vect[1], V[4].Vect[2]);

    /**** Bottom face ****/

    glVertex3f(V[2].Vect[0], V[2].Vect[1], V[2].Vect[2]);
    glVertex3f(V[3].Vect[0], V[3].Vect[1], V[3].Vect[2]);
    glVertex3f(V[7].Vect[0], V[7].Vect[1], V[7].Vect[2]);
    glVertex3f(V[6].Vect[0], V[6].Vect[1], V[6].Vect[2]);

    /**** Right face ****/

    glVertex3f(V[1].Vect[0], V[1].Vect[1], V[1].Vect[2]);
    glVertex3f(V[2].Vect[0], V[2].Vect[1], V[2].Vect[2]);
    glVertex3f(V[6].Vect[0], V[6].Vect[1], V[6].Vect[2]);
    glVertex3f(V[5].Vect[0], V[5].Vect[1], V[5].Vect[2]);

    /**** Left face ****/

    glVertex3f(V[0].Vect[0], V[0].Vect[1], V[0].Vect[2]);
    glVertex3f(V[3].Vect[0], V[3].Vect[1], V[3].Vect[2]);
    glVertex3f(V[7].Vect[0], V[7].Vect[1], V[7].Vect[2]);
    glVertex3f(V[4].Vect[0], V[4].Vect[1], V[4].Vect[2]);

  glEnd();
}


/**** Add a centered cube ****/

void GL_ObjRenderCCube(float *V0, float S)
{
  VG_GLVERTEX    V1[8];

  S /= 2.0;

  V1[0].Vect[0] = V1[4].Vect[0] = V0[0] - S;    V1[0].Vect[1] = V1[4].Vect[1] = V0[1] + S;
  V1[1].Vect[0] = V1[5].Vect[0] = V0[0] + S;    V1[1].Vect[1] = V1[5].Vect[1] = V0[1] + S;
  V1[2].Vect[0] = V1[6].Vect[0] = V0[0] + S;    V1[2].Vect[1] = V1[6].Vect[1] = V0[1] - S;
  V1[3].Vect[0] = V1[7].Vect[0] = V0[0] - S;    V1[3].Vect[1] = V1[7].Vect[1] = V0[1] - S;

  V1[0].Vect[2] = V1[1].Vect[2] = V1[2].Vect[2] = V1[3].Vect[2] = V0[2] + S;
  V1[4].Vect[2] = V1[5].Vect[2] = V1[6].Vect[2] = V1[7].Vect[2] = V0[2] - S;

  GL_ObjRenderCube(V1);
}


/**** Add text ****/

VG_GLOBJ *GL_ObjText3(float *V0, VG_LONG Color, VG_LONG Flags, VG_UWORD Group,
                      char *Str, ...)
{
  char          Buf[256];
  va_list       vl;
  VG_GLOBJ      *Obj;

  va_start(vl, Str);
  vsprintf(Buf, Str, vl);
  va_end(vl);

  if ((Obj = GL_ObjAdd(VG_GLOBJ_TEXT3, sizeof(VG_GLTEXT) + strlen(Buf), Flags, Group)) != NULL) {
    strcpy(((VG_GLTEXT *)Obj -> Data) -> Str, Buf);
    GL_ObjChgVertex(&((VG_GLTEXT *)Obj -> Data) -> Vertex, V0, Color);
  }

  return Obj;
}


/**** Add a torsion monitor between four atoms ****/

VG_GLOBJ *GL_ObjTorA(ATOMO **Atm, VG_LONG Color, VG_LONG Flags,
                     VG_UWORD Group, char *LabelStr)
{
  VG_GLOBJ      *Obj;
  VG_ULONG      k;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_TORA, sizeof(VG_GLTORA) + strlen(LabelStr), Flags, Group)) != NULL) {
    for(k = 0; k < 4; ++k)
      ((VG_GLTORA *)Obj -> Data) -> Atm[k] = Atm[k];
    ((VG_GLTORA *)Obj -> Data) -> Color    = Color;
    strcpy(((VG_GLTORA *)Obj -> Data) -> LabelStr, LabelStr);
  }

  return Obj;
}


/**** Add a triangle ****/

VG_GLOBJ *GL_ObjTriangle(float V[3][3], VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_TRIANGLE, sizeof(VG_GLTRIANGLE), Flags, Group)) != NULL) {
    GL_ObjChgVertex(&((VG_GLTRIANGLE *)Obj -> Data) -> Vertex[0], V[0], Color);
    GL_ObjChgVertex(&((VG_GLTRIANGLE *)Obj -> Data) -> Vertex[1], V[1], Color);
    GL_ObjChgVertex(&((VG_GLTRIANGLE *)Obj -> Data) -> Vertex[2], V[2], Color);
  }

  return Obj;
}


/**** Add a triangle (atom coordinates) ****/

VG_GLOBJ *GL_ObjTriangleA(ATOMO **Atm, VG_LONG Color, VG_LONG Flags, VG_UWORD Group)
{
  VG_GLOBJ      *Obj;

  if ((Obj = GL_ObjAdd(VG_GLOBJ_TRIANGLEA, sizeof(VG_GLTRIANGLEA), Flags, Group)) != NULL) {
    GL_ObjChgVertexA(&((VG_GLTRIANGLEA *)Obj -> Data) -> Vertex[0], Atm[0], Color);
    GL_ObjChgVertexA(&((VG_GLTRIANGLEA *)Obj -> Data) -> Vertex[1], Atm[1], Color);
    GL_ObjChgVertexA(&((VG_GLTRIANGLEA *)Obj -> Data) -> Vertex[2], Atm[2], Color);
  }

  return Obj;
}


/**** Draw vertexes ****/

void GL_ObjVertex(register VG_GLVERTEX *Vertex, register VG_ULONG k)
{
  while(k--) {
    GL_ObjColor(Vertex -> Color);
    glVertex3fv(Vertex -> Vect);
    ++Vertex;
  } /* End of while */
}


/**** Draw atom vertexes ****/

void GL_ObjVertexA(register VG_GLVERTEXA *Vertex, register VG_ULONG k)
{
  while(k--) {
    GL_ObjColor(Vertex -> Color);
    glVertex3fv(&Vertex -> Atm -> x);
    ++Vertex;
  } /* End of while */
}




