
/*************************************************
****        VEGA - Selection utilities        ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <GL\gl.h>
#include <GL\glu.h>
#include <stdio.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "gl_global.h"
#include "gl_colors.h"
#include "gl_menu.h"
#include "gl_prefs.h"
#include "compar.h"

/**** Proximity selection modes ****/

#define  VG_GLSELWHAT_ATOM      0
#define  VG_GLSELWHAT_RESIDUE   1


VG_ULONG GL_SelectRes(ATOMO *, VG_UBYTE);


/**** Match atom ****/

VG_BOOL GL_MatchAtm(ATOMO *Atm, char **MyArg, VG_ULONG NumArgs, char *NMol)
{
  switch(NumArgs) {
  case 0:       /* Only one argument*/
    if (Wildncmp(MyArg[0], Atm -> Name.C, 4)) {
      return TRUE;
    }
    break;

  case 1:       /* Two arguments: AtomName:ResidueName */
    if ((Wildncmp(MyArg[0], Atm -> Name.C, 4)) &&
        (Wildncmp(MyArg[1], Atm -> ResName.C, 4))) {
      return TRUE;
    }
    break;

  case 2:       /* Three arguments: AtomName:ResidueName:ResidueNumber */
    if ((Wildncmp(MyArg[0], Atm -> Name.C, 4)) &&
        (Wildncmp(MyArg[1], Atm -> ResName.C, 4)) &&
        (Wildncmp(MyArg[2], Atm -> ResSeq.C, 4))) {
      return TRUE;
    }
    break;

  case 3:       /* Four arguments: AtomName:ResidueName:ResidueNumber:Chain */
    if ((Wildncmp(MyArg[0], Atm -> Name.C   , 4)) &&
        (Wildncmp(MyArg[1], Atm -> ResName.C, 4)) &&
        (Wildncmp(MyArg[2], Atm -> ResSeq.C , 4)) &&
        (Wildncmp(MyArg[3], &Atm -> ChainID , 1))) {
      return TRUE;
    }

  case 4:       /* Five arguments: AtomName:ResidueName:ResidueNumber:Chain:Molecule */
    if ((Wildncmp(MyArg[0], Atm -> Name.C   , 4)) &&
        (Wildncmp(MyArg[1], Atm -> ResName.C, 4)) &&
        (Wildncmp(MyArg[2], Atm -> ResSeq.C , 4)) &&
        (Wildncmp(MyArg[3], &Atm -> ChainID , 1)) &&
        (Wildncmp(MyArg[4], NMol            , 12))) {
      return TRUE;
    }
    break;
  } /* End of switch */

  return FALSE;
}


/**** Format the match string ****/

VG_ULONG GL_MatchFormatPattern(VG_ULONG *AtmNum, char **MyArg, char *Ptr)
{
  VG_ULONG              k    = 0;

  *AtmNum  = 0;
  MyArg[0] = Ptr;

  while(*Ptr) {
    if (*Ptr == ':') {
      ++k;
      if (k <= 4) {
        *Ptr = 0;
        MyArg[k] = Ptr + 1;
      } else break;
    }
    ++Ptr;
  } /* End of while */

  if ((!k) &&
      (ChkArgType(MyArg[0], strlen(MyArg[0]), (VG_UBYTE)(VG_PARTYPE_UINT|VG_PARTYPE_NOZERO)))) {
    sscanf(MyArg[0], "%d", AtmNum);
  }

  return k;
}


/**** Select atom(s) ****/

VG_ULONG GL_Select(ATOMO *Atm, char *Sel, VG_UBYTE Val, VG_BOOL Add)
{
  char          *MyArg[5], NMolStr[12];
  VG_ULONG      NumArgs, AtmNum;

  VG_ULONG      NMol     = 1;
  VG_ULONG      Selected = 0;

  NumArgs = GL_MatchFormatPattern(&AtmNum, MyArg, Sel);

  if ((!NumArgs) && (AtmNum)) {
    while(Atm) {
      if (Atm -> Num == AtmNum) {
        Atm -> Active = Val;
        Selected = 1;
        if (Add) break;
      } else if (!Add) Atm -> Active = !Val;
      Atm = Atm -> Ptr;
    } /* End of while */
  } else {
    NMolStr[0] = '1';
    NMolStr[1] = 0;
    while(Atm) {
      if (GL_MatchAtm(Atm, MyArg, NumArgs, NMolStr)) {
        Atm -> Active = Val;
        ++Selected;
      } else if (!Add) Atm -> Active = !Val;
      if (Atm -> Flags & VG_ATMF_MOLEND)
        sprintf(NMolStr, "%d", ++NMol);
      Atm = Atm -> Ptr;
    } /* End of while */
  }

  return Selected;
}


/**** Select backbone ****/

void GL_SelectAll(register ATOMO *Atm, register VG_BOOL Val)
{
  while(Atm) {
    Atm -> Active = Val;
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Select backbone ****/

VG_ULONG GL_SelectBackBone(ATOMO *Atm, VG_UBYTE Val, VG_BOOL ActOnly)
{
  VG_ULONG     Selected = 0;

  while(Atm) {
    if ((!ActOnly) || (Atm -> Active)) {
      if (IsBackBoneAtm(Atm)) {
        Atm -> Active = Val;
        ++Selected;
      } else Atm -> Active = !Val;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Selected;
}


/**** Select an element ****/

VG_ULONG GL_SelectElem(ATOMO *Atm, VG_UWORD Elem, VG_UBYTE Val, VG_BOOL Add)
{
  VG_ULONG     Selected = 0;

  while(Atm) {
    if (Atm -> Elem.S == Elem) {
      Atm -> Active = Val;
      ++Selected;
    } else if (!Add) Atm -> Active = !Val;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Selected;
}


/**** Invert the selection ****/

void GL_SelectInv(register ATOMO *Atm)
{
  while(Atm) {
    Atm -> Active = !Atm -> Active;
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Select around object ****/

VG_ULONG GL_SelectProx(ATOMO *InizAtm, char *Sel, float Rad, VG_UBYTE Val,
                       VG_UWORD What, VG_BOOL Add)
{
  ATOMO         *Atm1, *Atm2, *BegRes;
  char          *MyArg[5], NMolStr[12];
  VG_ULONG      AtmNum, NumArgs, NMol2, ResMol;

  VG_ULONG      NMol1    = 1;
  VG_ULONG      Selected = 0;

  Rad        *= Rad;
  NMolStr[0]  = '1';
  NMolStr[1]  = 0;
  NumArgs     = GL_MatchFormatPattern(&AtmNum, MyArg, Sel);

  if (!Add) {
    for(Atm1 = InizAtm; Atm1; Atm1 = Atm1 -> Ptr)
      Atm1 -> Active = !Val;
  }

  for(Atm1 = InizAtm; (Atm1) && (Atm1 -> Num != AtmNum); Atm1 = Atm1 -> Ptr) {
    if ((!NumArgs) && (AtmNum)) {
      if (Atm1 -> Num != AtmNum) {
        Atm1 -> Active = Val;
        BegRes         = InizAtm;
        NMol2          = 1;
        ResMol         = 1;
        Selected       = 1;
        for(Atm2 = InizAtm; Atm2; Atm2 = Atm2 -> Ptr) {
          if ((BegRes -> ResName.L != Atm2 -> ResName.L) ||
              (BegRes -> ResSeq.L  != Atm2 -> ResSeq.L) ||
              (BegRes -> ChainID   != Atm2 -> ChainID) ||
              (ResMol != NMol2)) {
            BegRes = Atm2;
            ResMol = NMol2;
          }

          if ((Atm1 != Atm2) && (Atm2 -> Active != Val)) {
            if ((Quad(Atm1 -> x - Atm2 -> x) + Quad(Atm1 -> y - Atm2 -> y) +
                 Quad(Atm1 -> z - Atm2 -> z)) <= Rad) {
              if (What = VG_GLSELWHAT_ATOM) {
                Atm2 -> Active = Val;
                ++Selected;
              } else {
                Selected += GL_SelectRes(BegRes, Val);
              }
            }
          }
          if (Atm2 -> Flags & VG_ATMF_MOLEND) ++NMol2;
        } /* End of for */
        break;
      }
    } else if (GL_MatchAtm(Atm1, MyArg, NumArgs, NMolStr)) {
      Atm1 -> Active = Val;
      BegRes         = InizAtm;
      NMol2          = 1;
      ResMol         = 1;
      ResMol         = 1;
      ++Selected;
      for(Atm2 = InizAtm; Atm2; Atm2 = Atm2 -> Ptr) {
        if ((BegRes -> ResName.L != Atm2 -> ResName.L) ||
            (BegRes -> ResSeq.L  != Atm2 -> ResSeq.L) ||
            (BegRes -> ChainID   != Atm2 -> ChainID) ||
            (ResMol != NMol2)) {
          BegRes = Atm2;
          ResMol = NMol2;
        }
        if ((Atm1 != Atm2) && (Atm2 -> Active != Val)) {
          if ((Quad(Atm1 -> x - Atm2 -> x) + Quad(Atm1 -> y - Atm2 -> y) +
               Quad(Atm1 -> z - Atm2 -> z)) <= Rad) {
            if (What == VG_GLSELWHAT_ATOM) {
              Atm2 -> Active = Val;
              ++Selected;
            } else {
              Selected += GL_SelectRes(BegRes, Val);
            }
          }
        }
        if (Atm2 -> Flags & VG_ATMF_MOLEND) ++NMol2;
      } /* End of for */
    }
    if (Atm1 -> Flags & VG_ATMF_MOLEND)
      sprintf(NMolStr, "%d", ++NMol1);
  } /* End of for */

  return Selected;
}


/**** Select a residue by atom ****/

VG_ULONG GL_SelectRes(ATOMO *BegRes, VG_UBYTE Val)
{
  ATOMO         *Atm     = BegRes;
  VG_ULONG      Selected = 0;

  while((Atm) &&
        (Atm -> ResName.L == BegRes -> ResName.L) &&
        (Atm -> ResSeq.L == BegRes -> ResSeq.L) &&
        (Atm -> ChainID == Atm -> ChainID) &&
        (!(Atm -> Flags & VG_ATMF_MOLEND))) {
    Atm -> Active = Val;
    Atm = Atm -> Ptr;
    ++Selected;
  } /* End of while */

  return Selected;
}


/**** Select range ****/

VG_ULONG GL_SelectRng(ATOMO *Atm, VG_UWORD Mode, VG_ULONG Start, VG_ULONG End,
                      VG_UBYTE Val, VG_BOOL Add)
{
  VG_ULONG              Temp;

  VG_ULONG              Selected = 0;

  if (Start > End) {
    Temp = Start;
    Start = End;
    End   = Temp;
  }

  while(Atm) {
    if (!Mode)                   /* Atom range   */
      Temp = Atm -> Num;
    else                        /* Residue range */
      Temp = Char2Int(Atm -> ResSeq.C, 4);

    if ((Temp >= Start) && (Temp <= End)) {
      Atm -> Active = Val;
      ++Selected;
    } else if (!Add) Atm -> Active = !Val;
    Atm = Atm -> Ptr;
  } /* End of while */

  return Selected;
}


/**** Select an element ****/

VG_ULONG GL_SelectWat(ATOMO *Atm, VG_UBYTE Val, VG_BOOL Add)
{
  ATOMO        *H1, *H2;

  VG_ULONG     Selected = 0;

  if (!Add) {
    for(H1 = Atm; H1; H1 = H1 -> Ptr)
      H1 -> Active = !Val;
  }

  while(Atm) {
    H1 = Atm -> Conn[0];
    H2 = Atm -> Conn[1];

#ifdef LITTLE_ENDIAN
    if ((Atm -> Elem.S == 0x4f) && (Atm -> NSost == 2) &&
        (H1 -> Elem.S == 0x48) && (H2 -> Elem.S == 0x48)) {
#else
    if ((Atm -> Elem.S == 0x4f00) && (Atm -> NSost == 2) &&
        (H1 -> Elem.S == 0x4800) && (H2 -> Elem.S == 0x4800)) {
#endif
      Atm -> Active = Val;
      H1  -> Active = Val;
      H2  -> Active = Val;
      ++Selected;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Selected;
}


/**** Check if the atom is in the backbone ****/

VG_BOOL IsBackBoneAtm(register ATOMO *Atm)
{
  register VG_ULONG     j, k;

  const VG_QCHAR        BackList[] = {{"N" }, {"CA"}, {"C"}, {"O"}, {"H"},
                                      {"O5*"}, {"P" }, {"O3*"}, {"C5*"}, {"C4*"}, {"C3*"}
                                     };
  VG_BOOL               Ret        = FALSE;

  for(k = 0; k < 11; ++k) {
    if (Atm -> Name.L == BackList[k].L) {
      for(j = 0; (!Ret) && (j < (VG_ULONG)Atm -> NSost); ++j) {
        for(k = 0; k < 11; ++k) {
          if (Atm -> Conn[j] -> Name.L == BackList[k].L) {
            Ret = TRUE;
            break;
          }
        } /* End of for */
      } /* End of for */
      break;
    }
  } /* End of for */
  return Ret;
}

