
/*************************************************
****        VEGA - Mol2 loader & saver        ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "bond.h"
#include "count.h"
#include "mol2.h"

#ifndef WIN32
#include <unistd.h>
#endif

/**** Local variables ****/

static const char       *Mol2Tags[] = { "@<TRIPOS>ATOM",
                                        "@<TRIPOS>BOND",
                                        "@<TRIPOS>MOLECULE",
                                        "@<TRIPOS>SUBSTRUCTURE",
                                        NULL
                                      };



/**** Assign the Tripos force field ****/

VG_BOOL AssignTriposFF(ATOMO *InizAtm, VG_ULONG TotAtm)
{
  char			Field[VG_TRIPOS_FIELDLEN], FieldName[VG_TRIPOS_FIELDLEN];
  FILE                  *IN;
  register ATOMO	*Atm;
  VG_OCHAR		*Template, *TmpPtr;
  VG_ULONG		Count;

  VG_BOOL               Ret    = FALSE;

  /**** Load the force field template ****/

  GetPrgPath(FieldName, TRUE);
  strcat(FieldName, VG_TRIPOS_TEMPLATE);
  if ((IN = fopen(FieldName, "r"))) {
    Count = 0;
    while(fgets(Field, VG_TRIPOS_FIELDLEN, IN))
      if ((*Field != '\n') && (*Field != ';')) ++Count;
    fclose(IN);
    if (Count) {
      if ((Template = (VG_OCHAR *)Alloca((Count + 1) * sizeof(VG_OCHAR)))) {
        if ((IN = fopen(FieldName, "r"))) {
          TmpPtr = Template;
          while(fgets(Field, VG_TRIPOS_FIELDLEN, IN))
            if ((*Field != '\n') && (*Field != ';')) {
              sscanf(Field, "%8s", TmpPtr -> C);
              ++TmpPtr;
            }
          fclose(IN);

          /**** Check atom types ****/

          Count = 0;
          for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
            for(TmpPtr = Template; TmpPtr -> DL; ++TmpPtr)
              if (TmpPtr -> DL == Atm -> Pot.DL) {
                ++Count;
                break;
              }
            if (!TmpPtr -> DL) break;
          }
          Ret = TRUE;
          if (Count != TotAtm) {

            /**** Assign TRIPOS force field ****/

            PrintProg(MSG_MOL2_ASSIGNPROG);
            AssignFF(FieldName, InizAtm, TotAtm, FALSE, TRUE);
          }
         } else Ret = PrintDosErr();
        FREE(Template);
      }
    } else CatErr(MSG_ERR_MOL2_ILLTEMP);
  } else CatErr(MSG_ERR_MOL2_NOTFOUND);

  return Ret;
}


/**** Check the connected substructures ****/

VG_BOOL Mol2ChkConnSub(VG_MOL2SUB *MainSub, ATOMO *MainAtm)
{
  ATOMO                 *Atm;
  VG_MOL2SUB            *Sub, *PrecSub;
  VG_UBYTE              k;

  for(k = 0; k < MainAtm -> NSost; ++k) {
    Atm = MainAtm -> Conn[k];
    if  ((Atm -> ResName.L != MainSub -> ResName.L) ||
         (Atm -> ResSeq.L  != MainSub -> ResSeq.L) ||
         (Atm -> ChainID   != MainSub -> ChainID)) {
      for(Sub = MainSub -> ConnSub; (Sub) &&
          ((Atm -> ResName.L != Sub -> ResName.L) ||
           (Atm -> ResSeq.L  != Sub -> ResSeq.L) ||
           (Atm -> ChainID   != Sub -> ChainID));
          Sub = Sub -> Ptr) {
        PrecSub = Sub;
      } /* End of for */

      if (!Sub) {
        if ((Sub = (VG_MOL2SUB *)Alloca(sizeof(VG_MOL2SUB))) != NULL) {
          if (MainSub -> ConnSub) PrecSub -> Ptr = Sub;
          else MainSub -> ConnSub = Sub;
          Sub -> ResName.L = Atm -> ResName.L;
          Sub -> ResSeq.L  = Atm -> ResSeq.L;
          Sub -> ChainID   = Atm -> ChainID;
          ++MainSub -> ConnSubNum;
        } else {
          Mol2FreeSub(MainSub -> ConnSub);
          return FALSE;
        }
      }
    }
  } /* End of for */

  return TRUE;
}


/**** Find the substructures ****/

VG_MOL2SUB *Mol2FindSub(ATOMO *InizAtm, VG_ULONG *TotSub)
{
  ATOMO                 *Atm;
  VG_MOL2SUB            *PrecSub, *Sub;

  VG_MOL2SUB            *BegSub = NULL;

  *TotSub = 0;
  for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
    if ((Atm == InizAtm) ||
        (Atm -> ResName.L != Atm -> Prev -> ResName.L) ||
        (Atm -> ResSeq.L  != Atm -> Prev -> ResSeq.L) ||
        (Atm -> ChainID   != Atm -> Prev -> ChainID)) {
      for(Sub = BegSub; (Sub) &&
          ((Atm -> ResName.L != Sub -> ResName.L) ||
           (Atm -> ResSeq.L  != Sub -> ResSeq.L) ||
           (Atm -> ChainID   != Sub -> ChainID));
         Sub = Sub -> Ptr);

      if (!Sub) {
        if ((Sub = (VG_MOL2SUB *)Alloca(sizeof(VG_MOL2SUB))) != NULL) {
          if (BegSub) PrecSub -> Ptr = Sub;
          else BegSub = Sub;
          PrecSub = Sub;
          Sub -> ResName.L = Atm -> ResName.L;
          Sub -> ResSeq.L  = Atm -> ResSeq.L;
          Sub -> ChainID   = Atm -> ChainID;
          Sub -> AtmNum    = Atm -> Num;
          Sub -> SubNum    = ++*TotSub;
          Mol2ChkConnSub(Sub, Atm);
        } else {
          *TotSub = 0;
          Mol2FreeSub(BegSub);
          return NULL;
        }
      }
    } else {
      if (Atm -> Name.L == VG_TRIPOS_ATM_CA) Sub -> AtmNum = Atm -> Num;
      Mol2ChkConnSub(Sub, Atm);
    }
  } /* End of for */

  return BegSub;
}


/**** Free the substructures ****/

void Mol2FreeSub(VG_MOL2SUB *Sub)
{
  VG_MOL2SUB    *NextSub;

  while(Sub) {
    NextSub = Sub -> Ptr;
    if (Sub -> ConnSub) Mol2FreeSub(Sub -> ConnSub);
    FREE(Sub);
    Sub = NextSub;
  } /* End of while */
}


/**** Get the substructure for an atom ****/

VG_MOL2SUB *Mol2GetSub(VG_MOL2SUB *Sub, ATOMO *Atm)
{
  while((Sub) &&
        ((Atm -> ResName.L != Sub -> ResName.L) ||
         (Atm -> ResSeq.L  != Sub -> ResSeq.L) ||
         (Atm -> ChainID   != Sub -> ChainID))) {
    Sub = Sub -> Ptr;
  } /* End of while */

  return Sub;
}


/**** Sybyl Mol2 Loader ****/

ATOMO *Mol2Load(FILE *IN, RECORD *Lin, VG_ULONG *TotAtomi)
{
  ATOMO         *Atm, **ConMtx;
  char          Name[10], ResName[12], ResSeq[10];
  VG_BYTE       ChainID;
  VG_LONG       Mol2Atms;
  VG_UBYTE      Bond;
  VG_ULONG      *SubPtr, LineNum;
  VG_ULONG      i, j;

  ATOMO         *InizAtm    = NULL;
  ATOMO         *Mol2Last   = NULL;
  VG_BOOL       Err         = FALSE;
  VG_BYTE       PrecChainID = 0;
  VG_LONG       SubCount    = 0;
  VG_ULONG      *SubStr     = NULL;
  VG_ULONG      Tag         = VG_TRIPOS_TAG_UNK;


  fseek(IN, 0, SEEK_SET);
  while(fgets(Lin -> Line, VG_LINELEN, IN)) {
    if ((*Lin -> Line) && (*Lin -> Line != '\n')) {
      if (*Lin -> Line != '@') {
        switch(Tag) {
        case VG_TRIPOS_TAG_ATOM:
          if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
            ResName[3] = 0;
            sscanf(Lin -> Line, "%*s %8s %f %f %f %8s %8s %10s %f",
                   Name, &Atm -> x, &Atm -> y, &Atm -> z,
                   Atm -> Pot.C, ResSeq, ResName, &Atm -> Charge);
            Str2Qchar(&Atm -> Name   , Name   );
            AtmName2Elem(Atm);

            if (SubCount >= Mol2Atms) {
              if (!Err) {
                CatErr(MSG_ERR_MOL2_CORRUPTED);
                Err = TRUE;
              }
            } else {
              sscanf(ResSeq, "%d", &SubStr[SubCount]);
              ConMtx[SubCount] = Atm;
              ++SubCount;
            }

            j = strlen(ResName);
            for(i = 0; i < j; ++i) {
              if ((ResName[i] >= '1') && (ResName[i] <= '9')) {
                Str2Qchar(&Atm -> ResSeq, &ResName[i]);
                ResName[i] = 0;
                break;
              }
            } /* End of for */
            Str2Qchar(&Atm -> ResName, ResName);

            if (i == j) Str2Qchar(&Atm -> ResSeq , ResSeq);

            Mol2Last = Atm;
          } else break;
          break;

        case VG_TRIPOS_TAG_BOND:
          i = j = 0;
          sscanf(Lin -> Line, "%*s %d %d %11s", &i, &j, ResName);
          if ((!ResName[1]) && (ResName[0] > '0') && (ResName[0] < '4')) Bond = *ResName - '0';
          else if ((ResName[0] == 'a') && (ResName[1] == 'r')) Bond = VG_BOND_PARDOUBLE;
               else Bond = VG_BOND_SINGLE;

          if ((i > 0) && (i <= (VG_ULONG)Mol2Atms) && (j > 0) && (j <= (VG_ULONG)Mol2Atms)) {
            BndMake(ConMtx[--i], ConMtx[--j], Bond);
            GLOBSW_CONNCALC = FALSE;
          } else if (!Err) {
            CatErr(MSG_ERR_MOL2_CORRUPTED);
            Err = TRUE;
          }
          break;

        case VG_TRIPOS_TAG_MOLECULE:
          if (LineNum == 2) {
            Mol2Atms = 0;
            sscanf(Lin -> Line, "%d", &Mol2Atms);
            if ((Mol2Atms <= 0) ||
                ((SubStr = (VG_ULONG *)Alloca(Mol2Atms * sizeof(VG_ULONG))) == NULL) ||
                ((ConMtx = (ATOMO **)Alloca(Mol2Atms * sizeof(ATOMO *))) == NULL)) {
              CatErr(MSG_ERR_MOL2_CORRUPTED);
              return NULL;
            }
          }
          break;

        case VG_TRIPOS_TAG_SUBSTR:
          if ((Mol2Last) && (!Err)) {
            *ResName = 0;
            *ResSeq  = 0;
            sscanf(Lin -> Line, "%d %*s %*s %*s %*s %4s", &j, ResSeq);
            ChainID = *ResSeq;
            if ((!ChainID) || (ChainID == '*')) ChainID = ' ';

            SubPtr = SubStr;
            for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
              if (j == *SubPtr++)
                Atm -> ChainID = ChainID;
            } /* End of for */
          }
          break;
        } /* End of switch */
      } else {
        for(Tag = 0; (Mol2Tags[Tag]) && strncmp(Lin -> Line, Mol2Tags[Tag], strlen(Mol2Tags[Tag])); ++Tag);
        LineNum = 0;
      }
    }
    ++LineNum;
  } /* End of while */

  if (Mol2Last) {

    /**** Fix the segments end ****/

    PrecChainID = InizAtm -> ChainID;
    for(Atm = InizAtm; (Atm); Atm = Atm -> Ptr) {
      if (Atm -> ChainID != PrecChainID) {
        if (Atm -> Prev) Atm -> Prev -> Flags |= VG_ATMF_SEGEND;
        PrecChainID = Atm -> ChainID;
      }
    } /* End of for */
    LastAtm = Mol2Last;
    LastAtm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
  } else CatErr(MSG_ERR_MOL2_CORRUPTED);

  if (ConMtx) FREE(ConMtx);
  if (SubStr) FREE(SubStr);

  return InizAtm;
}


/**** Sybyl Mol2 saver ****/

VG_BOOL Mol2Save(FILE *OUT, ATOMO *InizAtm, VG_ULONG TotAtm, char *Name)
{
  ATOMO                 *Atm;
  char			*CharTyp, *LogName, *Ptr;
  char                  ResName[12];
  FILE			*IN;
  time_t		Orario;
  VG_MOL2SUB            *BegSub, *PrecSub, *Sub;
  VG_BYTE               ChainID, PrecChainID;
  VG_ULONG		Count, k;

  char                  *DotPtr  = NULL;
  char                  *BegName = Name;
  VG_BOOL               IsRoot   = TRUE;
  VG_BOOL		Ret      = TRUE;

#ifdef __VG_OPENGL
  VG_DLONG              *Store   = NULL;

  if (GLOBSW_OPENGL)
    Store = FFPush(InizAtm, TotAtm);
#endif

  if ((Ret = AssignTriposFF(InizAtm, TotAtm))) {
    if (*Name) {
      for(Ptr = Name + strlen(Name); Ptr != Name; --Ptr) {
        if ((!DotPtr) && (*Ptr == '.')) {
          DotPtr = Ptr;
          *Ptr   = 0;
        }
#ifdef WIN32
        if (*Ptr == '\\') {
#else
#  ifdef AMIGA
        if ((*Ptr == '/') || (*Ptr == ':')) {
#  else
        if (*Ptr == '/') {
#  endif
#endif
          BegName = Ptr + 1;
          break;
        }
      } /* End of for */
    } else BegName = "Unknown";
#ifndef WIN32
    LogName = getlogin();
#else
    LogName = "User";
#endif
    Orario  = time(0);
    Ptr     = ctime(&Orario);
    if (ChkCharges(InizAtm, TotAtm, NULL)) CharTyp = "USER";
    else CharTyp = "NO";

    if ((BegSub = Mol2FindSub(InizAtm, &Count)) != NULL) {
      if (fprintf(OUT, "#       Name:                   %s\n" \
                       "#       Creating user name:     %s\n" \
                       "#       Creation time:          %s\n\n" \
                       "#       Modifying user name:    %s\n" \
                       "#       Modification time:      %s\n\n" \
                       "@<TRIPOS>MOLECULE\n%s\n" \
                       "%5d %5d %5d    0    0\n" \
                       "SMALL\n%s_CHARGES\n\n%s\n" \
                       "@<TRIPOS>ATOM\n",
                        BegName, LogName, Ptr, LogName, Ptr, BegName, TotAtm,
                        CountBond(InizAtm), Count, CharTyp, VEGAHdr) > 0) {
        Count   = 1;
        PrecSub = NULL;
        for(Atm = InizAtm; (Atm) && (Ret); Atm = Atm -> Ptr) {
          Sub = Mol2GetSub(BegSub, Atm);
          if (Sub != PrecSub) {
            strcpy(ResName, Qchar2Str(&Atm -> ResName));
            strcat(ResName, Qchar2Str(&Atm -> ResSeq));
            PrecSub = Sub;
          }
          if (fprintf(OUT, "%7d %-8.4s %9.4f %9.4f %9.4f %-6.6s%5d %-8s %9.4f\n",
                           Count++, Atm -> Name.C, Atm -> x, Atm -> y, Atm -> z,
                           Atm -> Pot.C, Sub -> SubNum, ResName, Atm -> Charge) <= 0) {
            Ret = PrintDosErr();
          }
        }

        /**** Bond tag ****/

        if ((Ret) && ((Ret = BndAssignTypes(InizAtm, TotAtm, FALSE)))) {
          if (fprintf(OUT, "%s\n", Mol2Tags[VG_TRIPOS_TAG_BOND]) > 0) {
            Count = 1;
            for(Atm = InizAtm; (Atm) && (Ret); Atm = Atm -> Ptr) {
              for(k = 0; (Ret) && (k < (VG_ULONG)Atm -> NSost); ++k) {
                if (Atm -> Conn[k] -> Num > Atm -> Num) {
                  if (Atm -> Order[k] < VG_BOND_TRIPLE) {
                    if (((Atm -> Pot.DL == VG_TRIPOS_FF_Nam) &&
                         (Atm -> Conn[k] -> Pot.DL == VG_TRIPOS_FF_C2)) ||
                        ((Atm -> Pot.DL == VG_TRIPOS_FF_C2) &&
                         (Atm -> Conn[k] -> Pot.DL == VG_TRIPOS_FF_Nam))) {
                        ResName[0] = 'a';
                        ResName[1] = 'm';
                        ResName[2] = 0;
                      } else {
                        ResName[0] = Atm -> Order[k] + '0';
                        ResName[1] = 0;
                      }
                    } else if (Atm -> Order[k] == VG_BOND_PARDOUBLE) {
                      ResName[0] = 'a';
                      ResName[1] = 'r';
                      ResName[2] = 0;
                    }

                    if (fprintf(OUT, "%6d %4d %4d %.4s\n",
                                Count++, Atm -> Num, Atm -> Conn[k] -> Num, ResName) <= 0)
                      Ret = PrintDosErr();
                  }
                } /* End of for */
              } /* End of for */

              /**** Substructure Tag ****/

              if (Ret) {
                if (fprintf(OUT, "%s\n", Mol2Tags[VG_TRIPOS_TAG_SUBSTR]) > 0) {
                Count       = 1;
                PrecChainID = 0;
                for(Sub = BegSub; Sub && Ret; Sub = Sub -> Ptr) {
                  strcpy(ResName, Qchar2Str(&Sub -> ResName));
                  strcat(ResName, Qchar2Str(&Sub -> ResSeq));
                  Ptr = "ROOT";
                  if (Sub -> ConnSubNum > 0) {
                    if ((!IsRoot) && (PrecChainID == Sub -> ChainID)) Ptr = "";
                    if (Sub -> ConnSubNum == 1)
                      IsRoot = !IsRoot;
                    PrecChainID = Sub -> ChainID;
                  } else {
                    PrecChainID = 0;
                    IsRoot      = TRUE;
                  }

                  if (Sub -> ChainID == ' ') ChainID = 'A';
                  else ChainID = Sub -> ChainID;

                  if (fprintf(OUT, "%6d %-8s %4d RESIDUE           1 %c     %-4.4s %4d %s\n",
                                   Count++, ResName, Sub -> AtmNum, ChainID,
                                   Sub -> ResName.C, Sub -> ConnSubNum, Ptr) <= 0)
                    Ret = PrintDosErr();
                } /* End of for */
              } else Ret = PrintDosErr();
            }
          } else Ret = PrintDosErr();
        }
      } else Ret = PrintDosErr();
      Mol2FreeSub(BegSub);
    }
    if (DotPtr) *DotPtr = '.';
  }

#ifdef __VG_OPENGL
  if (Store) {
    FFPop(InizAtm, Store);
    FREE(Store);
  }
#endif

  return Ret;
}


