
/*************************************************
****         VEGA - Molecule managment        ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "vglist.h"
#include "bond.h"
#include "selmol.h"

#ifdef __VG_OPENGL
#  include "gl_matrix.h"
#endif

/**** Macros ****/

#define  CRC_ROTATE_RIGHT(c)	if ((c) & 01) (c) = ((c) >>1) + 0x8000; else (c) >>= 1;


/**** Find molecule number ****/

VG_ULONG MolListFindMol(register ATOMO *Find)
{
  register ATOMO        *Atm;
  register VG_ULONG     NMol = 1;

  for(Atm = BegAtm; (Atm) && (Atm != Find); Atm = Atm -> Ptr) {
    if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
  } /* End of for */

  return NMol;
}


#ifdef __VG_OPENGL

/**** Calc the element checksum of a molecule ****/

VG_ULONG MolElemCrc(register ATOMO *Atm, register VG_BOOL ActiveOnly)
{
  register VG_ULONG	Crc = 0;

  while(Atm) {
    if (((!ActiveOnly) || (Atm -> Active)) &&
        (!(Atm -> Flags & VG_ATMF_CENTROID))) {
      CRC_ROTATE_RIGHT(Crc);
      Crc += Atm -> Elem.C[0];
      Crc &= 0xffff;
      CRC_ROTATE_RIGHT(Crc);
      Crc += Atm -> Elem.C[1];
      Crc &= 0xffff;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Crc;
}


/**** Find atom number in the list ****/

VG_MOLLIST *MolListAtm2List(VG_MOLLIST *List, VG_ULONG AtmNum)
{
  while(List) {
    if ((AtmNum >= List -> Begin -> Num) &&
        (AtmNum <= List -> End -> Num))
      break;
    List = List -> Ptr;
  } /* End of while */

  return List;
}


/**** Build the molecule list ****/

VG_MOLLIST *MolListBuild(ATOMO *Atm, VG_BYTE Flags)
{
  ATOMO                 *PrecAtm;
  VG_MOLLIST            *List, *PrecList;

  VG_BOOL               Add      = TRUE;
  VG_MOLLIST            *BegList = NULL;
  VG_ULONG              NMol     = 0;

  while(Atm) {
    if (Add) {
      if ((List = (VG_MOLLIST *)Alloca(sizeof(VG_MOLLIST))) != NULL) {
        if (BegList) {
          PrecList -> Ptr = List;
          PrecList        = List;
        } else BegList = PrecList = List;
        List -> Begin = Atm;
        List -> Num   = ++NMol;
      } else {
        MolListFree(&BegList);
        break;
      }
      Add = FALSE;
    }
    if (Atm -> Flags & Flags) {
      List -> End = Atm;
      Add = TRUE;
    }
    PrecAtm = Atm;
    Atm     = Atm -> Ptr;
  } /* End of while */

  if (BegList) List -> End = PrecAtm;

  return BegList;
}


/****Count the molecules ****/

VG_ULONG MolListCountMol(void)
{
  register ATOMO        *Atm;
  register VG_ULONG     NMol = 0;

  for(Atm = BegAtm; Atm; Atm = Atm -> Ptr) {
    if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
  } /* End of for */

  return NMol;
}


/**** Return the first atom, the last atom and ****/
/**** the number of atoms by molecule number   ****/

VG_ULONG MolListGet(VG_ULONG MolNum, ATOMO **FirstAtm, ATOMO **LastAtm)
{
  ATOMO         *Atm;

  ATOMO         *First = NULL;
  ATOMO         *Last  = NULL;
  VG_ULONG      Atoms  = 0;
  VG_ULONG      NMol   = 1;

  for(Atm = BegAtm; (Atm) && (NMol <= MolNum); Atm = Atm -> Ptr) {
    if (MolNum == NMol) {
      if (!First) First = Atm;
      if (Atm -> Flags & VG_ATMF_MOLEND) Last = Atm;
      ++Atoms;
    }
    if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
  } /* End of for */

  if (FirstAtm) *FirstAtm = First;
  if (LastAtm)  *LastAtm  = Last;

  return Atoms;
}


/**** Free the molecule list from the memory ****/

void MolListFree(VG_MOLLIST **BegList)
{
  register VG_MOLLIST   *Tmp;
  register VG_MOLLIST   *List = *BegList;

  while(List) {
    Tmp = List -> Ptr;
    FREE(List);
    List = Tmp;
  } /* End of while */

  *BegList = NULL;
}


/**** Remove an atom range ****/

void MolListRemAtm(ATOMO *InizAtm, VG_ULONG Start, VG_ULONG End)
{
  ATOMO                 *Atm, *PrecAtm;
  VG_UBYTE              i;

  VG_ULONG              Count = InizAtm -> Num;

  PrecAtm = InizAtm -> Prev;
  for(Atm = InizAtm; Atm; ) {
    if ((Atm -> Num >= Start) && (Atm -> Num <= End)) {
      for(i = 0; i < Atm -> NSost; ++i)
        BndUnlink(Atm -> Conn[i], Atm);
      --TotalAtm;
      RemoveAtm(&Atm, &PrecAtm, &BegAtm);
    } else {
      PrecAtm    = Atm;
      Atm -> Num = Count++;
      Atm        = Atm -> Ptr;
    }
  } /* End of while */
}



/**** Rotate a molecule ****/

void MolRotate(ATOMO *Atm, float Ax, float Ay, float Az,
               XYZ *RotCenter, float WorldMat[4][4])
{
  float         x, y, z;
  float         InvWorldMat[4][4], MolMat[4][4];
  float         Tmp[4][4];

  /**** Molecule matrix ****/

  GL_Mat4Reset(MolMat);
  if (Ax) GL_Mat4RotXYZ(1, 0, 0, Ax, MolMat);
  if (Ay) GL_Mat4RotXYZ(0, 1, 0, Ay, MolMat);
  if (Az) GL_Mat4RotXYZ(0, 0, 1, Az, MolMat);

  /**** Inverse world matrix ****/

  CopyMemory(InvWorldMat, WorldMat, sizeof(float) * 16);
  GL_Mtx4Inverse(InvWorldMat);

  /**** Calculate the new transformation matrix ****/

  GL_Mat4Mult(Tmp, WorldMat, MolMat);
  GL_Mat4Mult(MolMat, Tmp, InvWorldMat);

  /**** Apply the transformation ****/

  while(Atm) {

    x = Atm -> x - RotCenter -> x;
    y = Atm -> y - RotCenter -> y;
    z = Atm -> z - RotCenter -> z;

    Atm -> x = x * MolMat[0][0] + y * MolMat[1][0] + z * MolMat[2][0] + RotCenter -> x;
    Atm -> y = x * MolMat[0][1] + y * MolMat[1][1] + z * MolMat[2][1] + RotCenter -> y;
    Atm -> z = x * MolMat[0][2] + y * MolMat[1][2] + z * MolMat[2][2] + RotCenter -> z;

    if (Atm -> Flags & VG_ATMF_MOLEND) break;
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Translate a molecule ****/

void MolTranslate(ATOMO *Atm, float Dx, float Dy, float Dz,
                  XYZ *RotCenter, float WorldMat[4][4])
{
  float         x1, y1, z1;
  float         x2, y2, z2;
  float         InvWorldMat[4][4];

  /**** Inverse world matrix ****/

  CopyMemory(InvWorldMat, WorldMat, sizeof(float) * 16);
  GL_Mtx4Inverse(InvWorldMat);

  while(Atm) {
    x1 = Atm -> x - RotCenter -> x;
    y1 = Atm -> y - RotCenter -> y;
    z1 = Atm -> z - RotCenter -> z;

    /**** World transformation & translation ****/

    x2 = x1 * WorldMat[0][0] + y1 * WorldMat[1][0] + z1 * WorldMat[2][0] + Dx;
    y2 = x1 * WorldMat[0][1] + y1 * WorldMat[1][1] + z1 * WorldMat[2][1] + Dy;
    z2 = x1 * WorldMat[0][2] + y1 * WorldMat[1][2] + z1 * WorldMat[2][2] + Dz;

    /**** Inverse world transformation ****/

    Atm -> x = x2 * InvWorldMat[0][0] + y2 * InvWorldMat[1][0] + z2 * InvWorldMat[2][0] + RotCenter -> x;
    Atm -> y = x2 * InvWorldMat[0][1] + y2 * InvWorldMat[1][1] + z2 * InvWorldMat[2][1] + RotCenter -> y;
    Atm -> z = x2 * InvWorldMat[0][2] + y2 * InvWorldMat[1][2] + z2 * InvWorldMat[2][2] + RotCenter -> z;

    if (Atm -> Flags & VG_ATMF_MOLEND) break;
    Atm = Atm -> Ptr;
  } /* End of while */

  /**** Translate the rotation center ****/

  RotCenter -> x += Dx * InvWorldMat[0][0] + Dy * InvWorldMat[1][0] + Dz * InvWorldMat[2][0];
  RotCenter -> y += Dx * InvWorldMat[0][1] + Dy * InvWorldMat[1][1] + Dz * InvWorldMat[2][1];
  RotCenter -> z += Dx * InvWorldMat[0][2] + Dy * InvWorldMat[1][2] + Dz * InvWorldMat[2][2];
}


/**** Create the residue list ****/

VG_LIST *ResListBuild(ATOMO *Atm)
{
  VG_LIST       *ResList;
  VG_RESLIST    *CurRes;

  VG_ULONG      NMol = 1;

  if (!Atm) return NULL;

  if ((ResList = ListNew(sizeof(VG_RESLIST))) != NULL) {
    if ((CurRes = (VG_RESLIST *)ListAppend(ResList)) != NULL) {
      CurRes -> Begin = Atm;
      CurRes -> NMol  = NMol;
      for(;;) {
        if ((Atm -> ResName.L != CurRes -> Begin -> ResName.L) ||
            (Atm -> ResSeq.L  != CurRes -> Begin -> ResSeq.L ) ||
            (Atm -> ChainID   != CurRes -> Begin -> ChainID  )) {
          CurRes -> End = Atm -> Prev;
          if ((CurRes = (VG_RESLIST *)ListAppend(ResList)) != NULL) {
            CurRes -> Begin = Atm;
            CurRes -> NMol  = NMol;
          } else {
            ListClear(ResList);
            return NULL;
          }
        } else if ((Atm -> Flags & VG_ATMF_SEGEND) ||
                   (Atm -> Flags & VG_ATMF_MOLEND)) {
          CurRes -> End = Atm;
          if (Atm -> Ptr) {
            if ((CurRes = (VG_RESLIST *)ListAppend(ResList)) != NULL) {
              CurRes -> Begin = Atm -> Ptr;
              CurRes -> NMol  = NMol;
            } else {
              ListClear(ResList);
              return NULL;
            }
          }
        }
        if (Atm -> Flags & VG_ATMF_MOLEND) ++NMol;
        if (!Atm -> Ptr) {
          CurRes -> End = Atm;
          break;
        }
        Atm = Atm -> Ptr;
      } /* End of for */
    } else {
      ListClear(ResList);
      return NULL;
    }
  }

  return ResList;
}

#endif
