/***************************************************************************

  SODIUM: A program for arranging ions around biological macromolecules.

  VERSION: 1.5.1.

  AUTHOR:  Alexander Balaeff.

 (C) Copyright 2000 The Theoretical Biophysics Group, Beckman Institute, and
                    The Board of Trustees of the University of Illinois

 ***************************************************************************
  DESCRIPTION:

   This program places the required number of sodium ions around a system
 of electric charges, e.g., the atoms of a biological macromolecule (protein,
 DNA, protein/DNA complex).  The ions are placed in the nodes of a cubic grid,
 in which the electrostatic energy achieves the smallest values.  The energy
 is re-computed after placement of each ion.  A simple Coulombic formula
 is used for the energy:
           Energy(R) = Sum(i_atoms,ions)  Q_i / |R-R_i| .
 All the constants are dropped out from this formula, resulting in some
 weird energy units; that doesn't matter for the purpose of energy comparison.

  To speed the program up, the atoms of the macromolecule are re-located to
 the grid nodes, closest to their original locations.  The resulting error
 is believed to be minor, compared to that resulting from the one-by-one
 ions placement, or from using the simplified energy function.

  The coordinates of the placed ions are printed out in the PDB format for
 further usage.  It is recommended that the placed ions are equilibrated in
 a separate Monte Carlo or Molecular Dynamics simulation.

  Trivial modifications to the program should allow the placement of any
 combination of multivalent ions of different charges.

 ***************************************************************************
  SUGGESTED COMPILATION COMMAND LINE (FOR A DEC-ALPHA CC-COMPILER):

  cc -lm -fast -tune host -assume whole_program -arch host -o sodium sodium.c

 ***************************************************************************
  COMMAND LINE FOR THE PROGRAM EXECUTION:

  sodium < config_file > log_file &

 ***************************************************************************
  SPEED AND MEMORY REQUIREMENTS:

  The requested memory and speed heavily depend on the number of atoms in
 the macromolecule (as a linear function) and the cell size of the used grid
 (as an inverse cube).  For 24,000 atoms, 0.5 A grid and 48 ions to place,
 the program required 74Mb of RAM and was executed in about 1.5 hours on a
 single DEC Alpha processor.

 ***************************************************************************
  A SAMPLE CONFIGURATION FILE:

  NUM_IONS      48
  R_ION_PRODNA  6.5
  R_ION_ION     11.0
  GRID_STEP     0.5
  BORDER_WIDTH  10.0
  PDB_NAME      lac_w_dna_charges_orient.pdb
  PDB_OUT       ions_0.5.pdb

 ***************************************************************************
  EXPLANATION OF THE CONFIGURATION PARAMETERS:

  NUM_IONS  -  the number of ions to place around the macromolecule.
  R_ION_PRODNA  -  the closest distance (in Angstroms) allowed between any
                of the placed ions and any atom of the macromolecule.
		    Due to an error in atomic positions, introduced by using
		    the grid, it is recommended to make this parameter slightly
		    larger than the actually allowed closest distance.
  R_ION_ION  -  the closest distance between any two ions (in Angstroms).
                This parameter prevents the placed ions from clustering
		    around a single negative charge, and would prevent
		    positively and a negatively charged ions from being placed
		    too close to each other.
  GRID_STEP  -  the cell size of the cubic grid on which the ions are placed
                (in Angstroms).  Grids with smaller cells allow more precise
		    ions placement (in part, due to the least disturbance of the
		    macromolecular atoms, relocated to the grid nodes). However,
		    a finer grid slows the computations down and requires
		    significantly more RAM.
  BORDER_WIDTH  -  the thickness of the grid margin, surrounding the macromolecule
                   (in Angstroms).
  PDB_NAME  -  the name of the file, containing the coordinates of the atoms of
               the macromolecule in PDB format.  The 10th column (B column) of
		   the PDB file should contain the charges of the atoms.  Such file
		   can be generated from a usual PDB/PSF couple, e.g., by using
		   the following X-PLOR (http://atb.csb.yale.edu/xplor) script:

		          structure @lac_w_dna.psf end
		          coor @lac_w_dna.pdb
			    vector do (B=CHARGE) (all)
			    write coor output="lac_w_dna_charges.pdb" end
			    stop

		   To reduce the number of nodes in the grid, it is also recommended
		   that the molecule has its axes of inertia aligned along the
		   axes of the coordinate system.  This can be achieved by using
		   another X-PLOR script:

		          structure @lac_w_dna.psf end
			    coor @lac_w_dna_charges.pdb
			    coor orient end
			    write coor output="lac_w_dna_charges_orient.pdb" end
			    stop

  PDB_OUT  -  the name for the PDB file which will contain the coordinates
              of the placed ions.

 ***************************************************************************/


/**********************************/
/*** The required C libraries:  ***/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "globdef.h"
#include "globvar.h"



/************************************/
/*** The compilation parameters:  ***/
/***                              ***/
#define MAXATOMS 50000  /*** the largest allowed number of macromolecule atoms ***/
#define MAXIONS 500     /*** the largest allowed number of ions ***/
#define MAXGRID 10000   /*** the largest allowed number of grid nodes ***/
                        /*** in any one dimension                     ***/
//#define rint(X)     floor((X))

/*
double rint (double num) {
  return floor(num + 0.5);
}
*/

/*** If any of the above parameters is insufficient for your system, ***/
/*** recompile the program with that number accordingly changed.     ***/


/**********************************/
/*** The program return codes:  ***/
/***                            ***/
#define ERR_NO_ERROR             0
#define ERR_INCOMPLETE_CONFIG   -1
#define ERR_FAILED_OUTPUT       -2
#define ERR_BAD_PDBFILE         -3
#define ERR_INSUF_MEM_SQRTS    -31
#define ERR_INSUF_MEM_EXCLMAP  -32
#define ERR_INSUF_MEM_ENERGY   -33


/*********************************************/
/***  The list of auxiliary subroutines    ***/
/*** (each is described below in the code) ***/
/***                                       ***/
long int min_element (float *, long int , unsigned char *);
int excl_around( long int, long int, long int, float,
		     unsigned char *, long int, long int, long int);
int indices( long int,
		 long int, long int, long int,
		 long int *, long int *, long int *);
int read_config_line_int(char *, const char *, long int *);
int read_config_line_float(char *, const char *, float *);
int read_config_line_string(char *, const char *, char *);



/**********************************************************/
/******************** THE MAIN MODULE  ********************/
/**********************************************************/

VG_BOOL AddIons(char *Elem, float Charge, VG_ULONG ions, float excl_atom_rad,
                float excl_ion_rad, float delta, float border_width)
{
  ATOMO		*Atm;

  long int      N_ions = ions;

/************************************/
/***  THE LIST OF THE VARIABLES:  ***/
/************************************/

/************************************/
/*** Input files and file names:  ***/
/***                              ***/
  FILE *pdb_file, *pdb_outfile;
  char pdb_name[100]="\0", pdb_out_name[100]="\0";


/**********************************/
/*** Macromolecule parameters.  ***/
/***                            ***/
/*** number of atoms:  ***/
  long int N_atoms;
/*** atomic coordinates:  ***/
  static float x_at[MAXATOMS], y_at[MAXATOMS], z_at[MAXATOMS];
/*** atomic charges:  ***/
  static float q_at[MAXATOMS];
  float *q_cur;
/*** integer coordinates of the atoms, relocated on the grid:  ***/
  static long int ii_at[MAXATOMS], jj_at[MAXATOMS], kk_at[MAXATOMS];
/*** auxiliary arrays and pointers, storing the squares of the ***/
/*** distances between the atoms and the current grid node:    ***/
  static long int i_tmp_at_0[MAXATOMS], i_tmp_at_1[MAXATOMS];
  long int *i_tmp, *j_tmp;
/*** the dimensions of the macromolecule:  ***/
  float x_max=0.0, y_max=0.0, z_max=0.0;
  float x_min=0.0, y_min=0.0, z_min=0.0;
/*** the closest allowed distance from any ion to any atom of the macromolecule:  ***/


/************************/
/*** Ions parameters. ***/
/***                  ***/
/*** number of ions to place:  ***/
/*** the integer ions coordinates on the grid:  ***/
  long int ii_ions[MAXIONS], jj_ions[MAXIONS], kk_ions[MAXIONS];
/*** the float values of the ions coordinates:  ***/
  float x_ions[MAXIONS], y_ions[MAXIONS], z_ions[MAXIONS];
/*** the auxilliary variables storing the squares of the ***/
/*** distances between the current ion and the curent grid node:  ***/
  long int i_tmp_ions, j_tmp_ions;
/*** the index of the grid node where the current ion is placed:  ***/
  long int i_sod_point;
/*** the energies of the ions (at the moment of each ion's placement):  ***/
  float E_ions[MAXIONS];


/************************/
/*** Grid parameters. ***/
/***                  ***/
/*** grid margin around the macromolecule:  ***/
/*** the dimensions of the grid box:  ***/
  long int N_box_0, N_box_1, N_box_2;
/*** the total number of nodes in the grid:  ***/
  long int N_box_tot;
/*** the exclusion map for the grid (where 1's mark the allowed ***/
/*** grid nodes and 0's mark the grid nodes excluded from the   ***/
/*** consideration because of their proximity to an atom or an  ***/
/*** already placed ion):       ***/
  unsigned char *I_excl;
/*** the exclusion map running pointer:  ***/
  unsigned char *I_cur;
/*** the number of grid nodes left after the exclusion:  ***/
  long int N_Ener;
/*** the array of the energies of the grid nodes:   ***/
  float *Ener;
/*** running pointer and storage variable for the energy array:  ***/
  float Ener_cur, *E_cur;
/*** the array of pre-computed inverse distances between the grid nodes:  ***/
  static float *SQRTs;
/*** the size of that array ***/
  long int N_SQRTs;
/*** the index of the distance between two current grid points in that array ***/
  long int m;
/*** the array of pre-computed squares of distances between the grid layers:  ***/
  long int SQs[2*MAXGRID+1];
/*** the size of that array ***/
  long int N_SQs;


/**********************/
/*** Loop counters. ***/
/***                ***/
  long int i, k;
  long int ii, jj, kk;


/*************************************/
/***  Auxiliary (dummy) variables. ***/
/***                               ***/
  char s[100];
  float f_dum;
  long int i_dum, ii_dum, jj_dum, kk_dum;


/********************************/
/*** THE BODY OF THE PROGRAM. ***/
/********************************/

/*******************************/
/*** The signature printout. ***/
/*******************************/

  DosPrintf(stdout, "\n");
  DosPrintf(stdout, "=========================================================================\n");
  DosPrintf(stdout, " SODIUM: A program for arranging ions around biological macromolecules.  \n");
  DosPrintf(stdout, " VERSION: 1.5.1.                                                           \n");
  DosPrintf(stdout, " AUTHOR:  Alexander Balaeff.                                             \n");
  DosPrintf(stdout, " (C) Copyright 2000 The Theoretical Biophysics Group, Beckman Institute, \n");
  DosPrintf(stdout, "     and The Board of Trustees of the University of Illinois.            \n");
  DosPrintf(stdout, "=========================================================================\n");
  DosPrintf(stdout, "\n");

/************************************************/
/*** Readout of the configuration parameters. ***/
/************************************************/

  DosPrintf(stdout, "CONFIGURATION:\n\n");
  DosPrintf(stdout, "  %ld ions will be built within the %f A border\n", N_ions, border_width);
  DosPrintf(stdout, "  around the molecule %s,\n", pdb_name);
  DosPrintf(stdout, "  not closer than %f A to the molecule and\n", excl_atom_rad);
  DosPrintf(stdout, "  not closer than %f A to each other,\n", excl_ion_rad);
  DosPrintf(stdout, "  on the cubic grid with %f A cell size,\n", delta);
  DosPrintf(stdout, "  the results being output to the file %s in the PDB format.\n\n", pdb_out_name);


/*******************************************************/
/*** Readout of protein/DNA coordinates and charges. ***/
/*******************************************************/

  Atm = BegAtm;
  for (i = 0; i < TotalAtm; ++i) {
	x_at[i] = Atm -> x;
	y_at[i] = Atm -> y;
	z_at[i] = Atm -> z;
	q_at[i] = Atm -> Charge;
	if( i==0 || x_max < x_at[i] ) { x_max=x_at[i]; }
	if( i==0 || x_min > x_at[i] ) { x_min=x_at[i]; }
	if( i==0 || y_max < y_at[i] ) { y_max=y_at[i]; }
	if( i==0 || y_min > y_at[i] ) { y_min=y_at[i]; }
	if( i==0 || z_max < z_at[i] ) { z_max=z_at[i]; }
	if( i==0 || z_min > z_at[i] ) { z_min=z_at[i]; }
	Atm = Atm -> Ptr;
  }
  N_atoms = TotalAtm;

  DosPrintf(stdout, "The molecule dimensions are: \n");
  DosPrintf(stdout, "%f -- %f\n",x_min,x_max);
  DosPrintf(stdout, "%f -- %f\n",y_min,y_max);
  DosPrintf(stdout, "%f -- %f\n",z_min,z_max);
  DosPrintf(stdout, "\n");


/******************************/
/*** The setup of the grid. ***/
/******************************/

/** Finding the size of the grid box. **/
  N_box_0 = (int)( (x_max - x_min + 2.*border_width) / delta ) + 1 ;
  N_box_1 = (int)( (y_max - y_min + 2.*border_width) / delta ) + 1 ;
  N_box_2 = (int)( (z_max - z_min + 2.*border_width) / delta ) + 1 ;
  N_box_tot = N_box_0*N_box_1*N_box_2;
  DosPrintf(stdout, "The grid dimensions are:\n");
  DosPrintf(stdout, "    %ld  X  %ld  X  %ld  =  %ld nodes\n\n",N_box_0,N_box_1,N_box_2,N_box_tot);


/** Tying up the atoms to the grid nodes. **/
  for( i=0; i<N_atoms; i++) {
    x_at[i] = (x_at[i] - x_min + border_width) / delta;
    y_at[i] = (y_at[i] - y_min + border_width) / delta;
    z_at[i] = (z_at[i] - z_min + border_width) / delta;

    /*    ii_at[i] = nint( x_at[i] );
	    jj_at[i] = nint( y_at[i] );
	    kk_at[i] = nint( z_at[i] ); */
    ii_at[i] = rint( x_at[i] );
    jj_at[i] = rint( y_at[i] );
    kk_at[i] = rint( z_at[i] );
  }


/** Pre-computing the inverse distances between the grid nodes. **/
  N_SQRTs = N_box_0*N_box_0 + N_box_1*N_box_1 + N_box_2*N_box_2 + 1;
  if( (SQRTs=calloc(N_SQRTs,sizeof(float))) == NULL ) {
    DosPrintf(stdout, "Memory insufficient for the square roots array (%ld elements required).\n",N_SQRTs);
    return ERR_INSUF_MEM_SQRTS;
  }
  SQRTs[0] = 0;
  for( k=1; k<N_SQRTs; k++ )  SQRTs[k] = 1./sqrt( (float)k );


/** Pre-computing the squares of distances between the grid layers. **/
  N_SQs = N_box_0;
  if( N_box_1 > N_SQs ) N_SQs = N_box_1;
  if( N_box_2 > N_SQs ) N_SQs = N_box_2;
  for( k=-N_SQs; k<=N_SQs; k++ )  SQs[k+N_SQs] = k*k;


/** Setting up the exclusion array. **/
  DosPrintf(stdout, "Memory required for the exclusion map: %ld Mb. \n", N_box_tot/(1024*1024));

  if( (I_excl=calloc( N_box_tot,1)) != NULL )
    DosPrintf(stdout, "Memory for the exclusion map has been successfully allocated.\n\n");
  else {
    DosPrintf(stdout, "Memory insufficient.\n");
    return ERR_INSUF_MEM_EXCLMAP;
  }

  DosPrintf(stdout, "Now building the exclusion map...\n");
  for( k=0; k<N_box_tot; k++ )  I_excl[k] = 1;


/*** Excluding the grid nodes within excl_atom_rad of protein/DNA. ***/
  N_Ener = N_box_tot;
  excl_atom_rad /= delta;

  for( i=0; i<N_atoms; i++) {
    N_Ener -= excl_around( ii_at[i], jj_at[i], kk_at[i], excl_atom_rad,
				   I_excl, N_box_0, N_box_1, N_box_2);
  }
  DosPrintf(stdout, "Exclusion map has been succesfully built.\n");
  DosPrintf(stdout, "Grid nodes left after the exclusion:  %ld. \n\n", N_Ener);


/***********************************/
/*** Initial energy computation. ***/
/***********************************/

/*** Memory arrangement for the energy array. ***/
  DosPrintf(stdout, "Memory required for the energy array: %ld Mb.\n",
	   N_box_tot*sizeof(float)/(1024*1024));

  if( (Ener=calloc(N_box_tot,sizeof(float))) != NULL )
    DosPrintf(stdout, "Memory for the energy array has been successfully allocated.\n\n");
  else {
    DosPrintf(stdout, "Memory insufficient.\n");
    return ERR_INSUF_MEM_ENERGY;
  }


/*** Looping over the whole grid and computing the energy for each node. ***/
  DosPrintf(stdout, "Now computing the energy at the grid nodes...\n");

  for( I_cur=I_excl, E_cur=Ener, ii_dum=N_SQs,
	   ii=0; ii<N_box_0; ii++, ii_dum-- ) {
    /* DosPrintf(stdout, "   Layer %ld\n",ii); */
    for( i_tmp=i_tmp_at_0, i=0; i<N_atoms; i++, i_tmp++)  {
	i_dum = ii_dum + ii_at[i] ;
	*i_tmp = SQs[i_dum];
    }

  for( jj_dum=N_SQs, jj=0; jj<N_box_1; jj++, jj_dum-- ) {
    /*    DosPrintf(stdout, "   Line %ld\n",jj); */
    for( i_tmp=i_tmp_at_0, j_tmp=i_tmp_at_1,
	     i=0; i<N_atoms; i++,
	     i_tmp++, j_tmp++ )  {
	i_dum = jj_dum + jj_at[i] ;
	*j_tmp = SQs[i_dum] + (*i_tmp);
    }

    for( kk_dum=N_SQs, kk=0; kk<N_box_2; kk++,
	   kk_dum--, I_cur++, E_cur++ ) {

	if( *I_cur ) {
	  /* The energy is computed only if the current node is not excluded. */
	  for( Ener_cur=0, q_cur=q_at, i_tmp=i_tmp_at_1,
		   i=0; i<N_atoms; i++,
		   q_cur++, i_tmp++ ) {
	    i_dum = kk_dum + kk_at[i] ;
	    m = SQs[i_dum] + (*i_tmp);
	    Ener_cur += (*q_cur)*SQRTs[m];
	  }
	  *E_cur = Ener_cur;
	}

    }}}

  DosPrintf(stdout, "Energy at the grid nodes has been successfully computed.\n\n");


/***************************************************/
/***  Place N_ions ions, one by one.             ***/
/***  Recompute the grid energy after each step. ***/
/***************************************************/

  excl_ion_rad /= delta;

  DosPrintf(stdout, "Now we place %ld ions.\n",N_ions);

  for(k=0; k<N_ions; k++ ) {

    DosPrintf(stdout, "\nPlacing the %ld-th ion...\n",k+1);

    i_sod_point = min_element(Ener,N_box_tot,I_excl);
    E_ions[k] = Ener[i_sod_point];

    indices( i_sod_point, N_box_0, N_box_1, N_box_2,
		 ii_ions+k, jj_ions+k, kk_ions+k);
    DosPrintf(stdout, "...the placement point (E=%f) is no.%ld ( %ld, %ld, %ld)...\n",
	     E_ions[k],i_sod_point,ii_ions[k],jj_ions[k],kk_ions[k]);

    DosPrintf(stdout, "...excluding close grid nodes...\n");

    N_Ener -= excl_around( ii_ions[k], jj_ions[k], kk_ions[k], excl_ion_rad,
				   I_excl, N_box_0, N_box_1, N_box_2);

    DosPrintf(stdout, "...%ld grid nodes left after the exclusion...\n", N_Ener);

    DosPrintf(stdout, "...re-computing the grid energy...\n");

    for( I_cur=I_excl, E_cur=Ener, ii_dum=N_SQs,
	     ii=0; ii<N_box_0; ii++, ii_dum-- ) {
	i_dum = ii_dum + ii_ions[k];
	i_tmp_ions = SQs[i_dum] ;

	for( jj_dum=N_SQs, jj=0; jj<N_box_1; jj++, jj_dum-- ) {
	  i_dum = jj_dum + jj_ions[k] ;
	  j_tmp_ions = SQs[i_dum] + i_tmp_ions;

	  for( kk_dum=N_SQs, kk=0; kk<N_box_2; kk++,
		   kk_dum--, I_cur++, E_cur++ ) {

	    if( *I_cur ) {
		i_dum = kk_dum + kk_ions[k] ;
		m = SQs[i_dum] + j_tmp_ions;
		*E_cur += SQRTs[m];
	    }

	  }}}

    DosPrintf(stdout, "...done placing the %ld-th ion.\n",k+1);
  }


/*****************************************/
/***  Prinitng the ions' coordinates.  ***/
/*****************************************/

/*
  for( k=0; k<N_ions; k++ ) {

    x_ions[k] = delta*(float)ii_ions[k] - border_width + x_min;
    y_ions[k] = delta*(float)jj_ions[k] - border_width + y_min;
    z_ions[k] = delta*(float)kk_ions[k] - border_width + z_min;

    fDosPrintf(stdout, pdb_outfile,
		"ATOM%7d SOD  SOD %5d    %8.3f%8.3f%8.3f  1.00   .00      SOD\n",
		k+1, k+1, x_ions[k], y_ions[k], z_ions[k]);
  }

  fDosPrintf(stdout, pdb_outfile, "END\n");
  fclose(pdb_outfile);
*/

/******************************/
/*** Finishing the program. ***/
/******************************/

  DosPrintf(stdout, "\n");
  DosPrintf(stdout, "=========================================\n");
  DosPrintf(stdout, " The ions have been successfully placed. \n");
  DosPrintf(stdout, " Thank you for using SODIUM!             \n");
  DosPrintf(stdout, "=========================================\n");

  return ERR_NO_ERROR;

}


/************************/
/*** THE SUBROUTINES. ***/
/************************/

/******************************************************************************/
/*** This subroutine returs the index of the smallest element of the array. ***/
/******************************************************************************/

long int min_element (float *Ener, long int size, unsigned char *excl_map )
{
  long int i, i_min;
  float V;

  V = Ener[0];
  i_min = 0;
  for( i=0; i<size; i++ ) {
    if( excl_map[i] ) {
	if( Ener[i] < V ) {
	  V = Ener[i];
	  i_min = i;
	}
    }
  }
  return i_min;
}


/********************************************************************************/
/*** This subroutine computes the 3-D indices of the ind-th node of the grid. ***/
/*** For the grid box with dimensions N_box_0, N_box_1, N_box_2, each of the  ***/
/*** N_box_0 layers has a size of N_box_1*N_box_2, and each point's index is: ***/
/***   index(i,j,k) = N_box_1*N_box_2*i + N_box_2*j + k .                     ***/
/********************************************************************************/

int indices( long int ind,
		 long int N_box_0, long int N_box_1, long int N_box_2,
		 long int *i_pt, long int *j_pt, long int *k_pt)
{
  long int i_dum;

  i_dum = ind;
  *i_pt = i_dum/(N_box_1*N_box_2);
  i_dum -= (*i_pt)*(N_box_1*N_box_2);
  *j_pt = i_dum/N_box_2;
  i_dum -= (*j_pt)*N_box_2;
  *k_pt = i_dum;

  return ind;
}

/***********************************************************/
/*** This subroutine excludes all grid nodes closer than ***/
/*** the raidus R to the grid node (i_pt, j_pt, k_pt).   ***/
/*** Returned is the number of nodes, excluded this way. ***/
/***********************************************************/

int excl_around( long int i_pt, long int j_pt, long int k_pt, float excl_rad,
		     unsigned char *I_excl,
		     long int N_box_0, long int N_box_1, long int N_box_2)
{
  long int ii, jj, kk;
  long int i_max, j_max, k_max;
  long int i_min, j_min, k_min;
  long int ind, ind0, ind1, i_dum;
  long int N_excl = 0;
  float d0, d1, d2, excl_rad_sq;

  excl_rad_sq = excl_rad*excl_rad;

  i_max = i_pt + excl_rad;  if( i_max >= N_box_0 )  i_max = N_box_0 - 1;
  i_min = i_pt - excl_rad;  if( i_min < 0 )  i_min = 0;
  j_max = j_pt + excl_rad;  if( j_max >= N_box_1 )  j_max = N_box_1 - 1;
  j_min = j_pt - excl_rad;  if( j_min < 0 )  j_min = 0;
  k_max = k_pt + excl_rad;  if( k_max >= N_box_2 )  k_max = N_box_2 - 1;
  k_min = k_pt - excl_rad;  if( k_min < 0 )  k_min = 0;

  for( ind0 = i_min*N_box_1, ii = i_min;
	 ii <= i_max;
	 ind0 += N_box_1, ii++ ) {

    d0 = ii - i_pt ;
    d0 *= d0;

  for( ind1 = (ind0 + j_min)*N_box_2, jj = j_min;
	 jj <= j_max;
	 ind1 += N_box_2, jj++ ) {

    d1 = jj - j_pt ;
    d1 *= d1;

  for( ind = ind1 + k_min, kk=k_min;
	 kk<=k_max;
	 ind++, kk++ ) {

    if( I_excl[ind] ) {

	d2 = kk - k_pt ;
	d2 *= d2;

	if( d0 + d1 + d2 <= excl_rad_sq ) {
	  I_excl[ind] = 0;
	  N_excl++;
	}
    }

  }}}

  return N_excl;
}


/*****************************************************************/
/*** The following subroutines read configuration parameters   ***/
/*** from the given configuration line, if the required format ***/
/***       KEYWORD  PARAMETER                                  ***/
/*** is matched.                                               ***/
/*****************************************************************/

int read_config_line_int(char *line, const char *keyword, long int *num)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;

  sprintf(read_format,"%s %%ld",keyword);

  if( sscanf(sub,read_format,num) != 1 ) {
    DosPrintf(stdout, "Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

int read_config_line_float(char *line, const char *keyword, float *val)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;

  sprintf(read_format,"%s %%f",keyword);

  if( sscanf(sub,read_format,val) != 1 ) {
    DosPrintf(stdout, "Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

int read_config_line_string(char *line, const char *keyword, char *word)
{
  char *sub, read_format[100];

  if( (sub=strstr(line,keyword)) == NULL )  return 0;

  sprintf(read_format,"%s %%s",keyword);

  if( sscanf(sub,read_format,word) != 1 ) {
    DosPrintf(stdout, "Wrong configuration line:\n%s\n",line);
    return 0;
  }
  else  return 1;

}

/**********************************************************************************/
