
/*************************************************
****   VEGA - Molecular surface calculation   ****
**** Copyright 1996-2003, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <math.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"
#include "surface.h"


/**** Local variables ****/

static VG_ULONG        GlobNu = 0;


/**** Dot calculation of a sphere with radius = 1 ****/

XYZ *CalcSfUn(VG_ULONG Points, float MaxRad, VG_ULONG *Nu)
{
  double                Fi, Fj, Nequat, Z;
  register double       Xy;
  register XYZ          *SfPoint;
  VG_ULONG              j, k, Nhor, Nvert;
  XYZ                   *SfUn;

  Points *= (VG_ULONG)((double)4.0 * (double)PI * (double)Quad(MaxRad));
  if ((SfUn = (XYZ *)Alloca(Points * sizeof(XYZ)))) {
    Nequat = (double)sqrt(PI * Points);  /* Linear density                 */
                                         /* Nequat = 2 * PI * SQR(D) * RAD */
    Nvert   = (VG_ULONG)(Nequat / 2.0);
    *Nu     = 0;
    SfPoint = SfUn;
    for(k = 0; (k <= (VG_ULONG)Nvert) && (*Nu < Points); ++k) {
      Fi   = (double)(PI * (double)k) / (double)Nvert;
      Z    = (double)cos(Fi);
      Xy   = (double)sin(Fi);
      Nhor = (VG_ULONG)(Xy * Nequat);
      if (Nhor < 1) Nhor = 1;

      for(j = 0;(j <= (VG_ULONG)(Nhor - 1)) && (*Nu < Points); ++j) {
        Fj = (2 * PI * (double)j) / (double)Nhor;
        SfPoint -> x = (float)((double)cos(Fj) * (double)Xy);
        SfPoint -> y = (float)((double)sin(Fj) * (double)Xy);
        SfPoint -> z = (float)Z;
        ++*Nu;
        ++SfPoint;
      } /* End of loop */
    } /* End of loop */
  }
  GlobNu = *Nu;

  return SfUn;
}


/**** Calculate surface area and PSA ****/

float CalcSrfArea(ATOMO *InizAtm, float *InizRad, XYZ *InizSfUn,
                  VG_ULONG Nu, TRJPAR *Par, float MaxRad, float *AreaP,
                  VG_ULONG Flags)
{
  VG_BOOL       Chk;
  ATOMO         *Atm1, *Atm2;
  float         *Rad1, *Rad2;
  float         Xp, Yp, Zp;
  VG_ULONG      SfDots, SfDotsPol;
  VG_ULONG      j, k;
  XYZ           *SfUn;

  float         Ap   = 0.0;
  float         Area = 0.0;

  Rad1 = InizRad;
  for(Atm1 = InizAtm; (Atm1); Atm1 = Atm1 -> Ptr) {

    /**** Skip waters (for trajectory analysis) ****/

    if (((Flags & SRF_FLG_NOWAT) &&
#ifdef LITTLE_ENDIAN
       (Atm1 -> ResName.L == 0x00484F48)) ||
#else
       (Atm1 -> ResName.L == 0x484F4800)) ||
#endif
      ((Flags & SRF_FLG_SELONLY) && (!Atm1 -> Active)) ||
      (Atm1 -> Flags & VG_ATMF_CENTROID)) {
      ++Rad1;
      continue;
    }

    SfDots    = 0;
    SfDotsPol = 0;
    SfUn      = InizSfUn;
    if ((Par) && (Par -> Num)) {
      for(j = 0; (Atm1 -> Num != Par -> Atm[j].Num) && (j < (VG_ULONG)Par -> Num); ++ j);
      if (j == (VG_ULONG)Par -> Num) {
        ++Rad1;
        continue;
      }
    }

    for(k = 0; k < Nu; ++k) {
      Rad2 = InizRad;
      Chk  = TRUE;
      Xp   = (SfUn -> x * *Rad1) + Atm1 -> x;
      Yp   = (SfUn -> y * *Rad1) + Atm1 -> y;
      Zp   = (SfUn -> z * *Rad1) + Atm1 -> z;
      for(Atm2 = InizAtm; (Atm2) && (Chk); Atm2 = Atm2 -> Ptr) {

       /**** Skip waters & centroids (for trajectory analysis) ****/

       if ((Flags & SRF_FLG_NOWAT) &&
#ifdef LITTLE_ENDIAN
           (Atm2 -> ResName.L == 0x00484F48) ||
#else
           (Atm2 -> ResName.L == 0x484F4800) ||
#endif
           (Atm2 -> Flags & VG_ATMF_CENTROID)) {
          ++Rad2;
          continue;
        }

        if ((Atm1 != Atm2) &&
            ((Quad(Atm2 -> x - Xp) +
              Quad(Atm2 -> y - Yp) +
              Quad(Atm2 -> z - Zp)) < Quad(*Rad2)))
          Chk = FALSE;

        ++Rad2;
      } /* End of atom loop */
      if (Chk) {
        if ((AreaP) && (CalcSrfPsa(Atm1))) ++SfDotsPol;
        ++SfDots;
      }
      ++SfUn;
    }  /* End of dot loop */
    Ap   += (4.0 * PI * Quad(*Rad1) / Nu) * SfDotsPol;
    Area += (4.0 * PI * Quad(*Rad1) / Nu) * SfDots;
    ++Rad1;
  } /* End of atom loop */

  if (AreaP) *AreaP = Ap;
  return (float)Area;
}


/**** Check if the atom is polar for PSA calculation ****/

VG_BOOL CalcSrfPsa(register ATOMO *Atm)
{
  VG_BOOL       k;

  if (!Atm -> Elem.C[1]) {
    switch(*Atm -> Elem.C) {
    case 'C':
      k = FALSE;
      break;

    case 'O':
    case 'S':
    case 'N':
    case 'P':
      k = TRUE;
      break;
    case 'H':
      k = TRUE;
      if (Atm -> Conn[0]) {
#ifdef LITTLE_ENDIAN
        if (Atm -> Conn[0] -> Elem.S == 0x43) {
#else
        if (Atm -> Conn[0] -> Elem.S == 0x4300) {
#endif
          k = FALSE;
          break;
        }
      }
      break;
    }
  } else k = FALSE;

  return k;
}


/**** Allocate a surface dot ****/

VG_SURFACE *SrfAlloc(VG_SURFACE **InizSrf, VG_ULONG *TotDots)
{
  register VG_SURFACE  *Dot;
  static VG_SURFACE    *PrecDot;

  if ((Dot = (VG_SURFACE *)SOALLOC(sizeof(VG_SURFACE)))) {
    if (!*TotDots) {
      *InizSrf = Dot;
    } else {
      PrecDot -> Next = Dot;
      Dot     -> Prev = PrecDot;
    }
    ++*TotDots;
    PrecDot           = Dot;                 /* Bind the list           */
    Dot -> AtmNum     = 0;                   /* Atom number             */
    Dot -> Val        = 0.0;                 /* Default property value  */
    Dot -> Color[0]   = 0;                   /* Default dot color       */
    Dot -> Color[1]   = 0;
    Dot -> Color[2]   = 255;
    Dot -> Next       = NULL;                /* Pointer to next element */
  } else {
    if (*TotDots) {
      SrfFree(*InizSrf);
      *InizSrf = NULL;
    }
    CatErr(MSG_ERR_UTILS_OUTOFMEM);
  }

  return Dot;
}


/**** Calculates the surface accesible to solvent (SAS) ****/

VG_SURFACE *SrfCalc(VG_SURFACE **LastDot, VG_ULONG *TotDot, ATOMO *InizAtm,
                    VG_ULONG TotAtm, VG_ULONG Points, float ProRad,
                    VG_UWORD Form, float *Contrib, VG_ULONG Flags)
{
  ATOMO                 *Atm, *Atm2, **AtmTab;
  float                 X, Y, Z, Rad;
  float                 X2, Y2, Z2;
  float                 CutOff;
  VG_BOOL               Chk;
  VG_SURFACE            *Dot;
  VG_ULONG              k, Nu, SfDots, SfDotsPol;
  XYZ                   *SfPoint, *SfUn;

#ifdef VLOGP
  float                 Xyp      = 0;
  float                 Xyn      = 0;
#endif

  float                 Area     = 0.0;                 /* Total area */
  float                 AreaP    = 0.0;                 /* Polar area */
  float                 MaxRad   = InizAtm -> Rad;
  float                 ValMax   = -VG_INFINITE;
  float                 ValMin   = VG_INFINITE;
  VG_BOOL               Ret      = TRUE;
  VG_SURFACE            *InizSrf = NULL;

 switch (Form & 0xff00) {
#ifndef VLOGP
  case SRF_PROP_VLOGP:
    PrintVlogPErr();
    return NULL;
#endif

  case SRF_PROP_ILM:
    if (!RenameWat(InizAtm)) {
      CatErr(MSG_ERR_SURFACE_ILM);
      return NULL;
    } else CutOff = IlmClacCutOff(InizAtm);
    break;
  } /* End of switch */

  /**** Find the max radius mesure in the molecule ****/

  for(Atm = InizAtm;(Ret) && (Atm); Atm = Atm -> Ptr)
    if (MaxRad < Atm -> Rad) MaxRad = Atm -> Rad;

  MaxRad += ProRad;

  if ((SfUn = CalcSfUn(Points, MaxRad, &Nu))) {
    for(Atm = InizAtm;(Ret) && (Atm); Atm = Atm -> Ptr) {

      /**** Skip the waters if needed ****/

      if ((((Form & 0xff00) == SRF_PROP_ILM) &&
#ifdef LITTLE_ENDIAN
          (Atm -> ResName.L == 0x00484F48)) ||
#else
          (Atm -> ResName.L == 0x484F4800)) ||
#endif
         ((Flags & SRF_FLG_SELONLY) && (!Atm -> Active))) {
        continue;
      }

      SfDots    = 0;
      SfDotsPol = 0;
      SfPoint   = SfUn;
      Rad       = Atm -> Rad + ProRad;
      for(k = 0;(Ret) && (k < Nu);++k) {
        X   = Atm -> x + SfPoint -> x * Rad;
        Y   = Atm -> y + SfPoint -> y * Rad;
        Z   = Atm -> z + SfPoint -> z * Rad;
        Chk = TRUE;

        for(Atm2 = InizAtm;(Chk) && (Atm2); Atm2 = Atm2 -> Ptr) {

          /**** Skip waters (if needed) & centroids ****/

          if ((((Form & 0xff00) == SRF_PROP_ILM) &&
#ifdef LITTLE_ENDIAN
              (Atm2 -> ResName.L == 0x00484F48)) ||
#else
              (Atm2 -> ResName.L == 0x484F4800)) ||
#endif
              (Atm2 -> Flags & VG_ATMF_CENTROID)) {
            continue;
          }

          if (Atm2 != Atm) {
          if ((Quad(Atm2 -> x - X) +
               Quad(Atm2 -> y - Y) +
               Quad(Atm2 -> z - Z)) < Quad(Atm2 -> Rad + ProRad))

            Chk = FALSE;
          }
        } /* End of Atm2 loop */

        if (Chk)  {
          ++SfDots;
          if ((Dot = SrfAlloc(&InizSrf, TotDot)) != NULL) {
            Dot -> AtmNum = Atm -> Num;
            Dot -> x      = X;
            Dot -> y      = Y;
            Dot -> z      = Z;
            switch(Form & 0xff00) {
#ifdef VLOGP
            case SRF_PROP_VLOGP:        /* MLP surface */
              Dot -> Val = CalcSrfLogP(InizAtm, Contrib, X, Y, Z, NULL, Flags);
              if (Dot -> Val > 0) Xyp += Dot -> Val;
              else Xyn += Dot -> Val;
              break;
#endif
            case SRF_PROP_ILM:          /* ILM surface */
              Dot -> Val = CalcSrfIlm(InizAtm, X, Y, Z, CutOff);
              break;

            case SRF_PROP_MEP:          /* MEP surface */
              Dot -> Val = CalcSrfMep(InizAtm, X, Y, Z, Flags);
              break;

            case SRF_PROP_PSA:          /* PSA surface */
              if (CalcSrfPsa(Atm)) {
                Dot -> Val = 1.0;
                ++SfDotsPol;
              }
              break;

            default:                    /* VdW surface */
              Dot -> Val = 1.0;
            } /* End of switch */
            if (Dot -> Val > ValMax) ValMax = Dot -> Val;
            if (Dot -> Val < ValMin) ValMin = Dot -> Val;
          } else {
            SrfFree(InizSrf);
            InizSrf = NULL;
            *TotDot = 0;
            Ret     = FALSE;
          }
        }
        ++SfPoint;
      } /* End of dot loop */
      Area  += (4.0 * PI * Quad(Rad) / Nu) * SfDots;
      AreaP += (4.0 * PI * Quad(Rad) / Nu) * SfDotsPol;
    } /* End of Atm loop */

    if (Ret) {
      if (LastDot) *LastDot = Dot;
#ifdef __VG_OPENGL
      if (ReturnStr) {
        sprintf(ReturnStr, "%.8f %.8f %.8f %.8f", Area, SQR(Area / PI),
                ValMin, ValMax);
      }
#endif
      LocPrintf(stdout, "  ");
      if ((Form & 0xff00) == SRF_PROP_PSA) {
        CatPrintf(stdout, MSG_INFO_PSA, AreaP, Area - AreaP);
#ifdef __VG_OPENGL
        if (ReturnStr) {
          sprintf(ReturnStr + strlen(ReturnStr), " %.8f %.8f", AreaP, Area - AreaP);
        }
#endif
      } else {
        CatPrintf(stdout, MSG_INFO_SURFAREA, ProRad, Area, SQR(Area / PI));
      }

      if ((Form & 0xff00) != SRF_PROP_VDW) {
        CatPrintf(stdout, MSG_SURFACE_RANGES, ValMin, ValMax);
#ifdef VLOGP
        if ((Form & 0xff00) == SRF_PROP_VLOGP) {
          PrintLogP(Xyp, Xyn);
#  ifdef __VG_OPENGL
          if (ReturnStr) {
            sprintf(ReturnStr + strlen(ReturnStr), " %.8f",
                    2.86E-3 * Xyp + 1.52E-3 * Xyn - 0.1);
          }
#  endif
        }
#endif
      }
    }

    FREE(SfUn);
  }

  return InizSrf;
}


/**** Close the surface ****/

void SrfClose(void)
{
  SrfFree(BegSrf);
  BegSrf   = NULL;
  TotalSrf = 0;
}


/**** Color surface by values ****/

void SrfColorByVal(VG_SURFACE *InizSrf)
{
  float                 Step, Val1x, Val2x, Val3x;
  register VG_SURFACE   *Dot;

  float                 MaxVal = InizSrf -> Val;
  float                 MinVal = InizSrf -> Val;

  /**** Find max and min values ****/

  for(Dot = InizSrf -> Next; Dot; Dot = Dot -> Next) {
    if (Dot -> Val > MaxVal) MaxVal = Dot -> Val;
    if (Dot -> Val < MinVal) MinVal = Dot -> Val;
  }

  Step = (MaxVal - MinVal) / 4.0;
  if (Step == 0.0) Step = 1.0; /* To avoid division by zero */

  Val1x = MinVal + Step;
  Val2x = Val1x  + Step;
  Val3x = Val2x  + Step;

  /**** Set the color of each dot ****/

  for(Dot = InizSrf -> Next; Dot; Dot = Dot -> Next) {
    if (Dot -> Val < Val1x) {
      Dot -> Color[0] = 0;
      Dot -> Color[1] = (VG_UBYTE)(((Dot -> Val - Val1x) / Step) * 255.0);
      Dot -> Color[2] = 255;
    } else if (Dot -> Val < Val2x) {
      Dot -> Color[0] = 0;
      Dot -> Color[1] = 255;
      Dot -> Color[2] = (VG_UBYTE)(((Val2x - Dot -> Val) / Step) * 255.0);
    } else if (Dot -> Val < Val3x) {
      Dot -> Color[0] = (VG_UBYTE)(((Dot -> Val - Val2x) / Step) * 255.0);
      Dot -> Color[1] = 255;
      Dot -> Color[2] = 0;
    } else {
      Dot -> Color[0] = 255;
      Dot -> Color[1] = (VG_UBYTE)(((MaxVal - Dot -> Val) / Step) * 255.0);
      Dot -> Color[2] = 0;
    }
  } /* End of dot for */
}


/**** Extract the surface name from file path ****/

void SrfName(char *Src, char *Dest)
{
  register char         *Ptr1;

  /**** Map name definition ****/

  for(Ptr1 = Dest; *Src; ++Ptr1, ++Src) {
    if (*Src == ' ') *Ptr1 = '_';
    else *Ptr1 = *Src;
  }
  *Ptr1 = 0;

  for(Ptr1 = Dest + strlen(Dest); Ptr1 != Dest; --Ptr1) {
    if (*Ptr1 == '.') {
      *Ptr1 = 0;
      break;
    }
  } /* End of for */
}


/**** Initialize the PSA calculation ****/

ATOMO **SrfPsaInitCalc(register ATOMO *Atm1, VG_ULONG TotAtm)
{
  ATOMO                 **AtmTbl;
  register ATOMO        **Atm2;

  if ((AtmTbl = (ATOMO **)Alloca(sizeof(ATOMO *) * TotAtm))) {
    Atm2    = AtmTbl;
    while(Atm1) {
      *Atm2 = Atm1;
      Atm1 = Atm1 -> Ptr;
      ++Atm2;
    } /* End of while loop */
  }

  return AtmTbl;
}


/**** Calculate the polar surface area ****/

void SrfPsaCalc(VG_SURFACE *Srf, ATOMO **AtmTbl, ATOMO *InizAtm,
                VG_ULONG TotAtm, float *AreaP, float *AreaAP)
{
  ATOMO         *Atm1, **Atm2;
  VG_ULONG      AtmDots;
  VG_BOOL       k;

  VG_ULONG      PrecAtm = 0;

  *AreaAP = 0.0;
  *AreaP  = 0.0;

  while(Srf) {
    if ((Srf -> AtmNum != PrecAtm) || (!Srf -> Next)) {
      if(!Srf -> Next) ++AtmDots;
      if (PrecAtm) {
        if (k)
          *AreaP += (4.0 * PI * Quad(Atm1 -> Rad + Prefs.SAS_PROBERAD) / GlobNu) * (float)AtmDots;
        else
          *AreaAP += (4.0 * PI * Quad(Atm1 -> Rad + Prefs.SAS_PROBERAD) / GlobNu) * (float)AtmDots;
      }
      AtmDots = 0;
      PrecAtm = Srf -> AtmNum;
      Atm1 = AtmTbl[PrecAtm - 1];
      k = CalcSrfPsa(Atm1);
    }
    ++AtmDots;
    Srf = Srf -> Next;
  } /* Surface dot loop */
}


/**** Show the surface progress ****/

void SrfPrintProg(VG_UWORD Form)
{
  VG_ULONG      Str;

  switch(Form & 0xff00) {
  case SRF_PROP_ILM:
    Str = MSG_SURFACE_BUILDILM;
    break;

  case SRF_PROP_MEP:
    Str = MSG_SURFACE_BUILDMEP;
    break;

  case SRF_PROP_PSA:
    Str = MSG_SURFACE_BUILDPSA;
    break;
  case SRF_PROP_VDW:
    Str = MSG_SURFACE_BUILDVDW;
    break;
#ifdef VLOGP
  case SRF_PROP_VLOGP:
    Str = MSG_SURFACE_BUILDVLOGP;
    break;
#endif
  } /* End of switch */

  PrintProg(Str);
}

/**** Save the surface in specified format ****/

VG_BOOL SrfSave(FILE *FH, char *File, VG_SURFACE *InizSrf, VG_ULONG TotSrf,
                ATOMO *InizAtm, VG_UWORD Form)
{
  VG_BOOL               Ret;

    switch(Form & 0x00ff) {

    case SRF_CSV:
      Ret = SrfCsvSave(FH, InizSrf);
      break;

    case SRF_IFF:
      Ret = IFFSave(FH, BegAtm, TotalAtm);
      break;

    case SRF_INSIGHT:
      Ret = SrfInsightSave(FH, File, InizSrf, InizAtm);
      break;

    case SRF_QUANTA:
      Ret = SrfQuantaSave(FH, File, InizSrf, InizAtm);
      break;

    case SRF_RAW:
      Ret = SrfRawSave(FH, InizSrf);
      break;

    case SRF_VRML_SOLID:
      Ret = SrfVrmlSave(FH, InizSrf, TRUE);
      break;

    case SRF_VRML_POINTS:
      Ret = SrfVrmlSave(FH, InizSrf, FALSE);
      break;
    } /* End of switch */

  return Ret;
}


/**** Save the surface in Raw format ****/

VG_BOOL SrfRawSave(FILE *FH, register VG_SURFACE *Dot)
{
  VG_BOOL      Ret = TRUE;

  while((Dot) && (Ret)) {
    if (fwrite(((VG_UBYTE *)Dot) + sizeof(VG_SURFACE *),
                sizeof(VG_SURFACE) - sizeof(VG_SURFACE *), 1, FH) != 1)
      Ret = PrintDosErr();
    Dot = Dot -> Next;
  } /* End of while */

  return Ret;
}


/**** Save the surface in VRML format ****/

VG_BOOL SrfVrmlSave(FILE *FH, register VG_SURFACE *Dot, VG_BOOL Solid)
{
  VG_ULONG              k;

  VG_BOOL               Ret        = TRUE;
  register VG_ULONG     TotPts     = 0;

  if (fprintf(FH, "#VRML V1.0 ascii\n" \
                  "Separator {\n" \
                  "  Material{ diffuseColor  1.0 0.0 0.0 }\n" \
                  "  Coordinate3 { point [\n") > 0) {

    while((Dot) && (Ret)) {
      if (fprintf(FH, "    %8.3f %8.3f %8.3f,\n", Dot -> x, Dot -> y, Dot -> z) >= 0)
        ++TotPts;
      else Ret = PrintDosErr();
      Dot = Dot -> Next;
    } /* End of Dot loop */
  } else Ret = PrintDosErr();

  if (Ret) {

   /**** Solid surface ****/

    if (Solid) {
      if (fprintf(FH, "  ] }\n\n  IndexedFaceSet {\n" \
                      "    coordIndex [\n      ") > 0) {
        for(k = 0; (Ret) && (k < TotPts); ++k) {
          if (fprintf(FH, "%4d,", k) >= 0) {
            if (!(k % 5))
              if (fprintf(FH, "\n    ") < 0)
                Ret = PrintDosErr();
          } else Ret = PrintDosErr();
        }
        if ((Ret) && (fprintf(FH, "      -1 ]\n    }\n}\n") < 0))
          Ret = PrintDosErr();
      } else Ret = PrintDosErr();
    } else {

     /**** Dotted surface ****/

      if (fprintf(FH, "  ] }\n\n  PointSet {\n" \
                      "    startIndex 0\n    numPoints -1\n  }\n}\n") <= 0)
        Ret = PrintDosErr();
    }
  }

  return Ret;
}


/**** Surface calculation ****/

VG_BOOL Surface(FILE *FH, char *File, ATOMO *InizAtm, VG_ULONG TotAtm,
             VG_ULONG Points, float ProRad, VG_UWORD Form, float *Contrib)
{
  VG_SURFACE            *InizSrf;

  VG_BOOL               Ret    = TRUE;
  VG_ULONG              TotDot = 0;

  SrfPrintProg(Form);

  if ((Form & 0xff00) == SRF_PROP_VLOGP) {
    ProRad = VLOGP_PROBE_RAD;
    Points = VLOGP_DOT_DENSITY;
  }
  if ((InizSrf = SrfCalc(NULL, &TotDot, InizAtm, TotAtm, Points, ProRad,
                         Form, Contrib, SRF_FLG_ALLATM))) {
    if ((Form & 0xff00) != SRF_PROP_VDW) SrfColorByVal(InizSrf);
    Ret = SrfSave(FH, File, InizSrf, TotalSrf, InizAtm, Form);
    SrfFree(InizSrf);
  }

  return Ret;
}



