
/************************************************
****           VEGA - VRML savers            ****
**** Copyright 1996-2001, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"

#define  FIR                1.2566370614359173
#define  VR_CROCE_RAD       0.25
#define  VR_SOL_NTRIAN      256


/**** Structure definitions ****/

typedef struct {
  float		Rad;
  VG_UWORD	Col;
} COLMTR;

struct Pun {
  VG_LONG 	ito;
  ATOMO		*iso;
} *pun_1;


/**** Local prototypes ****/

void      CalCen(float *, float *);
void      CalVer(float *);
VG_BOOL   GeoCav(VG_ULONG, VG_ULONG *, COLMTR *, ATOMO *);
COLMTR    *VrmlAssign(ATOMO *, VG_ULONG, float);


/**** Local variables ****/

static char     ColStr[][12] = { "1.0 0.0 1.0", /* Violet = Generic atom */
                                 "0.0 1.0 0.0", /* Green  = Carbon       */
                                 "0.0 0.0 1.0", /* Blue   = Nitrogen     */
                                 "1.0 1.0 1.0", /* White  = Hydrogen     */
                                 "1.0 0.0 0.0", /* Red    = Oxygen       */
                                 "1.0 1.0 0.0"  /* Yellow = Sulfur       */
                               };

static float    *xc1, *yc1, *zc1;



/**** Sets colors and VDW radii ****/

COLMTR *VrmlAssign(ATOMO *InizAtm, VG_ULONG TotAtm, float ProbeRad)
{
  COLMTR		*ColMtr;
  register ATOMO        *Atm;
  register COLMTR	*Ptr;
  register VG_UWORD	k;

  VG_DCHAR              Elem[] = { {"C"}, {"N"}, {"H"}, {"O"}, {"S"} };
  float                 Rad[]  = { 1.8,  1.7, 1.3, 1.6, 1.8 };

  if ((ColMtr = (COLMTR *)Alloca(TotAtm * sizeof(COLMTR)))) {
    for(k = 0; k < 5; ++k) {
      Ptr = ColMtr;
      for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
        if (Atm -> Elem.S == Elem[k].S) {
          Ptr -> Col = k + 1;
          Ptr -> Rad = Rad[k] + ProbeRad;
        }
        ++Ptr;
      }
    }

    Ptr = ColMtr;
    for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
      if (!Ptr -> Col) {
        for(k = 0; *AtmTable[k].Atom.C; ++k) {
          if (AtmTable[k].Atom.S == Atm -> Elem.S) {
            Ptr -> Rad = AtmTable[k].VdwRad + ProbeRad;
            break;
          }
        }
      }
    }
  }

  return ColMtr;
}


/**** VRML 1.0 CPK ****/

VG_BOOL VrmlCpkSave(FILE *FH, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  VG_BOOL           First;
  COLMTR            *ColMtr, *Ptr;
  register ATOMO    *Atm;
  VG_UWORD          k;

  VG_BOOL				Ret = TRUE;

  if ((ColMtr = VrmlAssign(InizAtm, TotAtm, 0))) {
    if (fprintf(FH, VrmlHdr) >= 0) {
      for(k = 1; (Ret) && (k < 6); ++k) {
        First = FALSE;
        Ptr   = ColMtr;
        for(Atm = InizAtm; (Ret) && (Atm); Atm = Atm -> Ptr) {
      	  if (Ptr -> Col == k) {
            if (!First) {
              First = TRUE;
              if (fprintf(FH, "  Separator {\n"\
                              "   Material{ diffuseColor %s }\n" \
              	              "   DEF CPK%d Sphere { radius %.3f }\n",
                              ColStr[Ptr -> Col], k, Ptr -> Rad) < 0)
                Ret = PrintDosErr();
            }
            if ((Ret) && (fprintf(FH, "    Separator {\n" \
                                      "      Translation { translation %.3f %.3f %.3f }\n" \
                                      "      USE CPK%d }\n", Atm -> x, Atm -> y, Atm -> z, k) < 0))
              Ret = PrintDosErr();

      	  }
          ++Ptr;
        } /* End of atom loop */
        if ((First) &&(fprintf(FH, "  }\n") < 0))
          Ret = PrintDosErr();
      } /* End of loop */

      Ptr   = ColMtr;
      First = FALSE;
      for(Atm = InizAtm; (Ret) && (Atm); Atm = Atm -> Ptr) {
        if (!Ptr -> Col) {
          if (!First) {
            First = TRUE;
            if (fprintf(FH, "  Separator {\n" \
                        "   Material{ diffuseColor 1.0 0.0 1.0 }\n") < 0)
              Ret = PrintDosErr();
            }
            if ((Ret) && (fprintf(FH, "    Separator {\n" \
                                      "      Translation { translation %.3f %.3f %.3f }\n" \
                                      "      Sphere { radius %.3f }\n" \
                                      "    }\n",
                                  Atm -> x, Atm -> y, Atm -> z, 1.5) < 0))
              Ret = PrintDosErr();
        }
      } /* End of atom loop */
      if (Ret) {
        if ((First) && (fprintf(FH, "  }\n") < 0))
          Ret = PrintDosErr();
        if ((Ret) && (fprintf(FH, "}\n") < 0))
          Ret = PrintDosErr();
      }
    } else PrintDosErr();
    FREE(ColMtr);
  } else Ret = FALSE;

  return Ret;
}


/**** VRML 1.0 wireframe ****/

VG_BOOL VrmlSave(FILE *FH, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  VG_BOOL             First;
  COLMTR              *ColMtr;
  register ATOMO      *Atm;
  register COLMTR     *Ptr;
  VG_ULONG            Iat;
  VG_UWORD            k, j;

  VG_BOOL             Ret = TRUE;

  if ((ColMtr = VrmlAssign(InizAtm, TotAtm, 0))) {
    if (fprintf(FH, VrmlHdr) >= 0) {
      for(k = 0; (Ret) && (k < 6); ++k) {
        Ptr   = ColMtr;
        First = FALSE;
        for(Atm = InizAtm; (Ret) && (Atm); Atm = Atm -> Ptr) {
          if (Ptr -> Col == k) {
            if (!First) {
              First = TRUE;
              if (fprintf(FH, "  Separator {\n" \
                              "    Material{ diffuseColor %s }\n" \
                              "    Coordinate3 { point [\n", ColStr[k]) < 0) {
                Ret = PrintDosErr();
                break;
              }
            }
            if (Atm -> NSost) {
              if (fprintf(FH, "      %.3f %.3f %.3f,\n",
                              Atm -> x, Atm -> y, Atm -> z) >= 0) {
                for(j = 0; (Ret) && (j < Atm -> NSost); ++j) {
                  if (fprintf(FH, "      %.3f %.3f %.3f,\n",
                                  (Atm -> x + Atm -> Conn[j] -> x) / 2,
                                  (Atm -> y + Atm -> Conn[j] -> y) / 2,
                                  (Atm -> z + Atm -> Conn[j] -> z) / 2) < 0)
                    Ret = PrintDosErr();
                }
              } else Ret = PrintDosErr();
            } else if (fprintf(FH, "      %.3f %.3f %.3f,\n" \
                                   "      %.3f %.3f %.3f,\n" \
                                   "      %.3f %.3f %.3f,\n" \
                                   "      %.3f %.3f %.3f,\n" \
                                   "      %.3f %.3f %.3f,\n" \
                                   "      %.3f %.3f %.3f,\n",
                                   Atm -> x - VR_CROCE_RAD, Atm -> y                , Atm -> z,
                                   Atm -> x + VR_CROCE_RAD, Atm -> y                , Atm -> z,
                                   Atm -> x               , Atm -> y  - VR_CROCE_RAD, Atm -> z,
                                   Atm -> x               , Atm -> y  + VR_CROCE_RAD, Atm -> z,
                                   Atm -> x               , Atm -> y                , Atm -> z - VR_CROCE_RAD,
                                   Atm -> x               , Atm -> y                , Atm -> z + VR_CROCE_RAD) < 0)
              Ret = PrintDosErr();
          }
          ++Ptr;
        } /* End of atom loop */
        if (Ret) {
          if (First) {
            if (fprintf(FH, "    ] }\n"\
                            "    IndexedLineSet { coordIndex [\n") >= 0) {
              Iat = 0;
              Ptr = ColMtr;
              for(Atm = InizAtm; (Ret) && (Atm); Atm = Atm -> Ptr) {
                if (Ptr -> Col == k) {
                  if (Atm -> NSost) {
                    for(j = 1; (Ret) && (j <= Atm -> NSost); ++j)
                      if (fprintf(FH, "      %d, %d, -1,\n", Iat, Iat + j) < 0)
                        Ret = PrintDosErr();
                    Iat += (Atm -> NSost + 1);
                  } else {
                    for(j = 0; (Ret) && (j < 3); ++j) {
                      if (fprintf(FH, "      %d, %d, -1,\n", Iat, Iat + 1) >= 0)
                        Iat += 2;
                      else Ret = PrintDosErr();
                    }
                  }
                }
                ++Ptr;
              }
              if (fprintf(FH, "    ] }\n" \
                          "  }\n") < 0)
                Ret = PrintDosErr();
            } else Ret = PrintDosErr();
          }
        }
      } /* End of loop */
      if ((Ret) && (fprintf(FH, "}\n") < 0))
        Ret = PrintDosErr();
    } else Ret = PrintDosErr();
    FREE(ColMtr);
  } else Ret = FALSE;

  return Ret;
}


/**** VRML 1.0 solid surface ****/

VG_BOOL VrmlSolSave(FILE *FH, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  COLMTR            *ColMtr, *Ptr;
  register ATOMO    *Atm;
  VG_LONG           ii, i, ij, j, k, j2, j3, j4, j5, jv;
  float             fi, th, cth, sth;
  float             cc[3], cvn[9], cvn2[18], cvn3[18], cvn4[18], cvn5[18];
  float             plus1[18], tempex[18], vertex[18];
  float             *cv;
  VG_ULONG          fff, np;

  VG_BOOL           Ret         = TRUE;

  static const float    thev[6] = { 0.6523581397843682, 1.1071487177940905,
                                    1.3820857960113345, 1.7595068575784587,
                                    2.0344439357957027, 2.4892345138054251
                                  };

  static const float    fiv[6]  = { 0.6283185307179586, 0.,
                                    0.6283185307179586, 0.,
                                    0.6283185307179586, 0.
                                  };

  static const VG_UBYTE  jvt1[] = {  1,  6,  2,  1,  2,  3,  1,  3,  4,  1,  4,  5,  1,  5,  6,  7,  2,  6,  8,  3,
	    		             2,  9,  4,  3, 10,  5,  4, 11,  6,  5,  8,  2, 12,  9,  3, 13, 10,  4, 14, 11,
	    		             5, 15,  7,  6, 16,  7, 12,  2,  8, 13,  3,  9, 14,  4, 10, 15,  5, 11, 16,
	    		             6,  8, 12, 18,  9, 13, 19, 10, 14, 20, 11, 15, 21,  7, 16, 17,  7,
	    		            17, 12,  8, 18, 13,  9, 19, 14, 10, 20, 15, 11, 21, 16, 22, 12, 17,
	    		            23, 13, 18, 24, 14, 19, 25, 15, 20, 26, 16, 21, 22, 18, 12, 23,
	    		            19, 13, 24, 20, 14, 25, 21, 15, 26, 17, 16, 22, 17, 27, 23, 18,
	    		            28, 24, 19, 29, 25, 20, 30, 26, 21, 31, 22, 28, 18, 23, 29, 19,
	    		            24, 30, 20, 25, 31, 21, 26, 27, 17, 22, 27, 28, 23, 28, 29, 24,
	    		            29, 30, 25, 30, 31, 26, 31, 27, 32, 28, 27, 32, 29, 28, 32, 30,
	    		            29, 32, 31, 30, 32, 27, 31
	    		          };

  static const VG_UBYTE jvt2[]  = { 1, 5, 4, 5, 2, 6, 4, 6, 3, 6, 4, 5 };


  if ((xc1 = (float *)Alloca(15360 * sizeof(float)))) {
    if ((yc1 = (float *)Alloca(15360 * sizeof(float)))) {
      if ((zc1 = (float *)Alloca(15360 * sizeof(float)))) {
        if ((cv = (float *)Alloca(15360 * sizeof(float)))) {
          if ((ColMtr = VrmlAssign(InizAtm, TotAtm, Prefs.SAS_PROBERAD))) {
            *cv    = 0.;
            cv[32] = 0.;
            cv[64] = 1.;
            cv[31] = 0.;
            cv[63] = 0.;
            cv[95] = -1.;
            ii     = 1;
            for (i = 0; i < 6; ++i) {
              th  = thev[i];
              fi  = fiv[i];
              cth = cos(th);
              sth = sin(th);
              for (j = 1; j <= 5; ++j) {
                fi += FIR;
                if (j == 1) fi = fiv[i];
                ++ii;
                *(cv + ii - 1)  = sth * cos(fi);
                *(cv + ii + 31) = sth * sin(fi);
                *(cv + ii + 63) = cth;
              }
            }

            ij = 0;
            for (j = 0; j < 180; j += 3) {
              for (k = 0; k < 3; ++k) {
                for (i = 1; i <= 3; ++i)
	              cvn2[k + i * 6 - 6] = *(cv +jvt1[k + j] + (i << 5) - 33);
              }

              CalVer(cvn2);

              for (j2 = 0; j2 < 12; j2 += 3) {
                for (k = 0; k < 3; ++k) {
                  for (i = 0; i < 18; i += 6)
                    cvn3[k + i] = cvn2[jvt2[k + j2 ] + i - 1];
                }

                CalVer(cvn3);

                for (j3 = 0; j3 < 12; j3 += 3) {
                  for (k = 0; k < 3; ++k) {
                    for (i = 0; i < 18; i += 6)
                      cvn4[k + i] = cvn3[jvt2[k + j3] + i - 1];
                  }

                  CalVer(cvn4);

                  for (j4 = 0; j4 < 12; j4 += 3) {
                    for (k = 0; k < 3; ++k) {
                      for (i = 0; i < 18; i += 6)
                        cvn5[k + i] = cvn4[jvt2[k + j4] + i - 1];
                    }

                    CalVer(cvn5);

                    for (j5 = 0; j5 < 12; j5 += 3) {
                      for (k = 0; k < 3; ++k) {
                        for (i = 0; i < 9; i += 3)
                          cvn[k + i] = cvn5[jvt2[k + j5] + i * 2 - 1];
                      }

                      CalCen(cvn, cc);

                      xc1[ij] = (float) cc[0];
                      yc1[ij] = (float) cc[1];
                      zc1[ij] = (float) cc[2];
                      ++ij;
                    }
                  }
                }
              }
            }

            if ((pun_1 = (struct Pun *)Alloca(TotAtm * sizeof(struct Pun) * 60))) {
              if ((GeoCav(TotAtm, &np, ColMtr, InizAtm))) {
                if (fprintf(FH, VrmlHdr) >= 0) {
                  for (i = 0; (Ret) && ((VG_ULONG)i < np); ++i) {
                    Atm = (pun_1 + i) -> iso;
                    for (fff = 1; fff <= 3; ++fff) {
                      for (jv = 1; jv <= 3; ++jv) {
                        plus1[jv + fff * 6 - 7] = *(cv + jvt1[jv +
                             (pun_1 + i) -> ito * 3 - 4] + (fff << 5) - 33);
                      }
                    }
                    CalVer(plus1);

                    for (jv = 0; jv < 6; ++jv) {
                      for (fff = 0; fff <= 12; fff += 6)
                        tempex[jv + fff] = plus1[jv + fff];
                    }
                    for (j = 0; (Ret) && (j < 12); j += 3) {
                      for (fff = 0; fff <= 12; fff += 6) {
                        for (k = 0; k < 3; ++k)
                          plus1[k + fff] = tempex[jvt2[k + j] + fff - 1];
                      }
                      CalVer(plus1);
                      Ptr = ColMtr + (Atm -> Num - 1);
                      for (jv = 0; jv < 6; ++jv) {
                        vertex[jv]      = plus1[jv]      * Ptr -> Rad + Atm -> x;
                        vertex[jv + 6]  = plus1[jv + 6]  * Ptr -> Rad + Atm -> y;
                        vertex[jv + 12] = plus1[jv + 12] * Ptr -> Rad + Atm -> z;
                      }
                      if (fprintf(FH, "  Separator {\n" \
                                      "    Material{ diffuseColor %s\n" \
                                      "      transparency 0.5 }\n" \
                                      "    Coordinate3 {\n" \
                                      "      point [\n", ColStr[Ptr -> Col]) < 0) {
                        Ret = PrintDosErr();
                        break;
                      }
                      for (jv = 0; (Ret) && (jv <= 5); ++jv) {
                        if (fprintf(FH, "       ") >= 0) {
                          for (fff = 0; (Ret) && (fff <= 12); fff += 6)
                            if (fprintf(FH, " %.3f", vertex[jv + fff]) < 0)
                              Ret = PrintDosErr();
                          if ((Ret) && (fprintf(FH, ",\n") < 0)) Ret = PrintDosErr();
                        } else Ret = PrintDosErr();
                      } /* End of loop */
                      if ((Ret) && (fprintf(FH, "      ]}\n" \
                                                "      IndexedFaceSet {\n" \
                                                "        coordIndex [\n" \
                                                "          0, 4, 1,\n" \
                                                "          5, 2, 3,\n" \
                                                "           -1   ]}\n" \
                                                "      }\n") < 0))
                        Ret = PrintDosErr();
                    }
                  } /* End of vertex loop */
                } else Ret = PrintDosErr();
              }
              if ((Ret) && (fprintf(FH, "}\n") < 0))
                Ret = PrintDosErr();
              FREE(pun_1);
            } else Ret = FALSE;
            FREE(ColMtr);
          } else Ret = FALSE;
          FREE(cv);
        } else Ret = FALSE;
        FREE(zc1);
      } else Ret = FALSE;
      FREE(yc1);
    } else Ret = FALSE;
    FREE(xc1);
  } else Ret = FALSE;

  return Ret;
}


void CalVer(register float *cvn)
{
  register VG_UWORD	n, n1, n2;
  register float	rrr, xxx, yyy, zzz;

  cvn -= 7;
  for (n = 1; n <= 3; ++n) {
    n2 = n + 3;
    if (n == 1) n1 = 3;
    else n1 = n - 1;

    xxx = (cvn[n + 6]  + cvn[n1 + 6])  / 2.;
    yyy = (cvn[n + 12] + cvn[n1 + 12]) / 2.;
    zzz = (cvn[n + 18] + cvn[n1 + 18]) / 2.;
    rrr = 1.0 / SQR(Quad(xxx) + Quad(yyy) + Quad(zzz));

    cvn[n2 + 6]  = xxx * rrr;
    cvn[n2 + 12] = yyy * rrr;
    cvn[n2 + 18] = zzz * rrr;
  }
}


void CalCen(float *cv, float *cc)
{
  float		rrr, xxx, yyy, zzz;

  --cc;
  cv -= 4;

  xxx = (cv[4]  + cv[5]  + cv[6])  / 3.;
  yyy = (cv[7]  + cv[8]  + cv[9])  / 3.;
  zzz = (cv[10] + cv[11] + cv[12]) / 3.;
  rrr = 1.0 / SQR(Quad(xxx) + Quad(yyy) + Quad(zzz));
  cc[1] = xxx * rrr;
  cc[2] = yyy * rrr;
  cc[3] = zzz * rrr;
}


VG_BOOL GeoCav(VG_ULONG ncor, VG_ULONG *np, COLMTR *ColMtr, ATOMO *InizAtm)
{
  ATOMO			*Atm1, *Atm2;
  COLMTR		*Ptr;
  float			*InizR2, *r2, dd;
  float 		t, xsm, ysm, zsm;
  VG_LONG       	ninf, nsup, i, j, k, n4;
  VG_LONG		nts;

  *np = 0;

  if ((InizR2 = (float *)Alloca(ncor * sizeof(float)))) {
    r2  = InizR2;
    Ptr = ColMtr;
    for (i = 1; (VG_ULONG)i <= ncor; ++i) {
      *r2 = Quad(Ptr -> Rad);
      ++Ptr;
      ++r2;
    }

    Ptr  = ColMtr;
    Atm1 = InizAtm;
    for (i = 1; (VG_ULONG)i <= ncor; ++i) {
      nsup = 0;
      for (j = 1; j <= 60; ++j) {
        nts  = 0;
        ninf = nsup;
        nsup = ninf + VR_SOL_NTRIAN;
        for (k = ninf; k < nsup; ++k) {
          xsm  = xc1[k] * Ptr -> Rad + Atm1 -> x;
          ysm  = yc1[k] * Ptr -> Rad + Atm1 -> y;
          zsm  = zc1[k] * Ptr -> Rad + Atm1 -> z;
          Atm2 = InizAtm;
          r2   = InizR2;
          for (n4 = 1; (VG_ULONG)n4 <= ncor; ++n4) {
            if (n4 != i) {
              t   = xsm - Atm2 -> x;
              dd  = Quad(t);
              t   = ysm - Atm2 -> y;
              dd += Quad(t);
              t   = zsm - Atm2 -> z;
              dd += Quad(t);
              if (dd < *r2) goto L3;
            }
            Atm2 = Atm2 -> Ptr;
            ++r2;
          }
          ++nts;
L3:
          ;
        }
        if (nts) {
          ++(*np);
          (pun_1 + *np - 1) -> ito = j;
          (pun_1 + *np - 1) -> iso = Atm1;
        }
      }
      ++Ptr;
      Atm1 = Atm1 -> Ptr;
    }
    FREE(InizR2);
  } else return FALSE;

  return TRUE;
}
